"""
Node.js/Express Project Template
This template creates a basic Express.js application structure with:
- MVC pattern
- Middleware configuration
- API routes
- Database connection
- Configuration files
"""
import os
from pathlib import Path

def create_nodejs_project(project_name: str):
    """Create a basic Node.js/Express project structure."""
    project_path = Path(project_name)
    
    # Create project directories
    (project_path / "src").mkdir(parents=True, exist_ok=True)
    (project_path / "src" / "controllers").mkdir(parents=True, exist_ok=True)
    (project_path / "src" / "models").mkdir(parents=True, exist_ok=True)
    (project_path / "src" / "routes").mkdir(parents=True, exist_ok=True)
    (project_path / "src" / "middleware").mkdir(parents=True, exist_ok=True)
    (project_path / "src" / "utils").mkdir(parents=True, exist_ok=True)
    (project_path / "src" / "config").mkdir(parents=True, exist_ok=True)
    (project_path / "tests").mkdir(parents=True, exist_ok=True)
    
    # Create package.json
    with open(project_path / "package.json", "w") as f:
        f.write(f'''{{
  "name": "{project_name.lower().replace(' ', '-')}",
  "version": "1.0.3",
  "description": "A Node.js/Express project generated with Codeius AI",
  "main": "src/server.js",
  "scripts": {{
    "start": "node src/server.js",
    "dev": "nodemon src/server.js",
    "test": "jest",
    "test:watch": "jest --watch"
  }},
  "keywords": ["node", "express", "api"],
  "author": "Generated by Codeius AI",
  "license": "MIT",
  "dependencies": {{
    "express": "^4.21.2",
    "mongoose": "^8.8.3",
    "cors": "^2.8.5",
    "dotenv": "^16.4.7",
    "bcryptjs": "^2.4.3",
    "jsonwebtoken": "^9.0.2",
    "helmet": "^7.2.0",
    "express-rate-limit": "^7.4.1",
    "joi": "^17.13.3"
  }},
  "devDependencies": {{
    "nodemon": "^3.1.9",
    "jest": "^29.7.0",
    "supertest": "^7.0.0"
  }}
}}
''')
    
    # Create server.js
    with open(project_path / "src" / "server.js", "w") as f:
        f.write(f'''const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const rateLimit = require('express-rate-limit');
const mongoose = require('mongoose');
const { connectDB } = require('./config/database');
const userRoutes = require('./routes/userRoutes');

const app = express();

// Security middleware
app.use(helmet());
app.use(cors());

// Rate limiting
const limiter = rateLimit({{
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100 // limit each IP to 100 requests per windowMs
}});
app.use(limiter);

// Body parsing middleware
app.use(express.json({{ limit: '10mb' }}));
app.use(express.urlencoded({{ extended: true }}));

// Database connection
connectDB();

// Routes
app.get('/', (req, res) => {{
  res.json({{ message: 'Welcome to {project_name}!' }});
}});

app.use('/api/users', userRoutes);

// Error handling middleware
app.use((err, req, res, next) => {{
  console.error(err.stack);
  res.status(500).json({{
    success: false,
    error: 'Server Error'
  }});
}});

// Start server
const PORT = process.env.PORT || 3000;
app.listen(PORT, () => {{
  console.log(`Server is running on port ${{PORT}}`);
}});

module.exports = app;
''')
    
    # Create config/database.js
    with open(project_path / "src" / "config" / "database.js", "w") as f:
        f.write('''const mongoose = require('mongoose');

const connectDB = async () => {
  try {
    const conn = await mongoose.connect(process.env.MONGODB_URI || 'mongodb://localhost:27017/myapp', {
      useNewUrlParser: true,
      useUnifiedTopology: true,
    });

    console.log(`MongoDB Connected: $${conn.connection.host}`);
  } catch (error) {
    console.error(`Error: $${error.message}`);
    process.exit(1);
  }
};

module.exports = { connectDB };
''')
    
    # Create middleware/errorHandler.js
    with open(project_path / "src" / "middleware" / "errorHandler.js", "w") as f:
        f.write('''const errorHandler = (err, req, res, next) => {
  let statusCode = res.statusCode === 200 ? 500 : res.statusCode;
  let message = err.message;

  // Mongoose bad ObjectId
  if (err.name === 'CastError' && err.kind === 'ObjectId') {
    statusCode = 404;
    message = 'Resource not found';
  }
  
  // Mongoose duplicate key
  if (err.code === 11000) {
    statusCode = 400;
    message = 'Duplicate field value entered';
  }
  
  // Mongoose validation error
  if (err.name === 'ValidationError') {
    statusCode = 400;
    message = Object.values(err.errors).map(val => val.message).join(', ');
  }

  res.status(statusCode).json({
    success: false,
    error: message,
    stack: process.env.NODE_ENV === 'production' ? null : err.stack
  });
};

module.exports = { errorHandler };
''')
    
    # Create models/User.js
    with open(project_path / "src" / "models" / "User.js", "w") as f:
        f.write('''const mongoose = require('mongoose');
const bcrypt = require('bcryptjs');

const userSchema = new mongoose.Schema({
  name: {
    type: String,
    required: [true, 'Please add a name'],
    trim: true,
    maxlength: [50, 'Name cannot be more than 50 characters']
  },
  email: {
    type: String,
    required: [true, 'Please add an email'],
    unique: true,
    lowercase: true,
    match: [
      /^\w+([.-]?\w+)*@\w+([.-]?\w+)*(\.\w{2,3})+$/,
      'Please add a valid email'
    ]
  },
  password: {
    type: String,
    required: [true, 'Please add a password'],
    minlength: 6,
    select: false
  },
  role: {
    type: String,
    enum: ['user', 'admin'],
    default: 'user'
  }
}, {
  timestamps: true
});

// Encrypt password using bcrypt
userSchema.pre('save', async function(next) {
  if (!this.isModified('password')) {
    next();
  }

  const salt = await bcrypt.genSalt(10);
  this.password = await bcrypt.hash(this.password, salt);
});

// Match user entered password to hashed password in database
userSchema.methods.matchPassword = async function(enteredPassword) {
  return await bcrypt.compare(enteredPassword, this.password);
};

module.exports = mongoose.model('User', userSchema);
''')
    
    # Create controllers/userController.js
    with open(project_path / "src" / "controllers" / "userController.js", "w") as f:
        f.write('''const User = require('../models/User');
const jwt = require('jsonwebtoken');

// @desc    Register user
// @route   POST /api/users/register
// @access  Public
const registerUser = async (req, res) => {
  try {
    const { name, email, password } = req.body;

    // Create user
    const user = await User.create({
      name,
      email,
      password
    });

    res.status(201).json({
      success: true,
      data: {
        id: user._id,
        name: user.name,
        email: user.email
      }
    });
  } catch (error) {
    res.status(400).json({
      success: false,
      error: error.message
    });
  }
};

// @desc    Login user
// @route   POST /api/users/login
// @access  Public
const loginUser = async (req, res) => {
  try {
    const { email, password } = req.body;

    // Check if email and password exist
    if (!email || !password) {
      return res.status(400).json({
        success: false,
        error: 'Please provide an email and password'
      });
    }

    // Check for user
    const user = await User.findOne({ email }).select('+password');

    if (!user) {
      return res.status(401).json({
        success: false,
        error: 'Invalid credentials'
      });
    }

    // Check if password matches
    const isMatch = await user.matchPassword(password);

    if (!isMatch) {
      return res.status(401).json({
        success: false,
        error: 'Invalid credentials'
      });
    }

    // Create token
    const token = jwt.sign({ id: user._id }, process.env.JWT_SECRET || 'secret', {
      expiresIn: process.env.JWT_EXPIRE || '30d'
    });

    res.status(200).json({
      success: true,
      token
    });
  } catch (error) {
    res.status(400).json({
      success: false,
      error: error.message
    });
  }
};

// @desc    Get current user
// @route   GET /api/users/me
// @access  Private
const getMe = async (req, res) => {
  const user = await User.findById(req.user.id);

  res.status(200).json({
    success: true,
    data: {
      id: user._id,
      name: user.name,
      email: user.email
    }
  });
};

module.exports = {
  registerUser,
  loginUser,
  getMe
};
''')
    
    # Create routes/userRoutes.js
    with open(project_path / "src" / "routes" / "userRoutes.js", "w") as f:
        f.write('''const express = require('express');
const { registerUser, loginUser, getMe } = require('../controllers/userController');
const { protect } = require('../middleware/auth');

const router = express.Router();

router.route('/register').post(registerUser);
router.route('/login').post(loginUser);
router.route('/me').get(protect, getMe);

module.exports = router;
''')
    
    # Create middleware/auth.js
    with open(project_path / "src" / "middleware" / "auth.js", "w") as f:
        f.write('''const jwt = require('jsonwebtoken');
const User = require('../models/User');

const protect = async (req, res, next) => {
  let token;

  if (req.headers.authorization && req.headers.authorization.startsWith('Bearer')) {
    try {
      // Get token from header
      token = req.headers.authorization.split(' ')[1];

      // Verify token
      const decoded = jwt.verify(token, process.env.JWT_SECRET || 'secret');

      // Get user from the token
      req.user = await User.findById(decoded.id).select('-password');

      next();
    } catch (error) {
      res.status(401).json({
        success: false,
        error: 'Not authorized'
      });
    }
  }

  if (!token) {
    res.status(401).json({
      success: false,
      error: 'Not authorized, no token'
    });
  }
};

module.exports = { protect };
''')
    
    # Create .env file
    with open(project_path / ".env", "w") as f:
        f.write('''NODE_ENV=development
PORT=3000
MONGODB_URI=mongodb://localhost:27017/myapp
JWT_SECRET=your_jwt_secret_here
JWT_EXPIRE=30d
''')
    
    # Create .gitignore
    with open(project_path / ".gitignore", "w") as f:
        f.write('''node_modules/
.env
.env.local
.env.development.local
.env.test.local
.env.production.local
npm-debug.log*
yarn-debug.log*
yarn-error.log*
dist/
build/
.coverage
.nyc_output
.DS_Store
''')
    
    # Create README.md
    with open(project_path / "README.md", "w") as f:
        f.write(f'''# {project_name}

A Node.js/Express project template generated with Codeius AI.

## Setup

```bash
npm install
npm run dev  # for development with hot reloading
# or
npm start    # for production
```

## Environment Variables

Create a `.env` file in the root and add the following:

```env
NODE_ENV=development
PORT=3000
MONGODB_URI=mongodb://localhost:27017/myapp
JWT_SECRET=your_jwt_secret_here
JWT_EXPIRE=30d
```

## API Endpoints

- `GET /` - Health check
- `POST /api/users/register` - Register a new user
- `POST /api/users/login` - Login user
- `GET /api/users/me` - Get current user (requires auth)

## Folder Structure

- `src/` - Main source code
  - `controllers/` - Request handlers
  - `models/` - Database models
  - `routes/` - API routes
  - `middleware/` - Custom middleware
  - `utils/` - Utility functions
  - `config/` - Configuration files
- `tests/` - Test files
''')
    
    # Create test file
    with open(project_path / "tests" / "user.test.js", "w") as f:
        f.write('''const request = require('supertest');
const app = require('../src/server');

describe('User API', () => {
  it('should register a new user', async () => {
    const res = await request(app)
      .post('/api/users/register')
      .send({
        name: 'Test User',
        email: 'test@example.com',
        password: 'password123'
      });
    
    expect(res.status).toBe(201);
    expect(res.body).toHaveProperty('data');
    expect(res.body.data).toHaveProperty('email', 'test@example.com');
  });

  it('should login user', async () => {
    const res = await request(app)
      .post('/api/users/login')
      .send({
        email: 'test@example.com',
        password: 'password123'
      });
    
    expect(res.status).toBe(200);
    expect(res.body).toHaveProperty('token');
  });
});
''')
    
    print(f"Node.js/Express project '{project_name}' created successfully!")