"""
Django Project Template
This template creates a basic Django application structure with:
- Settings configuration
- Basic models
- Views and URLs
- Requirements file
- Django standards
"""
import os
from pathlib import Path

def create_django_project(project_name: str):
    """Create a basic Django project structure."""
    project_path = Path(project_name)
    
    # Create project directories
    (project_path / project_name).mkdir(parents=True, exist_ok=True)
    (project_path / "core").mkdir(parents=True, exist_ok=True)
    (project_path / "api").mkdir(parents=True, exist_ok=True)
    (project_path / "api" / "migrations").mkdir(parents=True, exist_ok=True)
    (project_path / "manage").mkdir(parents=True, exist_ok=True)
    (project_path / "tests").mkdir(parents=True, exist_ok=True)
    
    # Create Django settings in the project directory
    with open(project_path / project_name / "settings.py", "w") as f:
        f.write(f'''"""
Django settings for {project_name} project.
Generated by Codeius AI.
"""
import os
from pathlib import Path

BASE_DIR = Path(__file__).resolve().parent.parent

SECRET_KEY = 'django-insecure-your-secret-key-here'
DEBUG = True
ALLOWED_HOSTS = ['localhost', '127.0.0.1']

INSTALLED_APPS = [
    'django.contrib.admin',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.messages',
    'django.contrib.staticfiles',
    'rest_framework',
    'corsheaders',
    'core',
    'api',
]

MIDDLEWARE = [
    'corsheaders.middleware.CorsMiddleware',
    'django.middleware.security.SecurityMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'django.contrib.auth.middleware.AuthenticationMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
]

ROOT_URLCONF = '{project_name}.urls'

TEMPLATES = [
    {{
        'BACKEND': 'django.template.backends.django.DjangoTemplates',
        'DIRS': [BASE_DIR / 'templates'],
        'APP_DIRS': True,
        'OPTIONS': {{
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
            ],
        }},
    }},
]

WSGI_APPLICATION = '{project_name}.wsgi.application'

DATABASES = {{
    'default': {{
        'ENGINE': 'django.db.backends.sqlite3',
        'NAME': BASE_DIR / 'db.sqlite3',
    }}
}}

AUTH_PASSWORD_VALIDATORS = [
    {{
        'NAME': 'django.contrib.auth.password_validation.UserAttributeSimilarityValidator',
    }},
    {{
        'NAME': 'django.contrib.auth.password_validation.MinimumLengthValidator',
    }},
    {{
        'NAME': 'django.contrib.auth.password_validation.CommonPasswordValidator',
    }},
    {{
        'NAME': 'django.contrib.auth.password_validation.NumericPasswordValidator',
    }},
]

LANGUAGE_CODE = 'en-us'
TIME_ZONE = 'UTC'
USE_I18N = True
USE_TZ = True

STATIC_URL = '/static/'
STATICFILES_DIRS = [BASE_DIR / 'static']

DEFAULT_AUTO_FIELD = 'django.db.models.BigAutoField'
''')
    
    # Create main URLs
    with open(project_path / project_name / "urls.py", "w") as f:
        f.write(f'''"""{project_name} URL Configuration"""
from django.contrib import admin
from django.urls import path, include

urlpatterns = [
    path('admin/', admin.site.urls),
    path('api/', include('api.urls')),
    path('', include('core.urls')),
]
''')
    
    # Create WSGI file
    with open(project_path / project_name / "wsgi.py", "w") as f:
        f.write('''"""
WSGI config for project.
"""
import os
from django.core.wsgi import get_wsgi_application

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'project.settings')
application = get_wsgi_application()
''')
    
    # Create asgi.py
    with open(project_path / project_name / "asgi.py", "w") as f:
        f.write('''"""
ASGI config for project.
"""
import os
from django.core.asgi import get_asgi_application

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'project.settings')
application = get_asgi_application()
''')
    
    # Create manage.py
    with open(project_path / "manage.py", "w") as f:
        f.write(f'''#!/usr/bin/env python
"""Django's command-line utility for administrative tasks."""
import os
import sys


def main():
    """Run administrative tasks."""
    os.environ.setdefault('DJANGO_SETTINGS_MODULE', '{project_name}.settings')
    try:
        from django.core.management import execute_from_command_line
    except ImportError as exc:
        raise ImportError(
            "Couldn't import Django. Are you sure it's installed and "
            "available on your PYTHONPATH environment variable? Did you "
            "forget to activate a virtual environment?"
        ) from exc
    execute_from_command_line(sys.argv)


if __name__ == '__main__':
    main()
''')
    
    # Create core app files
    with open(project_path / "core" / "__init__.py", "w") as f:
        f.write('')
    
    with open(project_path / "core" / "apps.py", "w") as f:
        f.write(f'''from django.apps import AppConfig

class CoreConfig(AppConfig):
    default_auto_field = 'django.db.models.BigAutoField'
    name = 'core'
''')
    
    with open(project_path / "core" / "views.py", "w") as f:
        f.write('''from django.shortcuts import render
from django.http import JsonResponse

def index(request):
    return render(request, 'index.html')

def health_check(request):
    return JsonResponse({'status': 'ok'})
''')
    
    with open(project_path / "core" / "urls.py", "w") as f:
        f.write('''from django.urls import path
from . import views

urlpatterns = [
    path('', views.index, name='index'),
    path('health/', views.health_check, name='health'),
]
''')
    
    # Create API app files
    with open(project_path / "api" / "__init__.py", "w") as f:
        f.write('')
    
    with open(project_path / "api" / "apps.py", "w") as f:
        f.write(f'''from django.apps import AppConfig

class ApiConfig(AppConfig):
    default_auto_field = 'django.db.models.BigAutoField'
    name = 'api'
''')
    
    with open(project_path / "api" / "models.py", "w") as f:
        f.write('''from django.db import models

class User(models.Model):
    first_name = models.CharField(max_length=50)
    last_name = models.CharField(max_length=50)
    email = models.EmailField(unique=True)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{self.first_name} {self.last_name}"
''')
    
    with open(project_path / "api" / "serializers.py", "w") as f:
        f.write('''from rest_framework import serializers
from .models import User

class UserSerializer(serializers.ModelSerializer):
    class Meta:
        model = User
        fields = ['id', 'first_name', 'last_name', 'email', 'created_at']
''')
    
    with open(project_path / "api" / "views.py", "w") as f:
        f.write('''from rest_framework.decorators import api_view
from rest_framework.response import Response
from rest_framework import status
from .models import User
from .serializers import UserSerializer

@api_view(['GET', 'POST'])
def user_list(request):
    if request.method == 'GET':
        users = User.objects.all()
        serializer = UserSerializer(users, many=True)
        return Response(serializer.data)
    
    elif request.method == 'POST':
        serializer = UserSerializer(data=request.data)
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data, status=status.HTTP_201_CREATED)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

@api_view(['GET', 'PUT', 'DELETE'])
def user_detail(request, pk):
    try:
        user = User.objects.get(pk=pk)
    except User.DoesNotExist:
        return Response(status=status.HTTP_404_NOT_FOUND)

    if request.method == 'GET':
        serializer = UserSerializer(user)
        return Response(serializer.data)
    
    elif request.method == 'PUT':
        serializer = UserSerializer(user, data=request.data)
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    
    elif request.method == 'DELETE':
        user.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)
''')
    
    with open(project_path / "api" / "urls.py", "w") as f:
        f.write('''from django.urls import path
from . import views

urlpatterns = [
    path('users/', views.user_list, name='user-list'),
    path('users/<int:pk>/', views.user_detail, name='user-detail'),
]
''')
    
    # Create requirements.txt
    with open(project_path / "requirements.txt", "w") as f:
        f.write('''Django==5.2.10
djangorestframework==3.15.2
django-cors-headers==4.6.0
python-decouple==3.8
gunicorn==23.0.0
whitenoise==6.8.2
psycopg2-binary==3.2.3
''')
    
    # Create README.md
    with open(project_path / "README.md", "w") as f:
        f.write(f'''# {project_name}

A Django project template generated with Codeius AI.

## Setup

```bash
pip install -r requirements.txt
python manage.py migrate
python manage.py runserver
```

## Features

- Django REST Framework for API development
- CORS headers for frontend integration
- Basic user model and API endpoints
- Admin panel configured
- Environment configuration ready

## API Endpoints

- `GET /api/users/` - List all users
- `POST /api/users/` - Create a new user
- `GET /api/users/<id>/` - Get specific user
- `PUT /api/users/<id>/` - Update specific user
- `DELETE /api/users/<id>/` - Delete specific user
''')
    
    # Create .gitignore
    with open(project_path / ".gitignore", "w") as f:
        f.write('''*.pyc
__pycache__/
.env
db.sqlite3
media/
staticfiles/
.DS_Store
''')
    
    # Create basic templates directory
    (project_path / "templates").mkdir(parents=True, exist_ok=True)
    with open(project_path / "templates" / "index.html", "w") as f:
        f.write('''<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Django App</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-dark">
        <div class="container">
            <a class="navbar-brand" href="#">Django App</a>
        </div>
    </nav>
    
    <div class="container mt-5">
        <div class="jumbotron">
            <h1 class="display-4">Welcome to Django</h1>
            <p class="lead">Your Django project is ready!</p>
        </div>
    </div>
</body>
</html>
''')
    
    print(f"Django project '{project_name}' created successfully!")