"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StatefulSetUpdateStrategy = exports.StatefulSet = exports.PodManagementPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const workload = require("./workload");
/**
 * Controls how pods are created during initial scale up, when replacing pods on nodes,
 * or when scaling down.
 *
 * The default policy is `OrderedReady`, where pods are created in increasing order
 * (pod-0, then pod-1, etc) and the controller will wait until each pod is ready before
 * continuing. When scaling down, the pods are removed in the opposite order.
 *
 * The alternative policy is `Parallel` which will create pods in parallel to match the
 * desired scale without waiting, and on scale down will delete all pods at once.
 */
var PodManagementPolicy;
(function (PodManagementPolicy) {
    PodManagementPolicy["ORDERED_READY"] = "OrderedReady";
    PodManagementPolicy["PARALLEL"] = "Parallel";
})(PodManagementPolicy = exports.PodManagementPolicy || (exports.PodManagementPolicy = {}));
/**
 * StatefulSet is the workload API object used to manage stateful applications.
 *
 * Manages the deployment and scaling of a set of Pods, and provides guarantees
 * about the ordering and uniqueness of these Pods.
 *
 * Like a Deployment, a StatefulSet manages Pods that are based on an identical
 * container spec. Unlike a Deployment, a StatefulSet maintains a sticky identity
 * for each of their Pods. These pods are created from the same spec, but are not
 * interchangeable: each has a persistent identifier that it maintains across any
 * rescheduling.
 *
 * If you want to use storage volumes to provide persistence for your workload, you
 * can use a StatefulSet as part of the solution. Although individual Pods in a StatefulSet
 * are susceptible to failure, the persistent Pod identifiers make it easier to match existing
 * volumes to the new Pods that replace any that have failed.
 *
 * Using StatefulSets
 * ------------------
 * StatefulSets are valuable for applications that require one or more of the following.
 *
 * - Stable, unique network identifiers.
 * - Stable, persistent storage.
 * - Ordered, graceful deployment and scaling.
 * - Ordered, automated rolling updates.
 */
class StatefulSet extends workload.Workload {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.resourceType = 'statefulsets';
        this.apiObject = new k8s.KubeStatefulSet(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this._service = props.service;
        this.apiObject.addDependency(this._service);
        this.replicas = props.replicas ?? 1;
        this.strategy = props.strategy ?? StatefulSetUpdateStrategy.rollingUpdate(),
            this.podManagementPolicy = props.podManagementPolicy ?? PodManagementPolicy.ORDERED_READY;
        this.minReady = props.minReady ?? cdk8s_1.Duration.seconds(0);
        this._service.select(this);
    }
    /**
      * @internal
      */
    _toKube() {
        return {
            serviceName: this._service.name,
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: this._toLabelSelector(),
            podManagementPolicy: this.podManagementPolicy,
            updateStrategy: this.strategy._toKube(),
        };
    }
}
exports.StatefulSet = StatefulSet;
_a = JSII_RTTI_SYMBOL_1;
StatefulSet[_a] = { fqn: "cdk8s-plus-22.StatefulSet", version: "2.0.0-rc.52" };
/**
 * StatefulSet update strategies.
 */
class StatefulSetUpdateStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * The controller will not automatically update the Pods in a StatefulSet.
     * Users must manually delete Pods to cause the controller to create new Pods
     * that reflect modifications.
     */
    static onDelete() {
        return new StatefulSetUpdateStrategy({
            type: 'OnDelete',
        });
    }
    /**
     * The controller will delete and recreate each Pod in the StatefulSet.
     * It will proceed in the same order as Pod termination (from the largest ordinal to the smallest),
     * updating each Pod one at a time. The Kubernetes control plane waits until an updated
     * Pod is Running and Ready prior to updating its predecessor.
     */
    static rollingUpdate(options = {}) {
        return new StatefulSetUpdateStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { partition: options.partition ?? 0 },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.StatefulSetUpdateStrategy = StatefulSetUpdateStrategy;
_b = JSII_RTTI_SYMBOL_1;
StatefulSetUpdateStrategy[_b] = { fqn: "cdk8s-plus-22.StatefulSetUpdateStrategy", version: "2.0.0-rc.52" };
//# sourceMappingURL=data:application/json;base64,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