"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsRunTask = exports.EcsEc2LaunchTarget = exports.EcsFargateLaunchTarget = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Configuration for running an ECS task on Fargate.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-fargate
 * @stability stable
 */
class EcsFargateLaunchTarget {
    /**
     * @stability stable
     */
    constructor(options) {
        this.options = options;
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsFargateLaunchTargetOptions(options);
    }
    /**
     * Called when the Fargate launch type configured on RunTask.
     *
     * @stability stable
     */
    bind(_task, launchTargetOptions) {
        var _d;
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsRunTask(_task);
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_LaunchTargetBindOptions(launchTargetOptions);
        if (!launchTargetOptions.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not compatible with Fargate');
        }
        return {
            parameters: {
                LaunchType: 'FARGATE',
                PlatformVersion: (_d = this.options) === null || _d === void 0 ? void 0 : _d.platformVersion,
            },
        };
    }
}
exports.EcsFargateLaunchTarget = EcsFargateLaunchTarget;
_a = JSII_RTTI_SYMBOL_1;
EcsFargateLaunchTarget[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsFargateLaunchTarget", version: "1.144.0" };
/**
 * Configuration for running an ECS task on EC2.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-ec2
 * @stability stable
 */
class EcsEc2LaunchTarget {
    /**
     * @stability stable
     */
    constructor(options) {
        this.options = options;
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsEc2LaunchTargetOptions(options);
    }
    /**
     * Called when the EC2 launch type is configured on RunTask.
     *
     * @stability stable
     */
    bind(_task, launchTargetOptions) {
        var _d, _e, _f, _g, _h;
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsRunTask(_task);
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_LaunchTargetBindOptions(launchTargetOptions);
        if (!launchTargetOptions.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not compatible with EC2');
        }
        if (!((_d = launchTargetOptions.cluster) === null || _d === void 0 ? void 0 : _d.hasEc2Capacity)) {
            throw new Error('Cluster for this service needs Ec2 capacity. Call addCapacity() on the cluster.');
        }
        return {
            parameters: {
                LaunchType: 'EC2',
                // takes an array of placement constraints each of which contain a single item array of constraints, flattens it
                // and renders the Json to be passed as a parameter in the state machine.
                // input: [ecs.PlacementConstraint.distinctInstances()] - distinctInstances() returns [{ type: 'distinctInstance' }]
                // output: {Type: 'distinctInstance'}
                PlacementConstraints: noEmpty(flatten(((_f = (_e = this.options) === null || _e === void 0 ? void 0 : _e.placementConstraints) !== null && _f !== void 0 ? _f : []).map((c) => c.toJson().map(uppercaseKeys)))),
                PlacementStrategy: noEmpty(flatten(((_h = (_g = this.options) === null || _g === void 0 ? void 0 : _g.placementStrategies) !== null && _h !== void 0 ? _h : []).map((c) => c.toJson().map(uppercaseKeys)))),
            },
        };
        function uppercaseKeys(obj) {
            const ret = {};
            for (const key of Object.keys(obj)) {
                ret[key.slice(0, 1).toUpperCase() + key.slice(1)] = obj[key];
            }
            return ret;
        }
        function flatten(xs) {
            return Array.prototype.concat([], ...xs);
        }
        function noEmpty(xs) {
            if (xs.length === 0) {
                return undefined;
            }
            return xs;
        }
    }
}
exports.EcsEc2LaunchTarget = EcsEc2LaunchTarget;
_b = JSII_RTTI_SYMBOL_1;
EcsEc2LaunchTarget[_b] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsEc2LaunchTarget", version: "1.144.0" };
/**
 * Run a Task on ECS or Fargate.
 *
 * @stability stable
 */
class EcsRunTask extends sfn.TaskStateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e, _f;
        super(scope, id, props);
        this.props = props;
        /**
         * Manage allowed network traffic for this service.
         *
         * @stability stable
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsRunTaskProps(props);
        this.integrationPattern = (_d = props.integrationPattern) !== null && _d !== void 0 ? _d : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken((_e = props.containerOverrides) === null || _e === void 0 ? void 0 : _e.map(override => override.environment))) {
            throw new Error('Task Token is required in at least one `containerOverrides.environment` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (!this.props.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
        if (this.props.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworking();
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroup from ASG.
            this.validateNoNetworkingProps();
            this.connections.addSecurityGroup(...this.props.cluster.connections.securityGroups);
        }
        for (const override of (_f = this.props.containerOverrides) !== null && _f !== void 0 ? _f : []) {
            const name = override.containerDefinition.containerName;
            if (!cdk.Token.isUnresolved(name)) {
                const cont = this.props.taskDefinition.findContainer(name);
                if (!cont) {
                    throw new Error(`Overrides mention container with name '${name}', but no such container in task definition`);
                }
            }
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('ecs', 'runTask', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Cluster: this.props.cluster.clusterArn,
                TaskDefinition: this.props.taskDefinition.family,
                NetworkConfiguration: this.networkConfiguration,
                Overrides: renderOverrides(this.props.containerOverrides),
                ...this.props.launchTarget.bind(this, { taskDefinition: this.props.taskDefinition, cluster: this.props.cluster }).parameters,
            }),
        };
    }
    configureAwsVpcNetworking() {
        var _d, _e;
        const subnetSelection = (_d = this.props.subnets) !== null && _d !== void 0 ? _d : { subnetType: this.props.assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE };
        this.networkConfiguration = {
            AwsvpcConfiguration: {
                AssignPublicIp: this.props.assignPublicIp ? (this.props.assignPublicIp ? 'ENABLED' : 'DISABLED') : undefined,
                Subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                SecurityGroups: cdk.Lazy.list({ produce: () => { var _d; return (_d = this.securityGroups) === null || _d === void 0 ? void 0 : _d.map(sg => sg.securityGroupId); } }),
            },
        };
        // Make sure we have a security group if we're using AWSVPC networking
        this.securityGroups = (_e = this.props.securityGroups) !== null && _e !== void 0 ? _e : [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc: this.props.cluster.vpc })];
        this.connections.addSecurityGroup(...this.securityGroups);
    }
    validateNoNetworkingProps() {
        if (this.props.subnets !== undefined || this.props.securityGroups !== undefined) {
            throw new Error(`Supplied TaskDefinition must have 'networkMode' of 'AWS_VPC' to use 'vpcSubnets' and 'securityGroup'. Received: ${this.props.taskDefinition.networkMode}`);
        }
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/ecs-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.getTaskDefinitionFamilyArn()],
            }),
            new iam.PolicyStatement({
                actions: ['ecs:StopTask', 'ecs:DescribeTasks'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: this.taskExecutionRoles().map((r) => r.roleArn),
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForECSTaskRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Returns the ARN of the task definition family by removing the
     * revision from the task definition ARN
     * Before - arn:aws:ecs:us-west-2:123456789012:task-definition/hello_world:8
     * After - arn:aws:ecs:us-west-2:123456789012:task-definition/hello_world
     */
    getTaskDefinitionFamilyArn() {
        const arnComponents = cdk.Stack.of(this).splitArn(this.props.taskDefinition.taskDefinitionArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        let { resourceName } = arnComponents;
        if (resourceName) {
            resourceName = resourceName.split(':')[0];
        }
        return cdk.Stack.of(this).formatArn({
            partition: arnComponents.partition,
            service: arnComponents.service,
            account: arnComponents.account,
            region: arnComponents.region,
            resource: arnComponents.resource,
            arnFormat: arnComponents.arnFormat,
            resourceName,
        });
    }
    taskExecutionRoles() {
        // Need to be able to pass both Task and Execution role, apparently
        const ret = new Array();
        ret.push(this.props.taskDefinition.taskRole);
        if (this.props.taskDefinition.executionRole) {
            ret.push(this.props.taskDefinition.executionRole);
        }
        return ret;
    }
}
exports.EcsRunTask = EcsRunTask;
_c = JSII_RTTI_SYMBOL_1;
EcsRunTask[_c] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsRunTask", version: "1.144.0" };
EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderOverrides(containerOverrides) {
    var _d;
    if (!containerOverrides || containerOverrides.length === 0) {
        return undefined;
    }
    const ret = new Array();
    for (const override of containerOverrides) {
        ret.push({
            Name: override.containerDefinition.containerName,
            Command: override.command,
            Cpu: override.cpu,
            Memory: override.memoryLimit,
            MemoryReservation: override.memoryReservation,
            Environment: (_d = override.environment) === null || _d === void 0 ? void 0 : _d.map((e) => ({
                Name: e.name,
                Value: e.value,
            })),
        });
    }
    return { ContainerOverrides: ret };
}
//# sourceMappingURL=data:application/json;base64,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