"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
// CDK parameters
const CONSTRUCT_NAME = 'aws-cdk-dynamodb-global';
// DynamoDB table parameters
const TABLE_NAME = 'GlobalTable';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: aws_dynamodb_1.AttributeType.STRING };
const STACK_PROPS = {
    partitionKey: TABLE_PARTITION_KEY,
    tableName: TABLE_NAME,
    regions: ['us-east-1', 'us-east-2', 'us-west-2'],
};
module.exports = {
    'Default Global DynamoDB stack': {
        'global dynamo'(test) {
            const stack = new core_1.Stack();
            new lib_1.GlobalTable(stack, CONSTRUCT_NAME, STACK_PROPS);
            const topStack = stack.node.findChild(CONSTRUCT_NAME);
            for (const reg of STACK_PROPS.regions) {
                const tableStack = topStack.node.findChild(CONSTRUCT_NAME + '-' + reg);
                assert_1.expect(tableStack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                    'KeySchema': [
                        {
                            'AttributeName': 'hashKey',
                            'KeyType': 'HASH',
                        },
                    ],
                    'AttributeDefinitions': [
                        {
                            'AttributeName': 'hashKey',
                            'AttributeType': 'S',
                        },
                    ],
                    'StreamSpecification': {
                        'StreamViewType': 'NEW_AND_OLD_IMAGES',
                    },
                    'TableName': 'GlobalTable',
                }));
            }
            const customResourceStack = stack.node.findChild(CONSTRUCT_NAME + '-CustomResource');
            assert_1.expect(customResourceStack).to(assert_1.haveResource('AWS::Lambda::Function', {
                Description: 'Lambda to make DynamoDB a global table',
                Handler: 'index.handler',
                Timeout: 300,
            }));
            assert_1.expect(customResourceStack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
                Regions: STACK_PROPS.regions,
                ResourceType: 'Custom::DynamoGlobalTableCoordinator',
                TableName: TABLE_NAME,
            }));
            test.done();
        },
    },
    'GlobalTable generated stacks inherit their account from the parent stack'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'GlobalTableStack', { env: { account: '123456789012', region: 'us-east-1' } });
        const globalTable = new lib_1.GlobalTable(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            partitionKey: TABLE_PARTITION_KEY,
            regions: ['us-east-1', 'us-west-2'],
            stream: aws_dynamodb_1.StreamViewType.NEW_AND_OLD_IMAGES,
        });
        new core_1.CfnOutput(stack, 'DynamoDbOutput', {
            // this works, because both `stack` and `regionTables[0]` stack are in the same account & region
            value: globalTable.regionalTables[0].tableStreamArn,
        });
        assert_1.expect(stack).toMatch({
            'Outputs': {
                'DynamoDbOutput': {
                    'Value': {
                        'Fn::ImportValue': 'GlobalTableStackawscdkdynamodbglobalawscdkdynamodbglobaluseast19C1C8A14:awscdkdynamodbglobalawscdkdynamodbglobaluseast1ExportsOutputFnGetAttawscdkdynamodbglobalGlobalTableuseast1FC03DD69StreamArn28E90DB8',
                    },
                },
            },
        });
        test.done();
    },
    'Enforce StreamSpecification': {
        'global dynamo should only allow NEW_AND_OLD_IMAGES'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new lib_1.GlobalTable(stack, CONSTRUCT_NAME, {
                    tableName: TABLE_NAME,
                    stream: aws_dynamodb_1.StreamViewType.KEYS_ONLY,
                    partitionKey: TABLE_PARTITION_KEY,
                    regions: ['us-east-1', 'us-east-2', 'us-west-2'],
                });
            }, /dynamoProps.stream MUST be set to dynamodb.StreamViewType.NEW_AND_OLD_IMAGES/);
            test.done();
        },
    },
    'Check getting tables': {
        'global dynamo should only allow NEW_AND_OLD_IMAGES'(test) {
            const stack = new core_1.Stack();
            const regTables = new lib_1.GlobalTable(stack, CONSTRUCT_NAME, {
                tableName: TABLE_NAME,
                partitionKey: TABLE_PARTITION_KEY,
                regions: ['us-east-1', 'us-east-2', 'us-west-2'],
            });
            test.equal(regTables.regionalTables.length, 3);
            for (const table of regTables.regionalTables) {
                test.ok(table instanceof aws_dynamodb_1.Table);
            }
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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