"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToDynamoDB = void 0;
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const core_2 = require("@aws-solutions-constructs/core");
/**
 * @summary The ApiGatewayToDynamoDB class.
 */
class ApiGatewayToDynamoDB extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayToDynamoDB class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        let partitionKeyName;
        let dynamoTableProps;
        // Set the default props for DynamoDB table
        if (props.dynamoTableProps) {
            dynamoTableProps = core_2.overrideProps(defaults.DefaultTableProps, props.dynamoTableProps);
            partitionKeyName = dynamoTableProps.partitionKey.name;
        }
        else {
            partitionKeyName = defaults.DefaultTableProps.partitionKey.name;
            dynamoTableProps = defaults.DefaultTableProps;
        }
        if (props.existingTableObj) {
            partitionKeyName = core_2.getPartitionKeyNameFromTable(props.existingTableObj);
        }
        this.dynamoTable = defaults.buildDynamoDBTable(this, {
            existingTableObj: props.existingTableObj,
            dynamoTableProps,
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway Resource
        const apiGatewayResource = this.apiGateway.root.addResource("{" + partitionKeyName + "}");
        // Setup API Gateway Method
        // Create
        if (props.allowCreateOperation && props.allowCreateOperation === true && props.createRequestTemplate) {
            const createRequestTemplate = props.createRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:PutItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "PutItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "POST",
                apiResource: this.apiGateway.root,
                requestTemplate: createRequestTemplate
            });
        }
        // Read
        if (props.allowReadOperation === undefined || props.allowReadOperation === true) {
            const getRequestTemplate = "{\r\n\"TableName\": \"" + this.dynamoTable.tableName + "\",\r\n \"KeyConditionExpression\": \"" + partitionKeyName + " = :v1\",\r\n    \"ExpressionAttributeValues\": {\r\n        \":v1\": {\r\n            \"S\": \"$input.params('" + partitionKeyName + "')\"\r\n        }\r\n    }\r\n}";
            this.addActionToPolicy("dynamodb:Query");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "Query",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "GET",
                apiResource: apiGatewayResource,
                requestTemplate: getRequestTemplate
            });
        }
        // Update
        if (props.allowUpdateOperation && props.allowUpdateOperation === true && props.updateRequestTemplate) {
            const updateRequestTemplate = props.updateRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:UpdateItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "UpdateItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "PUT",
                apiResource: apiGatewayResource,
                requestTemplate: updateRequestTemplate
            });
        }
        // Delete
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            const deleteRequestTemplate = "{\r\n  \"TableName\": \"" + this.dynamoTable.tableName + "\",\r\n  \"Key\": {\r\n    \"" + partitionKeyName + "\": {\r\n      \"S\": \"$input.params('" + partitionKeyName + "')\"\r\n    }\r\n  },\r\n  \"ConditionExpression\": \"attribute_not_exists(Replies)\",\r\n  \"ReturnValues\": \"ALL_OLD\"\r\n}";
            this.addActionToPolicy("dynamodb:DeleteItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "DeleteItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "DELETE",
                apiResource: apiGatewayResource,
                requestTemplate: deleteRequestTemplate
            });
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.dynamoTable.tableArn
            ],
            actions: [`${action}`]
        }));
    }
}
exports.ApiGatewayToDynamoDB = ApiGatewayToDynamoDB;
//# sourceMappingURL=data:application/json;base64,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