"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
const create_transform_job_1 = require("../../lib/sagemaker/create-transform-job");
let stack;
let role;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
        ],
    });
});
test('create basic transform job', () => {
    // WHEN
    const task = new create_transform_job_1.SageMakerCreateTransformJob(stack, 'TransformTask', {
        transformJobName: 'MyTransformJob',
        modelName: 'MyModelName',
        transformInput: {
            transformDataSource: {
                s3DataSource: {
                    s3Uri: 's3://inputbucket/prefix',
                },
            },
        },
        transformOutput: {
            s3OutputPath: 's3://outputbucket/prefix',
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTransformJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            TransformJobName: 'MyTransformJob',
            ModelName: 'MyModelName',
            TransformInput: {
                DataSource: {
                    S3DataSource: {
                        S3Uri: 's3://inputbucket/prefix',
                        S3DataType: 'S3Prefix',
                    },
                },
            },
            TransformOutput: {
                S3OutputPath: 's3://outputbucket/prefix',
            },
            TransformResources: {
                InstanceCount: 1,
                InstanceType: 'ml.m4.xlarge',
            },
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new create_transform_job_1.SageMakerCreateTransformJob(stack, 'TransformTask', {
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            transformJobName: 'MyTransformJob',
            modelName: 'MyModelName',
            transformInput: {
                transformDataSource: {
                    s3DataSource: {
                        s3Uri: 's3://inputbucket/prefix',
                    },
                },
            },
            transformOutput: {
                s3OutputPath: 's3://outputbucket/prefix',
            },
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,RUN_JOB. Received: WAIT_FOR_TASK_TOKEN/);
});
test('create complex transform job', () => {
    // WHEN
    const kmsKey = new kms.Key(stack, 'Key');
    const task = new create_transform_job_1.SageMakerCreateTransformJob(stack, 'TransformTask', {
        transformJobName: 'MyTransformJob',
        modelName: 'MyModelName',
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        role,
        transformInput: {
            transformDataSource: {
                s3DataSource: {
                    s3Uri: 's3://inputbucket/prefix',
                    s3DataType: tasks.S3DataType.S3_PREFIX,
                },
            },
        },
        transformOutput: {
            s3OutputPath: 's3://outputbucket/prefix',
            encryptionKey: kmsKey,
        },
        transformResources: {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
            volumeEncryptionKey: kmsKey,
        },
        tags: {
            Project: 'MyProject',
        },
        batchStrategy: tasks.BatchStrategy.MULTI_RECORD,
        environment: {
            SOMEVAR: 'myvalue',
        },
        maxConcurrentTransforms: 3,
        maxPayload: cdk.Size.mebibytes(100),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTransformJob.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            TransformJobName: 'MyTransformJob',
            ModelName: 'MyModelName',
            TransformInput: {
                DataSource: {
                    S3DataSource: {
                        S3Uri: 's3://inputbucket/prefix',
                        S3DataType: 'S3Prefix',
                    },
                },
            },
            TransformOutput: {
                S3OutputPath: 's3://outputbucket/prefix',
                KmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            TransformResources: {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeKmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            Tags: [
                { Key: 'Project', Value: 'MyProject' },
            ],
            MaxConcurrentTransforms: 3,
            MaxPayloadInMB: 100,
            Environment: {
                SOMEVAR: 'myvalue',
            },
            BatchStrategy: 'MultiRecord',
        },
    });
});
test('pass param to transform job', () => {
    // WHEN
    const task = new create_transform_job_1.SageMakerCreateTransformJob(stack, 'TransformTask', {
        transformJobName: sfn.JsonPath.stringAt('$.TransformJobName'),
        modelName: sfn.JsonPath.stringAt('$.ModelName'),
        role,
        transformInput: {
            transformDataSource: {
                s3DataSource: {
                    s3Uri: 's3://inputbucket/prefix',
                    s3DataType: tasks.S3DataType.S3_PREFIX,
                },
            },
        },
        transformOutput: {
            s3OutputPath: 's3://outputbucket/prefix',
        },
        transformResources: {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTransformJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            'TransformJobName.$': '$.TransformJobName',
            'ModelName.$': '$.ModelName',
            'TransformInput': {
                DataSource: {
                    S3DataSource: {
                        S3Uri: 's3://inputbucket/prefix',
                        S3DataType: 'S3Prefix',
                    },
                },
            },
            'TransformOutput': {
                S3OutputPath: 's3://outputbucket/prefix',
            },
            'TransformResources': {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
            },
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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