"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('create basic model', () => {
    // WHEN
    const task = new tasks.SageMakerCreateModel(stack, 'SagemakerModel', {
        modelName: 'MyModel',
        primaryContainer: new tasks.ContainerDefinition({
            image: tasks.DockerImage.fromJsonExpression(sfn.JsonPath.stringAt('$.Model.imageName')),
            mode: tasks.Mode.SINGLE_MODEL,
            modelS3Location: tasks.S3Location.fromJsonExpression('$.TrainingJob.ModelArtifacts.S3ModelArtifacts'),
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createModel',
                ],
            ],
        },
        End: true,
        Parameters: {
            ExecutionRoleArn: { 'Fn::GetAtt': ['SagemakerModelSagemakerRoleF5035084', 'Arn'] },
            ModelName: 'MyModel',
            PrimaryContainer: {
                'Image.$': '$.Model.imageName',
                'ModelDataUrl.$': '$.TrainingJob.ModelArtifacts.S3ModelArtifacts',
                'Mode': 'SingleModel',
            },
        },
    });
});
test('create complex model', () => {
    // WHEN
    const vpc = new ec2.Vpc(stack, 'VPC');
    const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc, description: 'My SG' });
    securityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'allow ssh access from the world');
    const role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
        ],
    });
    const task = new tasks.SageMakerCreateModel(stack, 'SagemakerModel', {
        modelName: sfn.JsonPath.stringAt('$.ModelName'),
        primaryContainer: new tasks.ContainerDefinition({
            image: tasks.DockerImage.fromJsonExpression(sfn.JsonPath.stringAt('$.Model.imageName')),
            mode: tasks.Mode.MULTI_MODEL,
            modelS3Location: tasks.S3Location.fromJsonExpression('$.TrainingJob.ModelArtifacts.S3ModelArtifacts'),
        }),
        enableNetworkIsolation: true,
        role,
        tags: sfn.TaskInput.fromObject([{
                Key: 'Project',
                Value: 'ML',
            }]),
        vpc,
    });
    task.addSecurityGroup(securityGroup);
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createModel',
                ],
            ],
        },
        End: true,
        Parameters: {
            'ExecutionRoleArn': { 'Fn::GetAtt': ['Role1ABCC5F0', 'Arn'] },
            'ModelName.$': '$.ModelName',
            'PrimaryContainer': {
                'Image.$': '$.Model.imageName',
                'ModelDataUrl.$': '$.TrainingJob.ModelArtifacts.S3ModelArtifacts',
                'Mode': 'MultiModel',
            },
            'EnableNetworkIsolation': true,
            'Tags': [
                { Key: 'Project', Value: 'ML' },
            ],
            'VpcConfig': {
                SecurityGroupIds: [
                    { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
                    { 'Fn::GetAtt': ['SagemakerModelModelSecurityGroup4D2A2C36', 'GroupId'] },
                ],
                Subnets: [
                    { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                    { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                ],
            },
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new tasks.SageMakerCreateModel(stack, 'Sagemaker', {
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            modelName: 'MyModel',
            primaryContainer: new tasks.ContainerDefinition({
                image: tasks.DockerImage.fromJsonExpression(sfn.JsonPath.stringAt('$.Model.imageName')),
                mode: tasks.Mode.SINGLE_MODEL,
                modelS3Location: tasks.S3Location.fromJsonExpression('$.TrainingJob.ModelArtifacts.S3ModelArtifacts'),
            }),
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE. Received: WAIT_FOR_TASK_TOKEN/i);
});
//# sourceMappingURL=data:application/json;base64,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