"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('create endpoint', () => {
    // WHEN
    const task = new tasks.SageMakerCreateEndpoint(stack, 'SagemakerEndpoint', {
        endpointName: 'MyEndpoint',
        endpointConfigName: 'MyEndpointConfig',
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createEndpoint',
                ],
            ],
        },
        End: true,
        Parameters: {
            EndpointConfigName: 'MyEndpointConfig',
            EndpointName: 'MyEndpoint',
        },
    });
});
test('create endpoint with tags', () => {
    // WHEN
    const task = new tasks.SageMakerCreateEndpoint(stack, 'SagemakerEndpoint', {
        endpointName: 'MyEndpoint',
        endpointConfigName: 'MyEndpointConfig',
        tags: sfn.TaskInput.fromObject([{
                Key: 'Label',
                Value: 'ML',
            }]),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createEndpoint',
                ],
            ],
        },
        End: true,
        Parameters: {
            EndpointConfigName: 'MyEndpointConfig',
            EndpointName: 'MyEndpoint',
            Tags: [
                { Key: 'Label', Value: 'ML' },
            ],
        },
    });
});
test('create endpoint with input from task', () => {
    // WHEN
    const task = new tasks.SageMakerCreateEndpoint(stack, 'SagemakerEndpoint', {
        endpointName: sfn.JsonPath.stringAt('$.EndpointName'),
        endpointConfigName: sfn.JsonPath.stringAt('$.EndpointConfig'),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createEndpoint',
                ],
            ],
        },
        End: true,
        Parameters: {
            'EndpointConfigName.$': '$.EndpointConfig',
            'EndpointName.$': '$.EndpointName',
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new tasks.SageMakerCreateEndpoint(stack, 'TrainSagemaker', {
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            endpointConfigName: 'MyEndpointConfig',
            endpointName: 'MyEndpoint',
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE. Received: WAIT_FOR_TASK_TOKEN/i);
});
//# sourceMappingURL=data:application/json;base64,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