"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssembleWith = exports.SplitType = exports.BatchStrategy = exports.AcceleratorType = exports.AcceleratorClass = exports.Mode = exports.ContainerDefinition = exports.CompressionType = exports.InputMode = exports.RecordWrapperType = exports.S3DataDistributionType = exports.S3DataType = exports.DockerImage = exports.S3Location = void 0;
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
/**
 * Constructs `IS3Location` objects.
 *
 * @experimental
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.JsonPath.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
/**
 * Creates `IDockerImage` instances.
 *
 * @experimental
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
/**
 * S3 Data Type.
 *
 * @experimental
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 * @experimental
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 * @experimental
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 * @experimental
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 * @experimental
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Describes the container, as part of model definition.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
 * @experimental
 */
class ContainerDefinition {
    constructor(options) {
        this.options = options;
    }
    /**
     * Called when the ContainerDefinition type configured on Sagemaker Task
     */
    bind(task) {
        var _a, _b, _c;
        return {
            parameters: {
                ContainerHostname: this.options.containerHostName,
                Image: (_a = this.options.image) === null || _a === void 0 ? void 0 : _a.bind(task).imageUri,
                Mode: this.options.mode,
                ModelDataUrl: (_b = this.options.modelS3Location) === null || _b === void 0 ? void 0 : _b.bind(task, { forReading: true }).uri,
                ModelPackageName: this.options.modelPackageName,
                Environment: (_c = this.options.environmentVariables) === null || _c === void 0 ? void 0 : _c.value,
            },
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
/**
 * Specifies how many models the container hosts
 *
 * @experimental
 */
var Mode;
(function (Mode) {
    /**
     * Container hosts a single model
     */
    Mode["SINGLE_MODEL"] = "SingleModel";
    /**
     * Container hosts multiple models
     *
     * @see https://docs.aws.amazon.com/sagemaker/latest/dg/multi-model-endpoints.html
     */
    Mode["MULTI_MODEL"] = "MultiModel";
})(Mode = exports.Mode || (exports.Mode = {}));
/**
 * The generation of Elastic Inference (EI) instance
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 * @experimental
 */
class AcceleratorClass {
    /**
     * @param version - Elastic Inference accelerator generation
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom AcceleratorType
     * @param version - Elastic Inference accelerator generation
    */
    static of(version) { return new AcceleratorClass(version); }
}
exports.AcceleratorClass = AcceleratorClass;
/**
 * Elastic Inference accelerator 1st generation
 */
AcceleratorClass.EIA1 = AcceleratorClass.of('eia1');
/**
 * Elastic Inference accelerator 2nd generation
 */
AcceleratorClass.EIA2 = AcceleratorClass.of('eia2');
/**
 * The size of the Elastic Inference (EI) instance to use for the production variant.
 * EI instances provide on-demand GPU computing for inference
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 * @experimental
 */
class AcceleratorType {
    constructor(instanceTypeIdentifier) {
        this.instanceTypeIdentifier = instanceTypeIdentifier;
    }
    /**
     * AcceleratorType
     *
     * This class takes a combination of a class and size.
     */
    static of(acceleratorClass, instanceSize) {
        return new AcceleratorType(`ml.${acceleratorClass}.${instanceSize}`);
    }
    /**
     * Return the accelerator type as a dotted string
     */
    toString() {
        return this.instanceTypeIdentifier;
    }
}
exports.AcceleratorType = AcceleratorType;
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 * @experimental
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 * @experimental
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 * @experimental
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmFzZS10eXBlcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImJhc2UtdHlwZXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBRUEsNERBQWtGO0FBQ2xGLHdDQUF3QztBQUd4QyxrREFBa0Q7QUF1UmxEOzs7O0dBSUc7QUFDSCxNQUFzQixVQUFVO0lBQzlCOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxNQUFrQixFQUFFLFNBQWlCO1FBQzVELE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxFQUFFLE1BQU0sRUFBRSxTQUFTLEVBQUUsR0FBRyxFQUFFLE1BQU0sQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQzVGLENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksTUFBTSxDQUFDLGtCQUFrQixDQUFDLFVBQWtCO1FBQ2pELE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxFQUFFLEdBQUcsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDNUUsQ0FBQztDQU1GO0FBM0JELGdDQTJCQztBQW1DRDs7OztHQUlHO0FBQ0gsTUFBc0IsV0FBVztJQUMvQjs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxVQUEyQixFQUFFLE1BQWMsUUFBUTtRQUNqRixPQUFPLElBQUksbUJBQW1CLENBQUMsRUFBRSxVQUFVLEVBQUUsUUFBUSxFQUFFLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDaEcsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLGtCQUFrQixDQUFDLFVBQWtCLEVBQUUsb0JBQW9CLEdBQUcsSUFBSTtRQUM5RSxPQUFPLElBQUksbUJBQW1CLENBQUMsRUFBRSxRQUFRLEVBQUUsVUFBVSxFQUFFLG9CQUFvQixFQUFFLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxRQUFnQjtRQUN6QyxPQUFPLElBQUksbUJBQW1CLENBQUMsRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTRCO1FBQ2hGLE1BQU0sS0FBSyxHQUFHLElBQUksaUNBQWdCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNyRCxPQUFPLElBQUksbUJBQW1CLENBQUMsRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsRUFBRSxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7SUFDN0YsQ0FBQztDQU1GO0FBaERELGtDQWdEQztBQUVEOzs7O0dBSUc7QUFDSCxJQUFZLFVBZVg7QUFmRCxXQUFZLFVBQVU7SUFDcEI7O09BRUc7SUFDSCw0Q0FBOEIsQ0FBQTtJQUU5Qjs7T0FFRztJQUNILG9DQUFzQixDQUFBO0lBRXRCOztPQUVHO0lBQ0gsK0RBQWlELENBQUE7QUFDbkQsQ0FBQyxFQWZXLFVBQVUsR0FBVixrQkFBVSxLQUFWLGtCQUFVLFFBZXJCO0FBRUQ7Ozs7R0FJRztBQUNILElBQVksc0JBVVg7QUFWRCxXQUFZLHNCQUFzQjtJQUNoQzs7T0FFRztJQUNILDhEQUFvQyxDQUFBO0lBRXBDOztPQUVHO0lBQ0gsOERBQW9DLENBQUE7QUFDdEMsQ0FBQyxFQVZXLHNCQUFzQixHQUF0Qiw4QkFBc0IsS0FBdEIsOEJBQXNCLFFBVWpDO0FBRUQ7Ozs7R0FJRztBQUNILElBQVksaUJBVVg7QUFWRCxXQUFZLGlCQUFpQjtJQUMzQjs7T0FFRztJQUNILGtDQUFhLENBQUE7SUFFYjs7T0FFRztJQUNILDJDQUFzQixDQUFBO0FBQ3hCLENBQUMsRUFWVyxpQkFBaUIsR0FBakIseUJBQWlCLEtBQWpCLHlCQUFpQixRQVU1QjtBQUVEOzs7O0dBSUc7QUFDSCxJQUFZLFNBVVg7QUFWRCxXQUFZLFNBQVM7SUFDbkI7O09BRUc7SUFDSCwwQkFBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCwwQkFBYSxDQUFBO0FBQ2YsQ0FBQyxFQVZXLFNBQVMsR0FBVCxpQkFBUyxLQUFULGlCQUFTLFFBVXBCO0FBRUQ7Ozs7R0FJRztBQUNILElBQVksZUFVWDtBQVZELFdBQVksZUFBZTtJQUN6Qjs7T0FFRztJQUNILGdDQUFhLENBQUE7SUFFYjs7T0FFRztJQUNILGdDQUFhLENBQUE7QUFDZixDQUFDLEVBVlcsZUFBZSxHQUFmLHVCQUFlLEtBQWYsdUJBQWUsUUFVMUI7QUFxTEQ7Ozs7O0dBS0c7QUFDSCxNQUFhLG1CQUFtQjtJQUU5QixZQUE2QixPQUFtQztRQUFuQyxZQUFPLEdBQVAsT0FBTyxDQUE0QjtJQUFHLENBQUM7SUFFcEU7O09BRUc7SUFDSSxJQUFJLENBQUMsSUFBb0I7O1FBQzlCLE9BQU87WUFDTCxVQUFVLEVBQUU7Z0JBQ1YsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxpQkFBaUI7Z0JBQ2pELEtBQUssUUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssMENBQUUsSUFBSSxDQUFDLElBQUksRUFBRSxRQUFRO2dCQUM5QyxJQUFJLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJO2dCQUN2QixZQUFZLFFBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxlQUFlLDBDQUFFLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLEVBQUUsR0FBRztnQkFDaEYsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0I7Z0JBQy9DLFdBQVcsUUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLG9CQUFvQiwwQ0FBRSxLQUFLO2FBQ3REO1NBQ0YsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQW5CRCxrREFtQkM7QUEyQkQ7Ozs7R0FJRztBQUNILElBQVksSUFXWDtBQVhELFdBQVksSUFBSTtJQUNkOztPQUVHO0lBQ0gsb0NBQTRCLENBQUE7SUFDNUI7Ozs7T0FJRztJQUNILGtDQUEwQixDQUFBO0FBQzVCLENBQUMsRUFYVyxJQUFJLEdBQUosWUFBSSxLQUFKLFlBQUksUUFXZjtBQXlDRDs7Ozs7R0FLRztBQUNILE1BQWEsZ0JBQWdCO0lBYzNCOztPQUVHO0lBQ0gsWUFBb0MsT0FBZTtRQUFmLFlBQU8sR0FBUCxPQUFPLENBQVE7SUFBSSxDQUFDO0lBUnhEOzs7TUFHRTtJQUNLLE1BQU0sQ0FBQyxFQUFFLENBQUMsT0FBZSxJQUFJLE9BQU8sSUFBSSxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7O0FBYjdFLDRDQWtCQztBQWpCQzs7R0FFRztBQUNvQixxQkFBSSxHQUFHLGdCQUFnQixDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUMxRDs7R0FFRztBQUNvQixxQkFBSSxHQUFHLGdCQUFnQixDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQztBQVk1RDs7Ozs7O0dBTUc7QUFDSCxNQUFhLGVBQWU7SUFVMUIsWUFBNkIsc0JBQThCO1FBQTlCLDJCQUFzQixHQUF0QixzQkFBc0IsQ0FBUTtJQUMzRCxDQUFDO0lBVkQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxFQUFFLENBQUMsZ0JBQWtDLEVBQUUsWUFBOEI7UUFDakYsT0FBTyxJQUFJLGVBQWUsQ0FBQyxNQUFNLGdCQUFnQixJQUFJLFlBQVksRUFBRSxDQUFDLENBQUM7SUFDdkUsQ0FBQztJQUtEOztPQUVHO0lBQ0ksUUFBUTtRQUNiLE9BQU8sSUFBSSxDQUFDLHNCQUFzQixDQUFDO0lBQ3JDLENBQUM7Q0FDRjtBQW5CRCwwQ0FtQkM7QUFFRDs7OztHQUlHO0FBQ0gsSUFBWSxhQVdYO0FBWEQsV0FBWSxhQUFhO0lBRXZCOztPQUVHO0lBQ0gsNkNBQTRCLENBQUE7SUFFNUI7O09BRUc7SUFDSCwrQ0FBOEIsQ0FBQTtBQUNoQyxDQUFDLEVBWFcsYUFBYSxHQUFiLHFCQUFhLEtBQWIscUJBQWEsUUFXeEI7QUFFRDs7OztHQUlHO0FBQ0gsSUFBWSxTQXFCWDtBQXJCRCxXQUFZLFNBQVM7SUFFbkI7O09BRUc7SUFDSCwwQkFBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCwwQkFBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCxtQ0FBc0IsQ0FBQTtJQUV0Qjs7T0FFRztJQUNILG1DQUFzQixDQUFBO0FBQ3hCLENBQUMsRUFyQlcsU0FBUyxHQUFULGlCQUFTLEtBQVQsaUJBQVMsUUFxQnBCO0FBRUQ7Ozs7R0FJRztBQUNILElBQVksWUFZWDtBQVpELFdBQVksWUFBWTtJQUV0Qjs7T0FFRztJQUNILDZCQUFhLENBQUE7SUFFYjs7T0FFRztJQUNILDZCQUFhLENBQUE7QUFFZixDQUFDLEVBWlcsWUFBWSxHQUFaLG9CQUFZLEtBQVosb0JBQVksUUFZdkI7QUFFRCxNQUFNLG1CQUFvQixTQUFRLFdBQVc7SUFLM0MsWUFBWSxJQUF3RjtRQUNsRyxLQUFLLEVBQUUsQ0FBQztRQUVSLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDO1FBQ3hELElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQztRQUM5QixJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUM7SUFDcEMsQ0FBQztJQUVNLElBQUksQ0FBQyxJQUFvQjtRQUM5QixJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7WUFDbkIsSUFBSSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDakM7UUFDRCxJQUFJLElBQUksQ0FBQyxvQkFBb0IsRUFBRTtZQUM3QixJQUFJLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3RELE9BQU8sRUFBRTtvQkFDUCxpQ0FBaUM7b0JBQ2pDLDRCQUE0QjtvQkFDNUIsbUJBQW1CO2lCQUNwQjtnQkFDRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7YUFDakIsQ0FBQyxDQUFDLENBQUM7U0FDTDtRQUNELE9BQU87WUFDTCxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7U0FDeEIsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQUVELE1BQU0sa0JBQW1CLFNBQVEsVUFBVTtJQUt6QyxZQUFZLElBQThEO1FBQ3hFLEtBQUssRUFBRSxDQUFDO1FBQ1IsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDO1FBQzFCLElBQUksQ0FBQyxPQUFPLEdBQUcsR0FBRyxJQUFJLENBQUMsU0FBUyxJQUFJLEVBQUUsR0FBRyxDQUFDO1FBQzFDLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQztJQUN0QixDQUFDO0lBRU0sSUFBSSxDQUFDLElBQW9CLEVBQUUsSUFBMkI7UUFDM0QsSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ2YsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO2dCQUNuQixJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO2FBQzNDO1lBQ0QsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO2dCQUNuQixJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO2FBQzVDO1NBQ0Y7YUFBTTtZQUNMLE1BQU0sT0FBTyxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7WUFDcEMsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO2dCQUNuQixPQUFPLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRSxlQUFlLENBQUMsQ0FBQzthQUMvQztZQUNELElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtnQkFDbkIsT0FBTyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQzthQUM5QjtZQUNELElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztTQUN6RjtRQUNELE9BQU8sRUFBRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDO0lBQzNCLENBQUM7Q0FDRiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGVjciBmcm9tICdAYXdzLWNkay9hd3MtZWNyJztcbmltcG9ydCB7IERvY2tlckltYWdlQXNzZXQsIERvY2tlckltYWdlQXNzZXRQcm9wcyB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3ItYXNzZXRzJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGttcyBmcm9tICdAYXdzLWNkay9hd3Mta21zJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBzZm4gZnJvbSAnQGF3cy1jZGsvYXdzLXN0ZXBmdW5jdGlvbnMnO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBEdXJhdGlvbiwgU2l6ZSB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG4vKipcbiAqIFRhc2sgdG8gdHJhaW4gYSBtYWNoaW5lIGxlYXJuaW5nIG1vZGVsIHVzaW5nIEFtYXpvbiBTYWdlTWFrZXJcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJU2FnZU1ha2VyVGFzayBleHRlbmRzIGlhbS5JR3JhbnRhYmxlIHt9XG5cbi8qKlxuICogU3BlY2lmeSB0aGUgdHJhaW5pbmcgYWxnb3JpdGhtIGFuZCBhbGdvcml0aG0tc3BlY2lmaWMgbWV0YWRhdGFcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBbGdvcml0aG1TcGVjaWZpY2F0aW9uIHtcblxuICAvKipcbiAgICogTmFtZSBvZiB0aGUgYWxnb3JpdGhtIHJlc291cmNlIHRvIHVzZSBmb3IgdGhlIHRyYWluaW5nIGpvYi5cbiAgICogVGhpcyBtdXN0IGJlIGFuIGFsZ29yaXRobSByZXNvdXJjZSB0aGF0IHlvdSBjcmVhdGVkIG9yIHN1YnNjcmliZSB0byBvbiBBV1MgTWFya2V0cGxhY2UuXG4gICAqIElmIHlvdSBzcGVjaWZ5IGEgdmFsdWUgZm9yIHRoaXMgcGFyYW1ldGVyLCB5b3UgY2FuJ3Qgc3BlY2lmeSBhIHZhbHVlIGZvciBUcmFpbmluZ0ltYWdlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGFsZ29yaXRobSBpcyBzcGVjaWZpZWRcbiAgICovXG4gIHJlYWRvbmx5IGFsZ29yaXRobU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgbWV0cmljIGRlZmluaXRpb24gb2JqZWN0cy4gRWFjaCBvYmplY3Qgc3BlY2lmaWVzIHRoZSBtZXRyaWMgbmFtZSBhbmQgcmVndWxhciBleHByZXNzaW9ucyB1c2VkIHRvIHBhcnNlIGFsZ29yaXRobSBsb2dzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIG1ldHJpY3NcbiAgICovXG4gIHJlYWRvbmx5IG1ldHJpY0RlZmluaXRpb25zPzogTWV0cmljRGVmaW5pdGlvbltdO1xuXG4gIC8qKlxuICAgKiBSZWdpc3RyeSBwYXRoIG9mIHRoZSBEb2NrZXIgaW1hZ2UgdGhhdCBjb250YWlucyB0aGUgdHJhaW5pbmcgYWxnb3JpdGhtLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIERvY2tlciBpbWFnZSBpcyBzcGVjaWZpZWRcbiAgICovXG4gIHJlYWRvbmx5IHRyYWluaW5nSW1hZ2U/OiBEb2NrZXJJbWFnZTtcblxuICAvKipcbiAgICogSW5wdXQgbW9kZSB0aGF0IHRoZSBhbGdvcml0aG0gc3VwcG9ydHMuXG4gICAqXG4gICAqIEBkZWZhdWx0ICdGaWxlJyBtb2RlXG4gICAqL1xuICByZWFkb25seSB0cmFpbmluZ0lucHV0TW9kZT86IElucHV0TW9kZTtcbn1cblxuLyoqXG4gKiAgRGVzY3JpYmVzIHRoZSB0cmFpbmluZywgdmFsaWRhdGlvbiBvciB0ZXN0IGRhdGFzZXQgYW5kIHRoZSBBbWF6b24gUzMgbG9jYXRpb24gd2hlcmUgaXQgaXMgc3RvcmVkLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDaGFubmVsIHtcblxuICAvKipcbiAgICogTmFtZSBvZiB0aGUgY2hhbm5lbFxuICAgKi9cbiAgcmVhZG9ubHkgY2hhbm5lbE5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogQ29tcHJlc3Npb24gdHlwZSBpZiB0cmFpbmluZyBkYXRhIGlzIGNvbXByZXNzZWRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBjb21wcmVzc2lvblR5cGU/OiBDb21wcmVzc2lvblR5cGU7XG5cbiAgLyoqXG4gICAqIFRoZSBNSU1FIHR5cGUgb2YgdGhlIGRhdGEuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgY29udGVudFR5cGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIExvY2F0aW9uIG9mIHRoZSBjaGFubmVsIGRhdGEuXG4gICAqL1xuICByZWFkb25seSBkYXRhU291cmNlOiBEYXRhU291cmNlO1xuXG4gIC8qKlxuICAgKiBJbnB1dCBtb2RlIHRvIHVzZSBmb3IgdGhlIGRhdGEgY2hhbm5lbCBpbiBhIHRyYWluaW5nIGpvYi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBpbnB1dE1vZGU/OiBJbnB1dE1vZGU7XG5cbiAgLyoqXG4gICAqIFNwZWNpZnkgUmVjb3JkSU8gYXMgdGhlIHZhbHVlIHdoZW4gaW5wdXQgZGF0YSBpcyBpbiByYXcgZm9ybWF0IGJ1dCB0aGUgdHJhaW5pbmcgYWxnb3JpdGhtIHJlcXVpcmVzIHRoZSBSZWNvcmRJTyBmb3JtYXQuXG4gICAqIEluIHRoaXMgY2FzZSwgQW1hem9uIFNhZ2VNYWtlciB3cmFwcyBlYWNoIGluZGl2aWR1YWwgUzMgb2JqZWN0IGluIGEgUmVjb3JkSU8gcmVjb3JkLlxuICAgKiBJZiB0aGUgaW5wdXQgZGF0YSBpcyBhbHJlYWR5IGluIFJlY29yZElPIGZvcm1hdCwgeW91IGRvbid0IG5lZWQgdG8gc2V0IHRoaXMgYXR0cmlidXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IHJlY29yZFdyYXBwZXJUeXBlPzogUmVjb3JkV3JhcHBlclR5cGU7XG5cbiAgLyoqXG4gICAqIFNodWZmbGUgY29uZmlnIG9wdGlvbiBmb3IgaW5wdXQgZGF0YSBpbiBhIGNoYW5uZWwuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgc2h1ZmZsZUNvbmZpZz86IFNodWZmbGVDb25maWc7XG59XG5cbi8qKlxuICogQ29uZmlndXJhdGlvbiBmb3IgYSBzaHVmZmxlIG9wdGlvbiBmb3IgaW5wdXQgZGF0YSBpbiBhIGNoYW5uZWwuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNodWZmbGVDb25maWcge1xuICAvKipcbiAgICogRGV0ZXJtaW5lcyB0aGUgc2h1ZmZsaW5nIG9yZGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgc2VlZDogbnVtYmVyO1xufVxuXG4vKipcbiAqIExvY2F0aW9uIG9mIHRoZSBjaGFubmVsIGRhdGEuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIERhdGFTb3VyY2Uge1xuICAvKipcbiAgICogUzMgbG9jYXRpb24gb2YgdGhlIGRhdGEgc291cmNlIHRoYXQgaXMgYXNzb2NpYXRlZCB3aXRoIGEgY2hhbm5lbC5cbiAgICovXG4gIHJlYWRvbmx5IHMzRGF0YVNvdXJjZTogUzNEYXRhU291cmNlO1xufVxuXG4vKipcbiAqIFMzIGxvY2F0aW9uIG9mIHRoZSBjaGFubmVsIGRhdGEuXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2FnZW1ha2VyL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1MzRGF0YVNvdXJjZS5odG1sXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFMzRGF0YVNvdXJjZSB7XG4gIC8qKlxuICAgKiBMaXN0IG9mIG9uZSBvciBtb3JlIGF0dHJpYnV0ZSBuYW1lcyB0byB1c2UgdGhhdCBhcmUgZm91bmQgaW4gYSBzcGVjaWZpZWQgYXVnbWVudGVkIG1hbmlmZXN0IGZpbGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYXR0cmlidXRlIG5hbWVzXG4gICAqL1xuICByZWFkb25seSBhdHRyaWJ1dGVOYW1lcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBTMyBEYXRhIERpc3RyaWJ1dGlvbiBUeXBlXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgczNEYXRhRGlzdHJpYnV0aW9uVHlwZT86IFMzRGF0YURpc3RyaWJ1dGlvblR5cGU7XG5cbiAgLyoqXG4gICAqIFMzIERhdGEgVHlwZVxuICAgKlxuICAgKiBAZGVmYXVsdCBTM19QUkVGSVhcbiAgICovXG4gIHJlYWRvbmx5IHMzRGF0YVR5cGU/OiBTM0RhdGFUeXBlO1xuXG4gIC8qKlxuICAgKiBTMyBVcmlcbiAgICovXG4gIHJlYWRvbmx5IHMzTG9jYXRpb246IFMzTG9jYXRpb247XG59XG5cbi8qKlxuICogQ29uZmlndXJlcyB0aGUgUzMgYnVja2V0IHdoZXJlIFNhZ2VNYWtlciB3aWxsIHNhdmUgdGhlIHJlc3VsdCBvZiBtb2RlbCB0cmFpbmluZ1xuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIE91dHB1dERhdGFDb25maWcge1xuICAvKipcbiAgICogT3B0aW9uYWwgS01TIGVuY3J5cHRpb24ga2V5IHRoYXQgQW1hem9uIFNhZ2VNYWtlciB1c2VzIHRvIGVuY3J5cHQgdGhlIG1vZGVsIGFydGlmYWN0cyBhdCByZXN0IHVzaW5nIEFtYXpvbiBTMyBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFtYXpvbiBTYWdlTWFrZXIgdXNlcyB0aGUgZGVmYXVsdCBLTVMga2V5IGZvciBBbWF6b24gUzMgZm9yIHlvdXIgcm9sZSdzIGFjY291bnRcbiAgICovXG4gIHJlYWRvbmx5IGVuY3J5cHRpb25LZXk/OiBrbXMuSUtleTtcblxuICAvKipcbiAgICogSWRlbnRpZmllcyB0aGUgUzMgcGF0aCB3aGVyZSB5b3Ugd2FudCBBbWF6b24gU2FnZU1ha2VyIHRvIHN0b3JlIHRoZSBtb2RlbCBhcnRpZmFjdHMuXG4gICAqL1xuICByZWFkb25seSBzM091dHB1dExvY2F0aW9uOiBTM0xvY2F0aW9uO1xufVxuXG4vKipcbiAqIFNwZWNpZmllcyBhIGxpbWl0IHRvIGhvdyBsb25nIGEgbW9kZWwgdHJhaW5pbmcgam9iIGNhbiBydW4uXG4gKiBXaGVuIHRoZSBqb2IgcmVhY2hlcyB0aGUgdGltZSBsaW1pdCwgQW1hem9uIFNhZ2VNYWtlciBlbmRzIHRoZSB0cmFpbmluZyBqb2IuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFN0b3BwaW5nQ29uZGl0aW9uIHtcbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIGxlbmd0aCBvZiB0aW1lLCBpbiBzZWNvbmRzLCB0aGF0IHRoZSB0cmFpbmluZyBvciBjb21waWxhdGlvbiBqb2IgY2FuIHJ1bi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSAxIGhvdXJcbiAgICovXG4gIHJlYWRvbmx5IG1heFJ1bnRpbWU/OiBEdXJhdGlvbjtcbn1cblxuLyoqXG4gKiBTcGVjaWZpZXMgdGhlIHJlc291cmNlcywgTUwgY29tcHV0ZSBpbnN0YW5jZXMsIGFuZCBNTCBzdG9yYWdlIHZvbHVtZXMgdG8gZGVwbG95IGZvciBtb2RlbCB0cmFpbmluZy5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVzb3VyY2VDb25maWcge1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIE1MIGNvbXB1dGUgaW5zdGFuY2VzIHRvIHVzZS5cbiAgICpcbiAgICogQGRlZmF1bHQgMSBpbnN0YW5jZS5cbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlQ291bnQ6IG51bWJlcjtcblxuICAvKipcbiAgICogTUwgY29tcHV0ZSBpbnN0YW5jZSB0eXBlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBpcyB0aGUgJ200LnhsYXJnZScgaW5zdGFuY2UgdHlwZS5cbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZTogZWMyLkluc3RhbmNlVHlwZTtcblxuICAvKipcbiAgICogS01TIGtleSB0aGF0IEFtYXpvbiBTYWdlTWFrZXIgdXNlcyB0byBlbmNyeXB0IGRhdGEgb24gdGhlIHN0b3JhZ2Ugdm9sdW1lIGF0dGFjaGVkIHRvIHRoZSBNTCBjb21wdXRlIGluc3RhbmNlKHMpIHRoYXQgcnVuIHRoZSB0cmFpbmluZyBqb2IuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQW1hem9uIFNhZ2VNYWtlciB1c2VzIHRoZSBkZWZhdWx0IEtNUyBrZXkgZm9yIEFtYXpvbiBTMyBmb3IgeW91ciByb2xlJ3MgYWNjb3VudFxuICAgKi9cbiAgcmVhZG9ubHkgdm9sdW1lRW5jcnlwdGlvbktleT86IGttcy5JS2V5O1xuXG4gIC8qKlxuICAgKiBTaXplIG9mIHRoZSBNTCBzdG9yYWdlIHZvbHVtZSB0aGF0IHlvdSB3YW50IHRvIHByb3Zpc2lvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgMTAgR0IgRUJTIHZvbHVtZS5cbiAgICovXG4gIHJlYWRvbmx5IHZvbHVtZVNpemU6IFNpemU7XG59XG5cbi8qKlxuICogU3BlY2lmaWVzIHRoZSBWUEMgdGhhdCB5b3Ugd2FudCB5b3VyIEFtYXpvbiBTYWdlTWFrZXIgdHJhaW5pbmcgam9iIHRvIGNvbm5lY3QgdG8uXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFZwY0NvbmZpZyB7XG4gIC8qKlxuICAgKiBWUENcbiAgICovXG4gIHJlYWRvbmx5IHZwYzogZWMyLklWcGM7XG5cbiAgLyoqXG4gICAqIFZQQyBzdWJuZXRzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFByaXZhdGUgU3VibmV0cyBhcmUgc2VsZWN0ZWRcbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldHM/OiBlYzIuU3VibmV0U2VsZWN0aW9uO1xufVxuXG4vKipcbiAqIFNwZWNpZmllcyB0aGUgbWV0cmljIG5hbWUgYW5kIHJlZ3VsYXIgZXhwcmVzc2lvbnMgdXNlZCB0byBwYXJzZSBhbGdvcml0aG0gbG9ncy5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTWV0cmljRGVmaW5pdGlvbiB7XG5cbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIG1ldHJpYy5cbiAgICovXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogUmVndWxhciBleHByZXNzaW9uIHRoYXQgc2VhcmNoZXMgdGhlIG91dHB1dCBvZiBhIHRyYWluaW5nIGpvYiBhbmQgZ2V0cyB0aGUgdmFsdWUgb2YgdGhlIG1ldHJpYy5cbiAgICovXG4gIHJlYWRvbmx5IHJlZ2V4OiBzdHJpbmc7XG59XG5cbi8qKlxuICogU3RvcmVzIGluZm9ybWF0aW9uIGFib3V0IHRoZSBsb2NhdGlvbiBvZiBhbiBvYmplY3QgaW4gQW1hem9uIFMzXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFMzTG9jYXRpb25Db25maWcge1xuXG4gIC8qKlxuICAgKiBVbmlxdWVseSBpZGVudGlmaWVzIHRoZSByZXNvdXJjZSBpbiBBbWF6b24gUzNcbiAgICovXG4gIHJlYWRvbmx5IHVyaTogc3RyaW5nO1xufVxuXG4vKipcbiAqIENvbnN0cnVjdHMgYElTM0xvY2F0aW9uYCBvYmplY3RzLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFMzTG9jYXRpb24ge1xuICAvKipcbiAgICogQW4gYElTM0xvY2F0aW9uYCBidWlsdCB3aXRoIGEgZGV0ZXJtaW5lZCBidWNrZXQgYW5kIGtleSBwcmVmaXguXG4gICAqXG4gICAqIEBwYXJhbSBidWNrZXQgICAgaXMgdGhlIGJ1Y2tldCB3aGVyZSB0aGUgb2JqZWN0cyBhcmUgdG8gYmUgc3RvcmVkLlxuICAgKiBAcGFyYW0ga2V5UHJlZml4IGlzIHRoZSBrZXkgcHJlZml4IHVzZWQgYnkgdGhlIGxvY2F0aW9uLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQnVja2V0KGJ1Y2tldDogczMuSUJ1Y2tldCwga2V5UHJlZml4OiBzdHJpbmcpOiBTM0xvY2F0aW9uIHtcbiAgICByZXR1cm4gbmV3IFN0YW5kYXJkUzNMb2NhdGlvbih7IGJ1Y2tldCwga2V5UHJlZml4LCB1cmk6IGJ1Y2tldC51cmxGb3JPYmplY3Qoa2V5UHJlZml4KSB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBbiBgSVMzTG9jYXRpb25gIGRldGVybWluZWQgZnVsbHkgYnkgYSBKU09OIFBhdGggZnJvbSB0aGUgdGFzayBpbnB1dC5cbiAgICpcbiAgICogRHVlIHRvIHRoZSBkeW5hbWljIG5hdHVyZSBvZiB0aG9zZSBsb2NhdGlvbnMsIHRoZSBJQU0gZ3JhbnRzIHRoYXQgd2lsbCBiZSBzZXQgYnkgYGdyYW50UmVhZGAgYW5kIGBncmFudFdyaXRlYFxuICAgKiBhcHBseSB0byB0aGUgYCpgIHJlc291cmNlLlxuICAgKlxuICAgKiBAcGFyYW0gZXhwcmVzc2lvbiB0aGUgSlNPTiBleHByZXNzaW9uIHJlc29sdmluZyB0byBhbiBTMyBsb2NhdGlvbiBVUkkuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Kc29uRXhwcmVzc2lvbihleHByZXNzaW9uOiBzdHJpbmcpOiBTM0xvY2F0aW9uIHtcbiAgICByZXR1cm4gbmV3IFN0YW5kYXJkUzNMb2NhdGlvbih7IHVyaTogc2ZuLkpzb25QYXRoLnN0cmluZ0F0KGV4cHJlc3Npb24pIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENhbGxlZCB3aGVuIHRoZSBTM0xvY2F0aW9uIGlzIGJvdW5kIHRvIGEgU3RlcEZ1bmN0aW9ucyB0YXNrLlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IGJpbmQodGFzazogSVNhZ2VNYWtlclRhc2ssIG9wdHM6IFMzTG9jYXRpb25CaW5kT3B0aW9ucyk6IFMzTG9jYXRpb25Db25maWc7XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYmluZGluZyBhbiBTMyBMb2NhdGlvbi5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUzNMb2NhdGlvbkJpbmRPcHRpb25zIHtcbiAgLyoqXG4gICAqIEFsbG93IHJlYWRpbmcgZnJvbSB0aGUgUzMgTG9jYXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBmb3JSZWFkaW5nPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQWxsb3cgd3JpdGluZyB0byB0aGUgUzMgTG9jYXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBmb3JXcml0aW5nPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciBhIHVzaW5nIERvY2tlciBpbWFnZS5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRG9ja2VySW1hZ2VDb25maWcge1xuICAvKipcbiAgICogVGhlIGZ1bGx5IHF1YWxpZmllZCBVUkkgb2YgdGhlIERvY2tlciBpbWFnZS5cbiAgICovXG4gIHJlYWRvbmx5IGltYWdlVXJpOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQ3JlYXRlcyBgSURvY2tlckltYWdlYCBpbnN0YW5jZXMuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgRG9ja2VySW1hZ2Uge1xuICAvKipcbiAgICogUmVmZXJlbmNlIGEgRG9ja2VyIGltYWdlIHN0b3JlZCBpbiBhbiBFQ1IgcmVwb3NpdG9yeS5cbiAgICpcbiAgICogQHBhcmFtIHJlcG9zaXRvcnkgdGhlIEVDUiByZXBvc2l0b3J5IHdoZXJlIHRoZSBpbWFnZSBpcyBob3N0ZWQuXG4gICAqIEBwYXJhbSB0YWcgYW4gb3B0aW9uYWwgYHRhZ2BcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUVjclJlcG9zaXRvcnkocmVwb3NpdG9yeTogZWNyLklSZXBvc2l0b3J5LCB0YWc6IHN0cmluZyA9ICdsYXRlc3QnKTogRG9ja2VySW1hZ2Uge1xuICAgIHJldHVybiBuZXcgU3RhbmRhcmREb2NrZXJJbWFnZSh7IHJlcG9zaXRvcnksIGltYWdlVXJpOiByZXBvc2l0b3J5LnJlcG9zaXRvcnlVcmlGb3JUYWcodGFnKSB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWZlcmVuY2UgYSBEb2NrZXIgaW1hZ2Ugd2hpY2ggVVJJIGlzIG9idGFpbmVkIGZyb20gdGhlIHRhc2sncyBpbnB1dC5cbiAgICpcbiAgICogQHBhcmFtIGV4cHJlc3Npb24gICAgICAgICAgIHRoZSBKU09OIHBhdGggZXhwcmVzc2lvbiB3aXRoIHRoZSB0YXNrIGlucHV0LlxuICAgKiBAcGFyYW0gYWxsb3dBbnlFY3JJbWFnZVB1bGwgd2hldGhlciBFQ1IgYWNjZXNzIHNob3VsZCBiZSBwZXJtaXR0ZWQgKHNldCB0byBgZmFsc2VgIGlmIHRoZSBpbWFnZSB3aWxsIG5ldmVyIGJlIGluIEVDUikuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Kc29uRXhwcmVzc2lvbihleHByZXNzaW9uOiBzdHJpbmcsIGFsbG93QW55RWNySW1hZ2VQdWxsID0gdHJ1ZSk6IERvY2tlckltYWdlIHtcbiAgICByZXR1cm4gbmV3IFN0YW5kYXJkRG9ja2VySW1hZ2UoeyBpbWFnZVVyaTogZXhwcmVzc2lvbiwgYWxsb3dBbnlFY3JJbWFnZVB1bGwgfSk7XG4gIH1cblxuICAvKipcbiAgICogUmVmZXJlbmNlIGEgRG9ja2VyIGltYWdlIGJ5IGl0J3MgVVJJLlxuICAgKlxuICAgKiBXaGVuIHJlZmVyZW5jaW5nIEVDUiBpbWFnZXMsIHByZWZlciB1c2luZyBgaW5FY3JgLlxuICAgKlxuICAgKiBAcGFyYW0gaW1hZ2VVcmkgdGhlIFVSSSB0byB0aGUgZG9ja2VyIGltYWdlLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tUmVnaXN0cnkoaW1hZ2VVcmk6IHN0cmluZyk6IERvY2tlckltYWdlIHtcbiAgICByZXR1cm4gbmV3IFN0YW5kYXJkRG9ja2VySW1hZ2UoeyBpbWFnZVVyaSB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWZlcmVuY2UgYSBEb2NrZXIgaW1hZ2UgdGhhdCBpcyBwcm92aWRlZCBhcyBhbiBBc3NldCBpbiB0aGUgY3VycmVudCBhcHAuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSB0aGUgc2NvcGUgaW4gd2hpY2ggdG8gY3JlYXRlIHRoZSBBc3NldC5cbiAgICogQHBhcmFtIGlkICAgIHRoZSBJRCBmb3IgdGhlIGFzc2V0IGluIHRoZSBjb25zdHJ1Y3QgdHJlZS5cbiAgICogQHBhcmFtIHByb3BzIHRoZSBjb25maWd1cmF0aW9uIHByb3BzIG9mIHRoZSBhc3NldC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUFzc2V0KHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBEb2NrZXJJbWFnZUFzc2V0UHJvcHMpOiBEb2NrZXJJbWFnZSB7XG4gICAgY29uc3QgYXNzZXQgPSBuZXcgRG9ja2VySW1hZ2VBc3NldChzY29wZSwgaWQsIHByb3BzKTtcbiAgICByZXR1cm4gbmV3IFN0YW5kYXJkRG9ja2VySW1hZ2UoeyByZXBvc2l0b3J5OiBhc3NldC5yZXBvc2l0b3J5LCBpbWFnZVVyaTogYXNzZXQuaW1hZ2VVcmkgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2FsbGVkIHdoZW4gdGhlIGltYWdlIGlzIHVzZWQgYnkgYSBTYWdlTWFrZXIgdGFzay5cbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCBiaW5kKHRhc2s6IElTYWdlTWFrZXJUYXNrKTogRG9ja2VySW1hZ2VDb25maWc7XG59XG5cbi8qKlxuICogUzMgRGF0YSBUeXBlLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGVudW0gUzNEYXRhVHlwZSB7XG4gIC8qKlxuICAgKiBNYW5pZmVzdCBGaWxlIERhdGEgVHlwZVxuICAgKi9cbiAgTUFOSUZFU1RfRklMRSA9ICdNYW5pZmVzdEZpbGUnLFxuXG4gIC8qKlxuICAgKiBTMyBQcmVmaXggRGF0YSBUeXBlXG4gICAqL1xuICBTM19QUkVGSVggPSAnUzNQcmVmaXgnLFxuXG4gIC8qKlxuICAgKiBBdWdtZW50ZWQgTWFuaWZlc3QgRmlsZSBEYXRhIFR5cGVcbiAgICovXG4gIEFVR01FTlRFRF9NQU5JRkVTVF9GSUxFID0gJ0F1Z21lbnRlZE1hbmlmZXN0RmlsZSdcbn1cblxuLyoqXG4gKiBTMyBEYXRhIERpc3RyaWJ1dGlvbiBUeXBlLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGVudW0gUzNEYXRhRGlzdHJpYnV0aW9uVHlwZSB7XG4gIC8qKlxuICAgKiBGdWxseSByZXBsaWNhdGVkIFMzIERhdGEgRGlzdHJpYnV0aW9uIFR5cGVcbiAgICovXG4gIEZVTExZX1JFUExJQ0FURUQgPSAnRnVsbHlSZXBsaWNhdGVkJyxcblxuICAvKipcbiAgICogU2hhcmRlZCBCeSBTMyBLZXkgRGF0YSBEaXN0cmlidXRpb24gVHlwZVxuICAgKi9cbiAgU0hBUkRFRF9CWV9TM19LRVkgPSAnU2hhcmRlZEJ5UzNLZXknXG59XG5cbi8qKlxuICogRGVmaW5lIHRoZSBmb3JtYXQgb2YgdGhlIGlucHV0IGRhdGEuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgZW51bSBSZWNvcmRXcmFwcGVyVHlwZSB7XG4gIC8qKlxuICAgKiBOb25lIHJlY29yZCB3cmFwcGVyIHR5cGVcbiAgICovXG4gIE5PTkUgPSAnTm9uZScsXG5cbiAgLyoqXG4gICAqIFJlY29yZElPIHJlY29yZCB3cmFwcGVyIHR5cGVcbiAgICovXG4gIFJFQ09SRF9JTyA9ICdSZWNvcmRJTydcbn1cblxuLyoqXG4gKiAgSW5wdXQgbW9kZSB0aGF0IHRoZSBhbGdvcml0aG0gc3VwcG9ydHMuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgZW51bSBJbnB1dE1vZGUge1xuICAvKipcbiAgICogUGlwZSBtb2RlXG4gICAqL1xuICBQSVBFID0gJ1BpcGUnLFxuXG4gIC8qKlxuICAgKiBGaWxlIG1vZGUuXG4gICAqL1xuICBGSUxFID0gJ0ZpbGUnXG59XG5cbi8qKlxuICogQ29tcHJlc3Npb24gdHlwZSBvZiB0aGUgZGF0YS5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBlbnVtIENvbXByZXNzaW9uVHlwZSB7XG4gIC8qKlxuICAgKiBOb25lIGNvbXByZXNzaW9uIHR5cGVcbiAgICovXG4gIE5PTkUgPSAnTm9uZScsXG5cbiAgLyoqXG4gICAqIEd6aXAgY29tcHJlc3Npb24gdHlwZVxuICAgKi9cbiAgR1pJUCA9ICdHemlwJ1xufVxuXG4vL1xuLy8gQ3JlYXRlIFRyYW5zZm9ybSBKb2IgdHlwZXNcbi8vXG5cbi8qKlxuICogIERhdGFzZXQgdG8gYmUgdHJhbnNmb3JtZWQgYW5kIHRoZSBBbWF6b24gUzMgbG9jYXRpb24gd2hlcmUgaXQgaXMgc3RvcmVkLlxuICpcbiAqICBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVHJhbnNmb3JtSW5wdXQge1xuXG4gIC8qKlxuICAgKiBUaGUgY29tcHJlc3Npb24gdHlwZSBvZiB0aGUgdHJhbnNmb3JtIGRhdGEuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5PTkVcbiAgICovXG4gIHJlYWRvbmx5IGNvbXByZXNzaW9uVHlwZT86IENvbXByZXNzaW9uVHlwZTtcblxuICAvKipcbiAgICogTXVsdGlwdXJwb3NlIGludGVybmV0IG1haWwgZXh0ZW5zaW9uIChNSU1FKSB0eXBlIG9mIHRoZSBkYXRhLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGNvbnRlbnRUeXBlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBTMyBsb2NhdGlvbiBvZiB0aGUgY2hhbm5lbCBkYXRhXG4gICAqL1xuICByZWFkb25seSB0cmFuc2Zvcm1EYXRhU291cmNlOiBUcmFuc2Zvcm1EYXRhU291cmNlO1xuXG4gIC8qKlxuICAgKiBNZXRob2QgdG8gdXNlIHRvIHNwbGl0IHRoZSB0cmFuc2Zvcm0gam9iJ3MgZGF0YSBmaWxlcyBpbnRvIHNtYWxsZXIgYmF0Y2hlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgTk9ORVxuICAgKi9cbiAgcmVhZG9ubHkgc3BsaXRUeXBlPzogU3BsaXRUeXBlO1xufVxuXG4vKipcbiAqIFMzIGxvY2F0aW9uIG9mIHRoZSBpbnB1dCBkYXRhIHRoYXQgdGhlIG1vZGVsIGNhbiBjb25zdW1lLlxuICpcbiAqICBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVHJhbnNmb3JtRGF0YVNvdXJjZSB7XG5cbiAgLyoqXG4gICAqIFMzIGxvY2F0aW9uIG9mIHRoZSBpbnB1dCBkYXRhXG4gICAqL1xuICByZWFkb25seSBzM0RhdGFTb3VyY2U6IFRyYW5zZm9ybVMzRGF0YVNvdXJjZTtcbn1cblxuLyoqXG4gKiBMb2NhdGlvbiBvZiB0aGUgY2hhbm5lbCBkYXRhLlxuICpcbiAqICBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVHJhbnNmb3JtUzNEYXRhU291cmNlIHtcblxuICAvKipcbiAgICogUzMgRGF0YSBUeXBlXG4gICAqXG4gICAqIEBkZWZhdWx0ICdTM1ByZWZpeCdcbiAgICovXG4gIHJlYWRvbmx5IHMzRGF0YVR5cGU/OiBTM0RhdGFUeXBlO1xuXG4gIC8qKlxuICAgKiBJZGVudGlmaWVzIGVpdGhlciBhIGtleSBuYW1lIHByZWZpeCBvciBhIG1hbmlmZXN0LlxuICAgKi9cbiAgcmVhZG9ubHkgczNVcmk6IHN0cmluZztcbn1cblxuLyoqXG4gKiBTMyBsb2NhdGlvbiB3aGVyZSB5b3Ugd2FudCBBbWF6b24gU2FnZU1ha2VyIHRvIHNhdmUgdGhlIHJlc3VsdHMgZnJvbSB0aGUgdHJhbnNmb3JtIGpvYi5cbiAqXG4gKiAgQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRyYW5zZm9ybU91dHB1dCB7XG5cbiAgLyoqXG4gICAqIE1JTUUgdHlwZSB1c2VkIHRvIHNwZWNpZnkgdGhlIG91dHB1dCBkYXRhLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGFjY2VwdD86IHN0cmluZztcblxuICAvKipcbiAgICogRGVmaW5lcyBob3cgdG8gYXNzZW1ibGUgdGhlIHJlc3VsdHMgb2YgdGhlIHRyYW5zZm9ybSBqb2IgYXMgYSBzaW5nbGUgUzMgb2JqZWN0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGFzc2VtYmxlV2l0aD86IEFzc2VtYmxlV2l0aDtcblxuICAvKipcbiAgICogQVdTIEtNUyBrZXkgdGhhdCBBbWF6b24gU2FnZU1ha2VyIHVzZXMgdG8gZW5jcnlwdCB0aGUgbW9kZWwgYXJ0aWZhY3RzIGF0IHJlc3QgdXNpbmcgQW1hem9uIFMzIHNlcnZlci1zaWRlIGVuY3J5cHRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBLTVMga2V5IGZvciBBbWF6b24gUzMgZm9yIHlvdXIgcm9sZSdzIGFjY291bnQuXG4gICAqL1xuICByZWFkb25seSBlbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG5cbiAgLyoqXG4gICAqIFMzIHBhdGggd2hlcmUgeW91IHdhbnQgQW1hem9uIFNhZ2VNYWtlciB0byBzdG9yZSB0aGUgcmVzdWx0cyBvZiB0aGUgdHJhbnNmb3JtIGpvYi5cbiAgICovXG4gIHJlYWRvbmx5IHMzT3V0cHV0UGF0aDogc3RyaW5nO1xufVxuXG4vKipcbiAqIE1MIGNvbXB1dGUgaW5zdGFuY2VzIGZvciB0aGUgdHJhbnNmb3JtIGpvYi5cbiAqXG4gKiAgQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRyYW5zZm9ybVJlc291cmNlcyB7XG5cbiAgLyoqXG4gICAqIE51bWJlciBvZiBNTCBjb21wdXRlIGluc3RhbmNlcyB0byB1c2UgaW4gdGhlIHRyYW5zZm9ybSBqb2JcbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlQ291bnQ6IG51bWJlcjtcblxuICAvKipcbiAgICogTUwgY29tcHV0ZSBpbnN0YW5jZSB0eXBlIGZvciB0aGUgdHJhbnNmb3JtIGpvYi5cbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZTogZWMyLkluc3RhbmNlVHlwZTtcblxuICAvKipcbiAgICogQVdTIEtNUyBrZXkgdGhhdCBBbWF6b24gU2FnZU1ha2VyIHVzZXMgdG8gZW5jcnlwdCBkYXRhIG9uIHRoZSBzdG9yYWdlIHZvbHVtZSBhdHRhY2hlZCB0byB0aGUgTUwgY29tcHV0ZSBpbnN0YW5jZShzKS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSB2b2x1bWVFbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyB0byBkZWZpbmUgYSBDb250YWluZXJEZWZpbml0aW9uXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2FnZW1ha2VyL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NvbnRhaW5lckRlZmluaXRpb24uaHRtbFxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbnRhaW5lckRlZmluaXRpb25PcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gRUMyIENvbnRhaW5lciBSZWdpc3RyeSAoQW1hem9uIEVDUikgcGF0aCB3aGVyZSBpbmZlcmVuY2UgY29kZSBpcyBzdG9yZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgaW1hZ2U/OiBEb2NrZXJJbWFnZTtcbiAgLyoqXG4gICAqIFRoZSBlbnZpcm9ubWVudCB2YXJpYWJsZXMgdG8gc2V0IGluIHRoZSBEb2NrZXIgY29udGFpbmVyXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gdmFyaWFibGVzXG4gICAqL1xuICByZWFkb25seSBlbnZpcm9ubWVudFZhcmlhYmxlcz86IHNmbi5UYXNrSW5wdXQ7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvciBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBvZiB0aGUgbW9kZWwgcGFja2FnZSB0byB1c2UgdG8gY3JlYXRlIHRoZSBtb2RlbC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBtb2RlbFBhY2thZ2VOYW1lPzogc3RyaW5nO1xuICAvKipcbiAgICogRGVmaW5lcyBob3cgbWFueSBtb2RlbHMgdGhlIGNvbnRhaW5lciBob3N0c1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIE1vZGUuU0lOR0xFX01PREVMXG4gICAqL1xuICByZWFkb25seSBtb2RlPzogTW9kZTtcbiAgLyoqXG4gICAqIFRoaXMgcGFyYW1ldGVyIGlzIGlnbm9yZWQgZm9yIG1vZGVscyB0aGF0IGNvbnRhaW4gb25seSBhIFByaW1hcnlDb250YWluZXIuXG4gICAqIFdoZW4gYSBDb250YWluZXJEZWZpbml0aW9uIGlzIHBhcnQgb2YgYW4gaW5mZXJlbmNlIHBpcGVsaW5lLFxuICAgKiB0aGUgdmFsdWUgb2YgdGhlIHBhcmFtZXRlciB1bmlxdWVseSBpZGVudGlmaWVzIHRoZSBjb250YWluZXIgZm9yIHRoZSBwdXJwb3NlcyBvZiBsb2dnaW5nIGFuZCBtZXRyaWNzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGNvbnRhaW5lckhvc3ROYW1lPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIFMzIHBhdGggd2hlcmUgdGhlIG1vZGVsIGFydGlmYWN0cywgd2hpY2ggcmVzdWx0IGZyb20gbW9kZWwgdHJhaW5pbmcsIGFyZSBzdG9yZWQuXG4gICAqIFRoaXMgcGF0aCBtdXN0IHBvaW50IHRvIGEgc2luZ2xlIGd6aXAgY29tcHJlc3NlZCB0YXIgYXJjaGl2ZSAoLnRhci5neiBzdWZmaXgpLlxuICAgKiBUaGUgUzMgcGF0aCBpcyByZXF1aXJlZCBmb3IgQW1hem9uIFNhZ2VNYWtlciBidWlsdC1pbiBhbGdvcml0aG1zLCBidXQgbm90IGlmIHlvdSB1c2UgeW91ciBvd24gYWxnb3JpdGhtcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBtb2RlbFMzTG9jYXRpb24/OiBTM0xvY2F0aW9uO1xufVxuXG4vKipcbiAqIERlc2NyaWJlcyB0aGUgY29udGFpbmVyLCBhcyBwYXJ0IG9mIG1vZGVsIGRlZmluaXRpb24uXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2FnZW1ha2VyL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NvbnRhaW5lckRlZmluaXRpb24uaHRtbFxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgQ29udGFpbmVyRGVmaW5pdGlvbiBpbXBsZW1lbnRzIElDb250YWluZXJEZWZpbml0aW9uIHtcblxuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IG9wdGlvbnM6IENvbnRhaW5lckRlZmluaXRpb25PcHRpb25zKSB7fVxuXG4gIC8qKlxuICAgKiBDYWxsZWQgd2hlbiB0aGUgQ29udGFpbmVyRGVmaW5pdGlvbiB0eXBlIGNvbmZpZ3VyZWQgb24gU2FnZW1ha2VyIFRhc2tcbiAgICovXG4gIHB1YmxpYyBiaW5kKHRhc2s6IElTYWdlTWFrZXJUYXNrKTogQ29udGFpbmVyRGVmaW5pdGlvbkNvbmZpZyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgQ29udGFpbmVySG9zdG5hbWU6IHRoaXMub3B0aW9ucy5jb250YWluZXJIb3N0TmFtZSxcbiAgICAgICAgSW1hZ2U6IHRoaXMub3B0aW9ucy5pbWFnZT8uYmluZCh0YXNrKS5pbWFnZVVyaSxcbiAgICAgICAgTW9kZTogdGhpcy5vcHRpb25zLm1vZGUsXG4gICAgICAgIE1vZGVsRGF0YVVybDogdGhpcy5vcHRpb25zLm1vZGVsUzNMb2NhdGlvbj8uYmluZCh0YXNrLCB7IGZvclJlYWRpbmc6IHRydWUgfSkudXJpLFxuICAgICAgICBNb2RlbFBhY2thZ2VOYW1lOiB0aGlzLm9wdGlvbnMubW9kZWxQYWNrYWdlTmFtZSxcbiAgICAgICAgRW52aXJvbm1lbnQ6IHRoaXMub3B0aW9ucy5lbnZpcm9ubWVudFZhcmlhYmxlcz8udmFsdWUsXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIG9mIHRoZSBjb250YWluZXIgdXNlZCB0byBob3N0IHRoZSBtb2RlbFxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NhZ2VtYWtlci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Db250YWluZXJEZWZpbml0aW9uLmh0bWxcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQ29udGFpbmVyRGVmaW5pdGlvbiB7XG4gIC8qKlxuICAgKiBDYWxsZWQgd2hlbiB0aGUgQ29udGFpbmVyRGVmaW5pdGlvbiBpcyB1c2VkIGJ5IGEgU2FnZU1ha2VyIHRhc2suXG4gICAqL1xuICBiaW5kKHRhc2s6IElTYWdlTWFrZXJUYXNrKTogQ29udGFpbmVyRGVmaW5pdGlvbkNvbmZpZztcbn1cblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIG9wdGlvbnMgZm9yIHRoZSBDb250YWluZXJEZWZpbml0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29udGFpbmVyRGVmaW5pdGlvbkNvbmZpZyB7XG4gIC8qKlxuICAgKiBBZGRpdGlvbmFsIHBhcmFtZXRlcnMgdG8gcGFzcyB0byB0aGUgYmFzZSB0YXNrXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYWRkaXRpb25hbCBwYXJhbWV0ZXJzIHBhc3NlZFxuICAgKi9cbiAgcmVhZG9ubHkgcGFyYW1ldGVycz86IHsgW2tleTogc3RyaW5nXTogYW55IH07XG59XG5cbi8qKlxuICogU3BlY2lmaWVzIGhvdyBtYW55IG1vZGVscyB0aGUgY29udGFpbmVyIGhvc3RzXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgZW51bSBNb2RlIHtcbiAgLyoqXG4gICAqIENvbnRhaW5lciBob3N0cyBhIHNpbmdsZSBtb2RlbFxuICAgKi9cbiAgU0lOR0xFX01PREVMID0gJ1NpbmdsZU1vZGVsJyxcbiAgLyoqXG4gICAqIENvbnRhaW5lciBob3N0cyBtdWx0aXBsZSBtb2RlbHNcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2FnZW1ha2VyL2xhdGVzdC9kZy9tdWx0aS1tb2RlbC1lbmRwb2ludHMuaHRtbFxuICAgKi9cbiAgTVVMVElfTU9ERUwgPSAnTXVsdGlNb2RlbCcsXG59XG5cbi8qKlxuICogSWRlbnRpZmllcyBhIG1vZGVsIHRoYXQgeW91IHdhbnQgdG8gaG9zdCBhbmQgdGhlIHJlc291cmNlcyB0byBkZXBsb3kgZm9yIGhvc3RpbmcgaXQuXG4gKlxuICogQHNlZSAgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NhZ2VtYWtlci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Qcm9kdWN0aW9uVmFyaWFudC5odG1sXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUHJvZHVjdGlvblZhcmlhbnQge1xuICAvKipcbiAgICogVGhlIHNpemUgb2YgdGhlIEVsYXN0aWMgSW5mZXJlbmNlIChFSSkgaW5zdGFuY2UgdG8gdXNlIGZvciB0aGUgcHJvZHVjdGlvbiB2YXJpYW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGFjY2VsZXJhdG9yVHlwZT86IEFjY2VsZXJhdG9yVHlwZTtcbiAgLyoqXG4gICAqIE51bWJlciBvZiBpbnN0YW5jZXMgdG8gbGF1bmNoIGluaXRpYWxseS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSAxXG4gICAqL1xuICByZWFkb25seSBpbml0aWFsSW5zdGFuY2VDb3VudD86IG51bWJlcjtcbiAgLyoqXG4gICAqIERldGVybWluZXMgaW5pdGlhbCB0cmFmZmljIGRpc3RyaWJ1dGlvbiBhbW9uZyBhbGwgb2YgdGhlIG1vZGVscyB0aGF0IHlvdSBzcGVjaWZ5IGluIHRoZSBlbmRwb2ludCBjb25maWd1cmF0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDEuMFxuICAgKi9cbiAgcmVhZG9ubHkgaW5pdGlhbFZhcmlhbnRXZWlnaHQ/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBUaGUgTUwgY29tcHV0ZSBpbnN0YW5jZSB0eXBlXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGU7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgcHJvZHVjdGlvbiB2YXJpYW50LlxuICAgKi9cbiAgcmVhZG9ubHkgdmFyaWFudE5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBtb2RlbCB0aGF0IHlvdSB3YW50IHRvIGhvc3QuIFRoaXMgaXMgdGhlIG5hbWUgdGhhdCB5b3Ugc3BlY2lmaWVkIHdoZW4gY3JlYXRpbmcgdGhlIG1vZGVsLlxuICAgKi9cbiAgcmVhZG9ubHkgbW9kZWxOYW1lOiBzdHJpbmc7XG59XG5cbi8qKlxuICogVGhlIGdlbmVyYXRpb24gb2YgRWxhc3RpYyBJbmZlcmVuY2UgKEVJKSBpbnN0YW5jZVxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NhZ2VtYWtlci9sYXRlc3QvZGcvZWkuaHRtbFxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgQWNjZWxlcmF0b3JDbGFzcyB7XG4gIC8qKlxuICAgKiBFbGFzdGljIEluZmVyZW5jZSBhY2NlbGVyYXRvciAxc3QgZ2VuZXJhdGlvblxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBFSUExID0gQWNjZWxlcmF0b3JDbGFzcy5vZignZWlhMScpO1xuICAvKipcbiAgICogRWxhc3RpYyBJbmZlcmVuY2UgYWNjZWxlcmF0b3IgMm5kIGdlbmVyYXRpb25cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgRUlBMiA9IEFjY2VsZXJhdG9yQ2xhc3Mub2YoJ2VpYTInKTtcbiAgLyoqXG4gICAqIEN1c3RvbSBBY2NlbGVyYXRvclR5cGVcbiAgICogQHBhcmFtIHZlcnNpb24gLSBFbGFzdGljIEluZmVyZW5jZSBhY2NlbGVyYXRvciBnZW5lcmF0aW9uXG4gICovXG4gIHB1YmxpYyBzdGF0aWMgb2YodmVyc2lvbjogc3RyaW5nKSB7IHJldHVybiBuZXcgQWNjZWxlcmF0b3JDbGFzcyh2ZXJzaW9uKTsgfVxuICAvKipcbiAgICogQHBhcmFtIHZlcnNpb24gLSBFbGFzdGljIEluZmVyZW5jZSBhY2NlbGVyYXRvciBnZW5lcmF0aW9uXG4gICAqL1xuICBwcml2YXRlIGNvbnN0cnVjdG9yKHB1YmxpYyByZWFkb25seSB2ZXJzaW9uOiBzdHJpbmcpIHsgfVxufVxuXG4vKipcbiAqIFRoZSBzaXplIG9mIHRoZSBFbGFzdGljIEluZmVyZW5jZSAoRUkpIGluc3RhbmNlIHRvIHVzZSBmb3IgdGhlIHByb2R1Y3Rpb24gdmFyaWFudC5cbiAqIEVJIGluc3RhbmNlcyBwcm92aWRlIG9uLWRlbWFuZCBHUFUgY29tcHV0aW5nIGZvciBpbmZlcmVuY2VcbiAqXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zYWdlbWFrZXIvbGF0ZXN0L2RnL2VpLmh0bWxcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGNsYXNzIEFjY2VsZXJhdG9yVHlwZSB7XG4gIC8qKlxuICAgKiBBY2NlbGVyYXRvclR5cGVcbiAgICpcbiAgICogVGhpcyBjbGFzcyB0YWtlcyBhIGNvbWJpbmF0aW9uIG9mIGEgY2xhc3MgYW5kIHNpemUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIG9mKGFjY2VsZXJhdG9yQ2xhc3M6IEFjY2VsZXJhdG9yQ2xhc3MsIGluc3RhbmNlU2l6ZTogZWMyLkluc3RhbmNlU2l6ZSkge1xuICAgIHJldHVybiBuZXcgQWNjZWxlcmF0b3JUeXBlKGBtbC4ke2FjY2VsZXJhdG9yQ2xhc3N9LiR7aW5zdGFuY2VTaXplfWApO1xuICB9XG5cbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBpbnN0YW5jZVR5cGVJZGVudGlmaWVyOiBzdHJpbmcpIHtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIGFjY2VsZXJhdG9yIHR5cGUgYXMgYSBkb3R0ZWQgc3RyaW5nXG4gICAqL1xuICBwdWJsaWMgdG9TdHJpbmcoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5pbnN0YW5jZVR5cGVJZGVudGlmaWVyO1xuICB9XG59XG5cbi8qKlxuICogU3BlY2lmaWVzIHRoZSBudW1iZXIgb2YgcmVjb3JkcyB0byBpbmNsdWRlIGluIGEgbWluaS1iYXRjaCBmb3IgYW4gSFRUUCBpbmZlcmVuY2UgcmVxdWVzdC5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBlbnVtIEJhdGNoU3RyYXRlZ3kge1xuXG4gIC8qKlxuICAgKiBGaXRzIG11bHRpcGxlIHJlY29yZHMgaW4gYSBtaW5pLWJhdGNoLlxuICAgKi9cbiAgTVVMVElfUkVDT1JEID0gJ011bHRpUmVjb3JkJyxcblxuICAvKipcbiAgICogVXNlIGEgc2luZ2xlIHJlY29yZCB3aGVuIG1ha2luZyBhbiBpbnZvY2F0aW9uIHJlcXVlc3QuXG4gICAqL1xuICBTSU5HTEVfUkVDT1JEID0gJ1NpbmdsZVJlY29yZCdcbn1cblxuLyoqXG4gKiBNZXRob2QgdG8gdXNlIHRvIHNwbGl0IHRoZSB0cmFuc2Zvcm0gam9iJ3MgZGF0YSBmaWxlcyBpbnRvIHNtYWxsZXIgYmF0Y2hlcy5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBlbnVtIFNwbGl0VHlwZSB7XG5cbiAgLyoqXG4gICAqIElucHV0IGRhdGEgZmlsZXMgYXJlIG5vdCBzcGxpdCxcbiAgICovXG4gIE5PTkUgPSAnTm9uZScsXG5cbiAgLyoqXG4gICAqIFNwbGl0IHJlY29yZHMgb24gYSBuZXdsaW5lIGNoYXJhY3RlciBib3VuZGFyeS5cbiAgICovXG4gIExJTkUgPSAnTGluZScsXG5cbiAgLyoqXG4gICAqIFNwbGl0IHVzaW5nIE1YTmV0IFJlY29yZElPIGZvcm1hdC5cbiAgICovXG4gIFJFQ09SRF9JTyA9ICdSZWNvcmRJTycsXG5cbiAgLyoqXG4gICAqIFNwbGl0IHVzaW5nIFRlbnNvckZsb3cgVEZSZWNvcmQgZm9ybWF0LlxuICAgKi9cbiAgVEZfUkVDT1JEID0gJ1RGUmVjb3JkJ1xufVxuXG4vKipcbiAqIEhvdyB0byBhc3NlbWJsZSB0aGUgcmVzdWx0cyBvZiB0aGUgdHJhbnNmb3JtIGpvYiBhcyBhIHNpbmdsZSBTMyBvYmplY3QuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgZW51bSBBc3NlbWJsZVdpdGgge1xuXG4gIC8qKlxuICAgKiBDb25jYXRlbmF0ZSB0aGUgcmVzdWx0cyBpbiBiaW5hcnkgZm9ybWF0LlxuICAgKi9cbiAgTk9ORSA9ICdOb25lJyxcblxuICAvKipcbiAgICogQWRkIGEgbmV3bGluZSBjaGFyYWN0ZXIgYXQgdGhlIGVuZCBvZiBldmVyeSB0cmFuc2Zvcm1lZCByZWNvcmQuXG4gICAqL1xuICBMSU5FID0gJ0xpbmUnXG5cbn1cblxuY2xhc3MgU3RhbmRhcmREb2NrZXJJbWFnZSBleHRlbmRzIERvY2tlckltYWdlIHtcbiAgcHJpdmF0ZSByZWFkb25seSBhbGxvd0FueUVjckltYWdlUHVsbDogYm9vbGVhbjtcbiAgcHJpdmF0ZSByZWFkb25seSBpbWFnZVVyaTogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlcG9zaXRvcnk/OiBlY3IuSVJlcG9zaXRvcnk7XG5cbiAgY29uc3RydWN0b3Iob3B0czogeyBhbGxvd0FueUVjckltYWdlUHVsbD86IGJvb2xlYW4sIGltYWdlVXJpOiBzdHJpbmcsIHJlcG9zaXRvcnk/OiBlY3IuSVJlcG9zaXRvcnkgfSkge1xuICAgIHN1cGVyKCk7XG5cbiAgICB0aGlzLmFsbG93QW55RWNySW1hZ2VQdWxsID0gISFvcHRzLmFsbG93QW55RWNySW1hZ2VQdWxsO1xuICAgIHRoaXMuaW1hZ2VVcmkgPSBvcHRzLmltYWdlVXJpO1xuICAgIHRoaXMucmVwb3NpdG9yeSA9IG9wdHMucmVwb3NpdG9yeTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kKHRhc2s6IElTYWdlTWFrZXJUYXNrKTogRG9ja2VySW1hZ2VDb25maWcge1xuICAgIGlmICh0aGlzLnJlcG9zaXRvcnkpIHtcbiAgICAgIHRoaXMucmVwb3NpdG9yeS5ncmFudFB1bGwodGFzayk7XG4gICAgfVxuICAgIGlmICh0aGlzLmFsbG93QW55RWNySW1hZ2VQdWxsKSB7XG4gICAgICB0YXNrLmdyYW50UHJpbmNpcGFsLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdlY3I6QmF0Y2hDaGVja0xheWVyQXZhaWxhYmlsaXR5JyxcbiAgICAgICAgICAnZWNyOkdldERvd25sb2FkVXJsRm9yTGF5ZXInLFxuICAgICAgICAgICdlY3I6QmF0Y2hHZXRJbWFnZScsXG4gICAgICAgIF0sXG4gICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICB9KSk7XG4gICAgfVxuICAgIHJldHVybiB7XG4gICAgICBpbWFnZVVyaTogdGhpcy5pbWFnZVVyaSxcbiAgICB9O1xuICB9XG59XG5cbmNsYXNzIFN0YW5kYXJkUzNMb2NhdGlvbiBleHRlbmRzIFMzTG9jYXRpb24ge1xuICBwcml2YXRlIHJlYWRvbmx5IGJ1Y2tldD86IHMzLklCdWNrZXQ7XG4gIHByaXZhdGUgcmVhZG9ubHkga2V5R2xvYjogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHVyaTogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKG9wdHM6IHsgYnVja2V0PzogczMuSUJ1Y2tldCwga2V5UHJlZml4Pzogc3RyaW5nLCB1cmk6IHN0cmluZyB9KSB7XG4gICAgc3VwZXIoKTtcbiAgICB0aGlzLmJ1Y2tldCA9IG9wdHMuYnVja2V0O1xuICAgIHRoaXMua2V5R2xvYiA9IGAke29wdHMua2V5UHJlZml4IHx8ICcnfSpgO1xuICAgIHRoaXMudXJpID0gb3B0cy51cmk7XG4gIH1cblxuICBwdWJsaWMgYmluZCh0YXNrOiBJU2FnZU1ha2VyVGFzaywgb3B0czogUzNMb2NhdGlvbkJpbmRPcHRpb25zKTogUzNMb2NhdGlvbkNvbmZpZyB7XG4gICAgaWYgKHRoaXMuYnVja2V0KSB7XG4gICAgICBpZiAob3B0cy5mb3JSZWFkaW5nKSB7XG4gICAgICAgIHRoaXMuYnVja2V0LmdyYW50UmVhZCh0YXNrLCB0aGlzLmtleUdsb2IpO1xuICAgICAgfVxuICAgICAgaWYgKG9wdHMuZm9yV3JpdGluZykge1xuICAgICAgICB0aGlzLmJ1Y2tldC5ncmFudFdyaXRlKHRhc2ssIHRoaXMua2V5R2xvYik7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IGFjdGlvbnMgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuICAgICAgaWYgKG9wdHMuZm9yUmVhZGluZykge1xuICAgICAgICBhY3Rpb25zLnB1c2goJ3MzOkdldE9iamVjdCcsICdzMzpMaXN0QnVja2V0Jyk7XG4gICAgICB9XG4gICAgICBpZiAob3B0cy5mb3JXcml0aW5nKSB7XG4gICAgICAgIGFjdGlvbnMucHVzaCgnczM6UHV0T2JqZWN0Jyk7XG4gICAgICB9XG4gICAgICB0YXNrLmdyYW50UHJpbmNpcGFsLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHsgYWN0aW9ucywgcmVzb3VyY2VzOiBbJyonXSB9KSk7XG4gICAgfVxuICAgIHJldHVybiB7IHVyaTogdGhpcy51cmkgfTtcbiAgfVxufVxuIl19