"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AzureDiskPersistentVolumeCachingMode = exports.AzureDiskPersistentVolumeKind = exports.EmptyDirMedium = exports.Volume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const k8s = require("./imports/k8s");
/**
 * Volume represents a named volume in a pod that may be accessed by any
 * container in the pod.
 *
 * Docker also has a concept of volumes, though it is somewhat looser and less
 * managed. In Docker, a volume is simply a directory on disk or in another
 * Container. Lifetimes are not managed and until very recently there were only
 * local-disk-backed volumes. Docker now provides volume drivers, but the
 * functionality is very limited for now (e.g. as of Docker 1.7 only one volume
 * driver is allowed per Container and there is no way to pass parameters to
 * volumes).
 *
 * A Kubernetes volume, on the other hand, has an explicit lifetime - the same
 * as the Pod that encloses it. Consequently, a volume outlives any Containers
 * that run within the Pod, and data is preserved across Container restarts. Of
 * course, when a Pod ceases to exist, the volume will cease to exist, too.
 * Perhaps more importantly than this, Kubernetes supports many types of
 * volumes, and a Pod can use any number of them simultaneously.
 *
 * At its core, a volume is just a directory, possibly with some data in it,
 * which is accessible to the Containers in a Pod. How that directory comes to
 * be, the medium that backs it, and the contents of it are determined by the
 * particular volume type used.
 *
 * To use a volume, a Pod specifies what volumes to provide for the Pod (the
 * .spec.volumes field) and where to mount those into Containers (the
 * .spec.containers[*].volumeMounts field).
 *
 * A process in a container sees a filesystem view composed from their Docker
 * image and volumes. The Docker image is at the root of the filesystem
 * hierarchy, and any volumes are mounted at the specified paths within the
 * image. Volumes can not mount onto other volumes
 */
class Volume {
    constructor(name, config) {
        this.name = name;
        this.config = config;
    }
    /**
     * Mounts an Amazon Web Services (AWS) EBS volume into your pod.
     * Unlike emptyDir, which is erased when a pod is removed, the contents of an EBS volume are
     * persisted and the volume is unmounted. This means that an EBS volume can be pre-populated with data,
     * and that data can be shared between pods.
     *
     * There are some restrictions when using an awsElasticBlockStore volume:
     *
     * - the nodes on which pods are running must be AWS EC2 instances.
     * - those instances need to be in the same region and availability zone as the EBS volume.
     * - EBS only supports a single EC2 instance mounting a volume.
     */
    static fromAwsElasticBlockStore(volumeId, options = {}) {
        var _b, _c, _d;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `ebs-${volumeId}`, {
            awsElasticBlockStore: {
                volumeId,
                fsType: (_c = options.fsType) !== null && _c !== void 0 ? _c : 'ext4',
                partition: options.partition,
                readOnly: (_d = options.readOnly) !== null && _d !== void 0 ? _d : false,
            },
        });
    }
    /**
     * Mounts a Microsoft Azure Data Disk into a pod.
     */
    static fromAzureDisk(diskName, diskUri, options = {}) {
        var _b, _c, _d, _e, _f;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `azuredisk-${diskName}`, {
            azureDisk: {
                diskName,
                diskUri,
                cachingMode: (_c = options.cachingMode) !== null && _c !== void 0 ? _c : AzureDiskPersistentVolumeCachingMode.NONE,
                fsType: (_d = options.fsType) !== null && _d !== void 0 ? _d : 'ext4',
                kind: (_e = options.kind) !== null && _e !== void 0 ? _e : AzureDiskPersistentVolumeKind.SHARED,
                readOnly: (_f = options.readOnly) !== null && _f !== void 0 ? _f : false,
            },
        });
    }
    /**
     * Mounts a Google Compute Engine (GCE) persistent disk (PD) into your Pod.
     * Unlike emptyDir, which is erased when a pod is removed, the contents of a PD are
     * preserved and the volume is merely unmounted. This means that a PD can be pre-populated
     * with data, and that data can be shared between pods.
     *
     * There are some restrictions when using a gcePersistentDisk:
     *
     * - the nodes on which Pods are running must be GCE VMs
     * - those VMs need to be in the same GCE project and zone as the persistent disk
     */
    static fromGcePersistentDisk(pdName, options = {}) {
        var _b, _c, _d;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `gcedisk-${pdName}`, {
            gcePersistentDisk: {
                pdName,
                fsType: (_c = options.fsType) !== null && _c !== void 0 ? _c : 'ext4',
                partition: options.partition,
                readOnly: (_d = options.readOnly) !== null && _d !== void 0 ? _d : false,
            },
        });
    }
    /**
     * Populate the volume from a ConfigMap.
     *
     * The configMap resource provides a way to inject configuration data into
     * Pods. The data stored in a ConfigMap object can be referenced in a volume
     * of type configMap and then consumed by containerized applications running
     * in a Pod.
     *
     * When referencing a configMap object, you can simply provide its name in the
     * volume to reference it. You can also customize the path to use for a
     * specific entry in the ConfigMap.
     *
     * @param configMap The config map to use to populate the volume.
     * @param options Options
     */
    static fromConfigMap(configMap, options = {}) {
        var _b;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `configmap-${configMap.name}`, {
            configMap: {
                name: configMap.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * An emptyDir volume is first created when a Pod is assigned to a Node, and
     * exists as long as that Pod is running on that node. As the name says, it is
     * initially empty. Containers in the Pod can all read and write the same
     * files in the emptyDir volume, though that volume can be mounted at the same
     * or different paths in each Container. When a Pod is removed from a node for
     * any reason, the data in the emptyDir is deleted forever.
     *
     * @see http://kubernetes.io/docs/user-guide/volumes#emptydir
     *
     * @param options - Additional options.
     */
    static fromEmptyDir(name, options = {}) {
        return new Volume(name, {
            emptyDir: {
                medium: options.medium,
                sizeLimit: options.sizeLimit
                    ? k8s.Quantity.fromString(`${options.sizeLimit.toMebibytes()}Mi`)
                    : undefined,
            },
        });
    }
    /**
     * Populate the volume from a Secret.
     *
     * A secret volume is used to pass sensitive information, such as passwords, to Pods.
     * You can store secrets in the Kubernetes API and mount them as files for use by pods
     * without coupling to Kubernetes directly.
     *
     * secret volumes are backed by tmpfs (a RAM-backed filesystem)
     * so they are never written to non-volatile storage.
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes/#secret
     *
     * @param secret The secret to use to populate the volume.
     * @param options Options
     */
    static fromSecret(secret, options = {}) {
        var _b;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `secret-${secret.name}`, {
            secret: {
                secretName: secret.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * Used to mount a PersistentVolume into a Pod.
     * PersistentVolumeClaims are a way for users to "claim" durable storage (such as a GCE PersistentDisk or an iSCSI volume)
     * without knowing the details of the particular cloud environment.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/
     */
    static fromPersistentVolumeClaim(pvc, options = {}) {
        var _b, _c;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `pvc-${pvc.name}`, {
            persistentVolumeClaim: {
                claimName: pvc.name,
                readOnly: (_c = options.readOnly) !== null && _c !== void 0 ? _c : false,
            },
        });
    }
    asVolume() {
        return this;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            name: this.name,
            ...this.config,
        };
    }
}
exports.Volume = Volume;
_a = JSII_RTTI_SYMBOL_1;
Volume[_a] = { fqn: "cdk8s-plus-22.Volume", version: "1.0.0-beta.191" };
/**
  * @internal
 */
Volume.renderItems = (items) => {
    if (!items) {
        return undefined;
    }
    const result = new Array();
    for (const key of Object.keys(items).sort()) {
        result.push({
            key,
            path: items[key].path,
            mode: items[key].mode,
        });
    }
    return result;
};
/**
 * The medium on which to store the volume.
 */
var EmptyDirMedium;
(function (EmptyDirMedium) {
    /**
     * The default volume of the backing node.
     */
    EmptyDirMedium["DEFAULT"] = "";
    /**
     * Mount a tmpfs (RAM-backed filesystem) for you instead. While tmpfs is very
     * fast, be aware that unlike disks, tmpfs is cleared on node reboot and any
     * files you write will count against your Container's memory limit.
     */
    EmptyDirMedium["MEMORY"] = "Memory";
})(EmptyDirMedium = exports.EmptyDirMedium || (exports.EmptyDirMedium = {}));
/**
 * Azure Disk kinds.
 */
var AzureDiskPersistentVolumeKind;
(function (AzureDiskPersistentVolumeKind) {
    /**
     * Multiple blob disks per storage account.
     */
    AzureDiskPersistentVolumeKind["SHARED"] = "Shared";
    /**
     * Single blob disk per storage account.
     */
    AzureDiskPersistentVolumeKind["DEDICATED"] = "Dedicated";
    /**
     * Azure managed data disk.
     */
    AzureDiskPersistentVolumeKind["MANAGED"] = "Managed";
})(AzureDiskPersistentVolumeKind = exports.AzureDiskPersistentVolumeKind || (exports.AzureDiskPersistentVolumeKind = {}));
/**
 * Azure disk caching modes.
 */
var AzureDiskPersistentVolumeCachingMode;
(function (AzureDiskPersistentVolumeCachingMode) {
    /**
     * None.
     */
    AzureDiskPersistentVolumeCachingMode["NONE"] = "None";
    /**
     * ReadOnly.
     */
    AzureDiskPersistentVolumeCachingMode["READ_ONLY"] = "ReadOnly";
    /**
     * ReadWrite.
     */
    AzureDiskPersistentVolumeCachingMode["READ_WRITE"] = "ReadWrite";
})(AzureDiskPersistentVolumeCachingMode = exports.AzureDiskPersistentVolumeCachingMode || (exports.AzureDiskPersistentVolumeCachingMode = {}));
//# sourceMappingURL=data:application/json;base64,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