import { ApiObject, ApiObjectMetadataDefinition, Duration } from 'cdk8s';
import { Construct } from 'constructs';
import { Resource, ResourceProps } from './base';
import { Container, ContainerProps } from './container';
import * as k8s from './imports/k8s';
import { RestartPolicy, PodTemplateProps, IPodTemplate, PodSecurityContext, HostAlias } from './pod';
import { IServiceAccount } from './service-account';
import { Volume } from './volume';
/**
 * Properties for initialization of `Job`.
 */
export interface JobProps extends ResourceProps, PodTemplateProps {
    /**
     * Specifies the duration the job may be active before the system tries to terminate it.
     *
     * @default - If unset, then there is no deadline.
     */
    readonly activeDeadline?: Duration;
    /**
     * Specifies the number of retries before marking this job failed.
     *
     * @default - If not set, system defaults to 6.
     */
    readonly backoffLimit?: number;
    /**
     * Limits the lifetime of a Job that has finished execution (either Complete
     * or Failed). If this field is set, after the Job finishes, it is eligible to
     * be automatically deleted. When the Job is being deleted, its lifecycle
     * guarantees (e.g. finalizers) will be honored. If this field is set to zero,
     * the Job becomes eligible to be deleted immediately after it finishes. This
     * field is alpha-level and is only honored by servers that enable the
     * `TTLAfterFinished` feature.
     *
     * @default - If this field is unset, the Job won't be automatically deleted.
     */
    readonly ttlAfterFinished?: Duration;
}
/**
 * A Job creates one or more Pods and ensures that a specified number of them successfully terminate. As pods successfully complete,
 * the Job tracks the successful completions. When a specified number of successful completions is reached, the task (ie, Job) is complete.
 * Deleting a Job will clean up the Pods it created. A simple case is to create one Job object in order to reliably run one Pod to completion.
 * The Job object will start a new Pod if the first Pod fails or is deleted (for example due to a node hardware failure or a node reboot).
 * You can also use a Job to run multiple Pods in parallel.
 */
export declare class Job extends Resource implements IPodTemplate {
    /**
     * Duration before job is terminated. If undefined, there is no deadline.
     */
    readonly activeDeadline?: Duration;
    /**
     * Number of retries before marking failed.
     */
    readonly backoffLimit?: number;
    /**
     * TTL before the job is deleted after it is finished.
     */
    readonly ttlAfterFinished?: Duration;
    /**
     * @see base.Resource.apiObject
     */
    protected readonly apiObject: ApiObject;
    private readonly _podTemplate;
    constructor(scope: Construct, id: string, props?: JobProps);
    get podMetadata(): ApiObjectMetadataDefinition;
    get containers(): Container[];
    get initContainers(): Container[];
    get hostAliases(): HostAlias[];
    get volumes(): Volume[];
    get restartPolicy(): RestartPolicy | undefined;
    get serviceAccount(): IServiceAccount | undefined;
    get securityContext(): PodSecurityContext;
    addContainer(container: ContainerProps): Container;
    addInitContainer(container: ContainerProps): Container;
    addHostAlias(hostAlias: HostAlias): void;
    addVolume(volume: Volume): void;
    /**
     * @internal
     */
    _toKube(): k8s.JobSpec;
}
