"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSageMakerEndpoint = void 0;
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToSageMakerEndpoint class.
 */
class ApiGatewayToSageMakerEndpoint extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayToSageMakerEndpoint class.
     * @since 1.68.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps);
        // Setup the API Gateway role
        if (props.apiGatewayExecutionRole !== undefined) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
            });
            // Setup the IAM policy for SageMaker endpoint
            const invokePolicy = new iam.Policy(this, 'InvokeEndpointPolicy', {
                statements: [
                    new iam.PolicyStatement({
                        actions: ['sagemaker:InvokeEndpoint'],
                        resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:endpoint/${props.endpointName}`]
                    })
                ]
            });
            invokePolicy.attachToRole(this.apiGatewayRole);
        }
        // Setup request validation
        const requestValidator = this.apiGateway.addRequestValidator('request-validator', {
            requestValidatorName: 'request-param-validator',
            // Setting this property to true makes sure the following are validated:
            // - Required request parameters in the URI
            // - Query string
            // - Headers
            validateRequestParameters: true
        });
        // Setup method and integration responses
        const methodResponses = [
            { statusCode: '200' },
            { statusCode: '500' },
            { statusCode: '400' }
        ];
        const integResponses = [];
        if (props.responseMappingTemplate !== undefined) {
            integResponses.push({
                statusCode: '200',
                responseTemplates: { 'application/json': props.responseMappingTemplate }
            });
        }
        else {
            integResponses.push({ statusCode: '200' });
        }
        integResponses.push({ statusCode: '500', selectionPattern: '5\\d{2}' }, { statusCode: '400', selectionPattern: '4\\d{2}' });
        // The SageMaker integration can be added either at the root of the API (GET https://execute-api.amazonaws.com/{some-param}),
        // or as a sub-resource (GET https://execute-api.amazonaws.com/inference/{some-param}).
        // The following lines will make sure only the necessary resources are created.
        let apiResource = this.apiGateway.root;
        if (props.resourceName !== undefined) {
            apiResource = apiResource.addResource(props.resourceName);
        }
        apiResource = apiResource.addResource(props.resourcePath);
        // Setup API Gateway method
        defaults.addProxyMethodToApiResource({
            service: 'runtime.sagemaker',
            path: `endpoints/${props.endpointName}/invocations`,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'GET',
            apiResource,
            requestValidator,
            requestTemplate: props.requestMappingTemplate,
            awsIntegrationProps: {
                options: { integrationResponses: integResponses }
            },
            methodOptions: { methodResponses }
        });
    }
}
exports.ApiGatewayToSageMakerEndpoint = ApiGatewayToSageMakerEndpoint;
//# sourceMappingURL=data:application/json;base64,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