"""Utilities specific to jupyter"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/jupyter_notebook.ipynb.

# %% auto 0
__all__ = ['save_notebook', 'backup_this_notebook', 'backup_all_notebooks', 'backup_folders_of_nbs', 'display_dfs_side_by_side',
           'show_big_dataframe', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'store_scrap', 'shutdown_current_notebook']

# %% ../nbs/jupyter_notebook.ipynb 2
import os, sys, json, time, hashlib
from .paths import stems, Glob, parent, P, stem, dumpdill
from .markup import read_json, writelines, makedir
from .logger import Info, Warn
from .loader import show, pd, is_in_notebook
from itertools import chain, cycle
from contextlib import contextmanager

# %% ../nbs/jupyter_notebook.ipynb 3
# | export

# %% ../nbs/jupyter_notebook.ipynb 4
def save_notebook(file_path):
    from IPython.display import display, Javascript

    start_md5 = hashlib.md5(open(file_path, "rb").read()).hexdigest()
    display(Javascript("IPython.notebook.save_checkpoint();"))
    current_md5 = start_md5

    while start_md5 == current_md5:
        time.sleep(1)
        current_md5 = hashlib.md5(open(file_path, "rb").read()).hexdigest()
    Info(f"Saved the notebook at {file_path}!")


def backup_this_notebook(
    this_file_path,
    save_html_to_dir=None,
    override_previous_backup=False,
    changelog=None,
    exclude_input=False,
    force_save_notebook=True,
):
    from nbconvert import HTMLExporter
    import nbformat

    if save_html_to_dir is None:
        save_html_to_dir = (
            parent(P(this_file_path)).resolve() / f"backups/{stem(this_file_path)}"
        )
        files = [f for f in stems(save_html_to_dir) if f.split("__")[-1].isdigit()]
        available_number = max([int(i.split("__")[-1]) for i in files], default=-1) + 1
        save_to = (
            f"{save_html_to_dir}/{stem(this_file_path)}__{available_number:04}.html"
        )
        if override_previous_backup:
            if available_number != 0:
                available_number -= 1
            if (
                input(
                    f"Are you sure you want to override `{save_html_to_dir}/{stem(this_file_path)}__{available_number:04}.html` ? [y/n]"
                ).lower()
                != "y"
            ):
                raise ValueError("Aborting")
            save_to = (
                f"{save_html_to_dir}/{stem(this_file_path)}__{available_number:04}.html"
            )
    else:
        save_to = f"{save_html_to_dir}/{stem(this_file_path)}.html"
        force_save_notebook = False
    makedir(parent(save_to))
    Info(f"Backing up this version of notebook to {save_to}")
    if force_save_notebook:
        save_notebook(this_file_path)
    this_notebook = nbformat.reads(
        json.dumps(read_json(this_file_path)),
        as_version=4,
    )

    html_exporter = HTMLExporter(template_name="classic")
    if exclude_input:
        html_exporter.exclude_input = True
    (body, resources) = html_exporter.from_notebook_node(this_notebook)
    writelines([body], save_to, mode="w")
    if changelog is None:
        Warn(
            "Use `changelog` argument to the devs know what is important in the backup"
        )
        changelog = ""
    changelog_file = P(save_html_to_dir) / "changelog.md"
    changelog_file.touch()
    changelog = f"\n## {stem(save_to)}\n{changelog}"
    changelog_file.write_lines(changelog.split("\n"), mode="a+")
    Info(f"Success! Visit {changelog_file} for detailed changes")
    return save_to


def backup_all_notebooks(folder):
    all_notebooks = P(folder).Glob("*.ipynb")
    for notebook in all_notebooks:
        backup_this_notebook(notebook, force_save_notebook=False)


def backup_folders_of_nbs(src, dest):
    if os.path.exists(dest):
        P(dest).rmtree()
    all_notebooks = Glob(f"{src}/*/*.ipynb")
    for nb in all_notebooks:
        _dest = f"{dest}/{parent(nb)}"
        backup_this_notebook(nb, _dest)
    for f in Glob(f"{dest}/*/changelog.md"):
        f.rm(confirm_prompt=False)

# %% ../nbs/jupyter_notebook.ipynb 6
def display_dfs_side_by_side(*args, titles=cycle([""]), max_rows=50):
    from IPython.display import display_html

    html_str = ""
    for df, title in zip(args, chain(titles, cycle(["</br>"]))):
        html_str += '<th style="text-align:center"><td style="vertical-align:top">'
        html_str += f'<h2 style="text-align: center;">{title}</h2>'
        html_str += df.to_html(max_rows=max_rows).replace(
            "table", 'table style="display:inline"'
        )
        html_str += "</td></th>"
    display_html(html_str, raw=True)


def show_big_dataframe(df, max_rows=30):
    with pd.option_context(
        "display.max_columns",
        1000,
        "display.max_colwidth",
        1000,
        "display.max_rows",
        max_rows,
    ):
        show(df, max_rows=max_rows, frame_count=2)

# %% ../nbs/jupyter_notebook.ipynb 7
def h1(text):
    from IPython.display import Markdown

    show(Markdown(f"## {text}"))


def h2(text):
    from IPython.display import Markdown

    show(Markdown(f"## {text}"))


def h3(text):
    from IPython.display import Markdown

    show(Markdown(f"### {text}"))


def h4(text):
    from IPython.display import Markdown

    show(Markdown(f"#### {text}"))


def h5(text):
    from IPython.display import Markdown

    show(Markdown(f"##### {text}"))


def h6(text):
    from IPython.display import Markdown

    show(Markdown(f"###### {text}"))

# %% ../nbs/jupyter_notebook.ipynb 8
@contextmanager
def store_scrap(at):
    from IPython.utils.capture import capture_output

    with capture_output() as scrap:
        yield scrap
    dumpdill(
        scrap,
        at,
        message='Stored scrap at {fpath} as a dill file.\nUse loaddill("{fpath}").show()',
    )
    return scrap

# %% ../nbs/jupyter_notebook.ipynb 9
# Function to shut down the current notebook session
def shutdown_current_notebook(delay: int = None):
    if delay:
        time.sleep(delay)
    os.kill(os.getpid(), 9)
