# Copyright (c) <2020> <Scott Russell>

# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

# Documentation available at https://github.com/scrussell24/hype-html

import enum


class Indent(enum.Enum):
    TWO_SPACES = '  '
    FOUR_SPACES = '    '
    TAB = '\t'


class Element:
    self_closing = False

    @property
    def tag(self):
        raise RuntimeError("Element requires tag property to render.")

    def __init__(self, *args, **kwargs):
        if self.self_closing and len(args):
            raise RuntimeError('Self closing elements cannot have inner elements.')

        self.inner_elements = list(args)
        self.props = {}
        for k, v in kwargs.items():
            key, val = self._process_attr(k, v)
            self.props[key] = val

    @staticmethod
    def _process_attr(k, v):
        if k.startswith('_'):
            k = k[1:]
        k = k.replace('_', '-')
        return k, v

    @staticmethod
    def _create_attr_string(k, v):
        if v is True:
            return k
        elif v:
            return f'{k}="{v}"'
        return None

    def append(self, el):
        self.inner_elements.append(el)

    def attrs(self, **kwargs):
        for k, v in kwargs.items():
            key, val = self._process_attr(k, v)
            self.props[key] = val

    def __call__(self, indent_level=0, indent=Indent.TWO_SPACES):
        # props
        props = [self._create_attr_string(k, v) for k, v in self.props.items() if v is not None]
        props = [p for p in props if p]
        prop_space = ' ' if len(props) else ''

        # indent
        indent_chars = '\n' + ''.join([indent.value for n in range(indent_level)])
        end_tag_indent = indent_chars if any(
            [True for el in self.inner_elements if isinstance(el, Element)]
        ) else ''

        # elements
        els = []
        for a in self.inner_elements:
            if isinstance(a, Element):
                els.append(a(indent_level=indent_level + 1, indent=indent))
            else:
                els.append(str(a))

        if self.self_closing:
            return f'{indent_chars}<{self.tag}{prop_space}{" ".join(props)}/>'
        return f'{indent_chars}<{self.tag}{prop_space}{" ".join(props)}' + \
            f'>{"".join(els)}{end_tag_indent}</{self.tag}>'

    def __str__(self):
        return self()


class SelfClosingElement(Element):
    self_closing = True

    def __init__(self, **kwargs):
        super().__init__(**kwargs)


class Doc:
    def __init__(self, *args, indent=Indent.TWO_SPACES):
        self.indent = indent
        self.elements = args

    def __call__(self):
        doc = '<!DOCTYPE html>'
        for el in self.elements:
            if isinstance(el, Element):
                doc += el(indent=self.indent)
            else:
                doc += str(el)
        return doc

    def __str__(self):
        return self()

# Below code is autogenerated by rendering the tmplt.element.py.j2 file


class A(Element):
    tag = 'a'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        download=None,
        hidden=None,
        href=None,
        hreflang=None,
        _id=None,
        itemprop=None,
        lang=None,
        ping=None,
        referrerpolicy=None,
        rel=None,
        shape=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        target=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "download": download,
            "hidden": hidden,
            "href": href,
            "hreflang": hreflang,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "ping": ping,
            "referrerpolicy": referrerpolicy,
            "rel": rel,
            "shape": shape,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "target": target,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Abbr(Element):
    tag = 'abbr'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Address(Element):
    tag = 'address'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Area(SelfClosingElement):
    tag = 'area'

    def __init__(
        self,
        *args,
        accesskey=None,
        alt=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        coords=None,
        _dir=None,
        download=None,
        hidden=None,
        href=None,
        hreflang=None,
        _id=None,
        itemprop=None,
        lang=None,
        ping=None,
        referrerpolicy=None,
        rel=None,
        shape=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        target=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "alt": alt,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "coords": coords,
            "_dir": _dir,
            "download": download,
            "hidden": hidden,
            "href": href,
            "hreflang": hreflang,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "ping": ping,
            "referrerpolicy": referrerpolicy,
            "rel": rel,
            "shape": shape,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "target": target,
            "title": title,
            "translate": translate,
        }
        super().__init__(**{**props, **kwargs})


class Article(Element):
    tag = 'article'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Aside(Element):
    tag = 'aside'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Audio(Element):
    tag = 'audio'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        autoplay=None,
        buffered=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        controls=None,
        crossorigin=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        loop=None,
        muted=None,
        preload=None,
        slot=None,
        spellcheck=None,
        src=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "autoplay": autoplay,
            "buffered": buffered,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "controls": controls,
            "crossorigin": crossorigin,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "loop": loop,
            "muted": muted,
            "preload": preload,
            "slot": slot,
            "spellcheck": spellcheck,
            "src": src,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class B(Element):
    tag = 'b'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Base(SelfClosingElement):
    tag = 'base'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        href=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        target=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "href": href,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "target": target,
            "title": title,
            "translate": translate,
        }
        super().__init__(**{**props, **kwargs})


class Bdi(Element):
    tag = 'bdi'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Bdo(Element):
    tag = 'bdo'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Blockquote(Element):
    tag = 'blockquote'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        cite=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "cite": cite,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Body(Element):
    tag = 'body'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        background=None,
        bgcolor=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "background": background,
            "bgcolor": bgcolor,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Br(SelfClosingElement):
    tag = 'br'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(**{**props, **kwargs})


class Button(Element):
    tag = 'button'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        autofocus=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        disabled=None,
        form=None,
        formaction=None,
        formenctype=None,
        formmethod=None,
        formnovalidate=None,
        formtarget=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        name=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        type=None,
        value=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "autofocus": autofocus,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "disabled": disabled,
            "form": form,
            "formaction": formaction,
            "formenctype": formenctype,
            "formmethod": formmethod,
            "formnovalidate": formnovalidate,
            "formtarget": formtarget,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "name": name,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "type": type,
            "value": value,
        }
        super().__init__(*args, **{**props, **kwargs})


class Canvas(Element):
    tag = 'canvas'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        height=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        width=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "height": height,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "width": width,
        }
        super().__init__(*args, **{**props, **kwargs})


class Caption(Element):
    tag = 'caption'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Cite(Element):
    tag = 'cite'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Code(Element):
    tag = 'code'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Col(SelfClosingElement):
    tag = 'col'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        autocapitalize=None,
        bgcolor=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        span=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "autocapitalize": autocapitalize,
            "bgcolor": bgcolor,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "span": span,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(**{**props, **kwargs})


class Colgroup(Element):
    tag = 'colgroup'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        autocapitalize=None,
        bgcolor=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        span=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "autocapitalize": autocapitalize,
            "bgcolor": bgcolor,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "span": span,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Data(Element):
    tag = 'data'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        value=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "value": value,
        }
        super().__init__(*args, **{**props, **kwargs})


class Datalist(Element):
    tag = 'datalist'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Dd(Element):
    tag = 'dd'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Del(Element):
    tag = 'del'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        cite=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        datetime=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "cite": cite,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "datetime": datetime,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Details(Element):
    tag = 'details'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        open=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "open": open,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Dfn(Element):
    tag = 'dfn'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Dialog(Element):
    tag = 'dialog'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Div(Element):
    tag = 'div'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Dl(Element):
    tag = 'dl'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Dt(Element):
    tag = 'dt'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Em(Element):
    tag = 'em'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Embed(SelfClosingElement):
    tag = 'embed'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        height=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        src=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        type=None,
        width=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "height": height,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "src": src,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "type": type,
            "width": width,
        }
        super().__init__(**{**props, **kwargs})


class Fieldset(Element):
    tag = 'fieldset'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        disabled=None,
        form=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        name=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "disabled": disabled,
            "form": form,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "name": name,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Figcaption(Element):
    tag = 'figcaption'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Figure(Element):
    tag = 'figure'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Footer(Element):
    tag = 'footer'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Form(Element):
    tag = 'form'

    def __init__(
        self,
        *args,
        accept=None,
        accept_charset=None,
        accesskey=None,
        action=None,
        autocapitalize=None,
        autocomplete=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        download=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        media=None,
        method=None,
        name=None,
        novalidate=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        target=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accept": accept,
            "accept_charset": accept_charset,
            "accesskey": accesskey,
            "action": action,
            "autocapitalize": autocapitalize,
            "autocomplete": autocomplete,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "download": download,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "media": media,
            "method": method,
            "name": name,
            "novalidate": novalidate,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "target": target,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class H1(Element):
    tag = 'h1'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class H2(Element):
    tag = 'h2'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class H3(Element):
    tag = 'h3'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class H4(Element):
    tag = 'h4'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class H5(Element):
    tag = 'h5'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class H6(Element):
    tag = 'h6'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Head(Element):
    tag = 'head'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Header(Element):
    tag = 'header'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Hgroup(Element):
    tag = 'hgroup'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Hr(SelfClosingElement):
    tag = 'hr'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        autocapitalize=None,
        _class=None,
        color=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "color": color,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(**{**props, **kwargs})


class Html(Element):
    tag = 'html'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        manifest=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "manifest": manifest,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class I(Element):
    tag = 'i'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Iframe(Element):
    tag = 'iframe'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        allow=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        csp=None,
        _dir=None,
        height=None,
        hidden=None,
        _id=None,
        importance=None,
        itemprop=None,
        lang=None,
        loading=None,
        name=None,
        referrerpolicy=None,
        sandbox=None,
        slot=None,
        spellcheck=None,
        src=None,
        srcdoc=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        width=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "allow": allow,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "csp": csp,
            "_dir": _dir,
            "height": height,
            "hidden": hidden,
            "_id": _id,
            "importance": importance,
            "itemprop": itemprop,
            "lang": lang,
            "loading": loading,
            "name": name,
            "referrerpolicy": referrerpolicy,
            "sandbox": sandbox,
            "slot": slot,
            "spellcheck": spellcheck,
            "src": src,
            "srcdoc": srcdoc,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "width": width,
        }
        super().__init__(*args, **{**props, **kwargs})


class Img(SelfClosingElement):
    tag = 'img'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        alt=None,
        autocapitalize=None,
        border=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        crossorigin=None,
        decoding=None,
        _dir=None,
        height=None,
        hidden=None,
        _id=None,
        importance=None,
        intrinsicsize=None,
        ismap=None,
        itemprop=None,
        lang=None,
        loading=None,
        referrerpolicy=None,
        sizes=None,
        slot=None,
        spellcheck=None,
        src=None,
        srcset=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        usemap=None,
        width=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "alt": alt,
            "autocapitalize": autocapitalize,
            "border": border,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "crossorigin": crossorigin,
            "decoding": decoding,
            "_dir": _dir,
            "height": height,
            "hidden": hidden,
            "_id": _id,
            "importance": importance,
            "intrinsicsize": intrinsicsize,
            "ismap": ismap,
            "itemprop": itemprop,
            "lang": lang,
            "loading": loading,
            "referrerpolicy": referrerpolicy,
            "sizes": sizes,
            "slot": slot,
            "spellcheck": spellcheck,
            "src": src,
            "srcset": srcset,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "usemap": usemap,
            "width": width,
        }
        super().__init__(**{**props, **kwargs})


class Input(SelfClosingElement):
    tag = 'input'

    def __init__(
        self,
        *args,
        accept=None,
        accesskey=None,
        alt=None,
        autocapitalize=None,
        autocomplete=None,
        autofocus=None,
        capture=None,
        checked=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        dirname=None,
        form=None,
        formaction=None,
        formenctype=None,
        formmethod=None,
        formnovalidate=None,
        formtarget=None,
        height=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        list=None,
        max=None,
        maxlength=None,
        minlength=None,
        min=None,
        multiple=None,
        name=None,
        pattern=None,
        placeholder=None,
        readonly=None,
        required=None,
        size=None,
        slot=None,
        spellcheck=None,
        src=None,
        step=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        type=None,
        usemap=None,
        value=None,
        width=None,
        **kwargs
    ):
        props = {
            "accept": accept,
            "accesskey": accesskey,
            "alt": alt,
            "autocapitalize": autocapitalize,
            "autocomplete": autocomplete,
            "autofocus": autofocus,
            "capture": capture,
            "checked": checked,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "dirname": dirname,
            "form": form,
            "formaction": formaction,
            "formenctype": formenctype,
            "formmethod": formmethod,
            "formnovalidate": formnovalidate,
            "formtarget": formtarget,
            "height": height,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "list": list,
            "max": max,
            "maxlength": maxlength,
            "minlength": minlength,
            "min": min,
            "multiple": multiple,
            "name": name,
            "pattern": pattern,
            "placeholder": placeholder,
            "readonly": readonly,
            "required": required,
            "size": size,
            "slot": slot,
            "spellcheck": spellcheck,
            "src": src,
            "step": step,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "type": type,
            "usemap": usemap,
            "value": value,
            "width": width,
        }
        super().__init__(**{**props, **kwargs})


class Ins(Element):
    tag = 'ins'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        cite=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        datetime=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "cite": cite,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "datetime": datetime,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Kbd(Element):
    tag = 'kbd'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Label(Element):
    tag = 'label'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        _for=None,
        form=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "_for": _for,
            "form": form,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Legend(Element):
    tag = 'legend'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Li(Element):
    tag = 'li'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        value=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "value": value,
        }
        super().__init__(*args, **{**props, **kwargs})


class Link(SelfClosingElement):
    tag = 'link'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        crossorigin=None,
        _dir=None,
        hidden=None,
        href=None,
        hreflang=None,
        _id=None,
        importance=None,
        integrity=None,
        itemprop=None,
        lang=None,
        referrerpolicy=None,
        rel=None,
        sizes=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "crossorigin": crossorigin,
            "_dir": _dir,
            "hidden": hidden,
            "href": href,
            "hreflang": hreflang,
            "_id": _id,
            "importance": importance,
            "integrity": integrity,
            "itemprop": itemprop,
            "lang": lang,
            "referrerpolicy": referrerpolicy,
            "rel": rel,
            "sizes": sizes,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(**{**props, **kwargs})


class Main(Element):
    tag = 'main'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Map(Element):
    tag = 'map'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        name=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "name": name,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Mark(Element):
    tag = 'mark'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Math(Element):
    tag = 'math'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Menu(Element):
    tag = 'menu'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        type=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "type": type,
        }
        super().__init__(*args, **{**props, **kwargs})


class Menuitem(SelfClosingElement):
    tag = 'menuitem'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(**{**props, **kwargs})


class Meta(SelfClosingElement):
    tag = 'meta'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        charset=None,
        _class=None,
        content=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        http_equiv=None,
        _id=None,
        itemprop=None,
        lang=None,
        name=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "charset": charset,
            "_class": _class,
            "content": content,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "http_equiv": http_equiv,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "name": name,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(**{**props, **kwargs})


class Meter(Element):
    tag = 'meter'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        form=None,
        hidden=None,
        high=None,
        _id=None,
        itemprop=None,
        lang=None,
        low=None,
        max=None,
        min=None,
        optimum=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        value=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "form": form,
            "hidden": hidden,
            "high": high,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "low": low,
            "max": max,
            "min": min,
            "optimum": optimum,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "value": value,
        }
        super().__init__(*args, **{**props, **kwargs})


class Nav(Element):
    tag = 'nav'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Noscript(Element):
    tag = 'noscript'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Object(Element):
    tag = 'object'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        border=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        data=None,
        _dir=None,
        form=None,
        height=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        name=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        type=None,
        usemap=None,
        width=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "border": border,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "data": data,
            "_dir": _dir,
            "form": form,
            "height": height,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "name": name,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "type": type,
            "usemap": usemap,
            "width": width,
        }
        super().__init__(*args, **{**props, **kwargs})


class Ol(Element):
    tag = 'ol'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        reversed=None,
        slot=None,
        spellcheck=None,
        start=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "reversed": reversed,
            "slot": slot,
            "spellcheck": spellcheck,
            "start": start,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Optgroup(Element):
    tag = 'optgroup'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        disabled=None,
        hidden=None,
        _id=None,
        itemprop=None,
        label=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "disabled": disabled,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "label": label,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Option(Element):
    tag = 'option'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        disabled=None,
        hidden=None,
        _id=None,
        itemprop=None,
        label=None,
        lang=None,
        selected=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        value=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "disabled": disabled,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "label": label,
            "lang": lang,
            "selected": selected,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "value": value,
        }
        super().__init__(*args, **{**props, **kwargs})


class Output(Element):
    tag = 'output'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        _for=None,
        form=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        name=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "_for": _for,
            "form": form,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "name": name,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class P(Element):
    tag = 'p'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Param(SelfClosingElement):
    tag = 'param'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        name=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        value=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "name": name,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "value": value,
        }
        super().__init__(**{**props, **kwargs})


class Picture(Element):
    tag = 'picture'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Pre(Element):
    tag = 'pre'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Progress(Element):
    tag = 'progress'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        form=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        max=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        value=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "form": form,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "max": max,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "value": value,
        }
        super().__init__(*args, **{**props, **kwargs})


class Q(Element):
    tag = 'q'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        cite=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "cite": cite,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Rb(Element):
    tag = 'rb'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Rp(Element):
    tag = 'rp'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Rt(Element):
    tag = 'rt'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Rtc(Element):
    tag = 'rtc'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Ruby(Element):
    tag = 'ruby'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class S(Element):
    tag = 's'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Samp(Element):
    tag = 'samp'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Script(Element):
    tag = 'script'

    def __init__(
        self,
        *args,
        accesskey=None,
        _async=None,
        autocapitalize=None,
        charset=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        crossorigin=None,
        defer=None,
        _dir=None,
        hidden=None,
        _id=None,
        importance=None,
        integrity=None,
        itemprop=None,
        lang=None,
        language=None,
        referrerpolicy=None,
        slot=None,
        spellcheck=None,
        src=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        type=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "_async": _async,
            "autocapitalize": autocapitalize,
            "charset": charset,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "crossorigin": crossorigin,
            "defer": defer,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "importance": importance,
            "integrity": integrity,
            "itemprop": itemprop,
            "lang": lang,
            "language": language,
            "referrerpolicy": referrerpolicy,
            "slot": slot,
            "spellcheck": spellcheck,
            "src": src,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "type": type,
        }
        super().__init__(*args, **{**props, **kwargs})


class Section(Element):
    tag = 'section'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Select(Element):
    tag = 'select'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        autocomplete=None,
        autofocus=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        disabled=None,
        form=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        multiple=None,
        name=None,
        required=None,
        size=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "autocomplete": autocomplete,
            "autofocus": autofocus,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "disabled": disabled,
            "form": form,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "multiple": multiple,
            "name": name,
            "required": required,
            "size": size,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Slot(Element):
    tag = 'slot'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Small(Element):
    tag = 'small'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Source(SelfClosingElement):
    tag = 'source'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        sizes=None,
        slot=None,
        spellcheck=None,
        src=None,
        srcset=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        type=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "sizes": sizes,
            "slot": slot,
            "spellcheck": spellcheck,
            "src": src,
            "srcset": srcset,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "type": type,
        }
        super().__init__(**{**props, **kwargs})


class Span(Element):
    tag = 'span'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Strong(Element):
    tag = 'strong'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Style(Element):
    tag = 'style'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        scoped=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        type=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "scoped": scoped,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "type": type,
        }
        super().__init__(*args, **{**props, **kwargs})


class Sub(Element):
    tag = 'sub'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Summary(Element):
    tag = 'summary'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Sup(Element):
    tag = 'sup'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Svg(Element):
    tag = 'svg'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Table(Element):
    tag = 'table'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        autocapitalize=None,
        background=None,
        bgcolor=None,
        border=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        summary=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "autocapitalize": autocapitalize,
            "background": background,
            "bgcolor": bgcolor,
            "border": border,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "summary": summary,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Tbody(Element):
    tag = 'tbody'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        autocapitalize=None,
        bgcolor=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "autocapitalize": autocapitalize,
            "bgcolor": bgcolor,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Td(Element):
    tag = 'td'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        autocapitalize=None,
        background=None,
        bgcolor=None,
        _class=None,
        colspan=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        headers=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        rowspan=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "autocapitalize": autocapitalize,
            "background": background,
            "bgcolor": bgcolor,
            "_class": _class,
            "colspan": colspan,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "headers": headers,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "rowspan": rowspan,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Template(Element):
    tag = 'template'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Textarea(Element):
    tag = 'textarea'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        autocomplete=None,
        autofocus=None,
        _class=None,
        cols=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        dirname=None,
        disabled=None,
        enterkeyhit=None,
        form=None,
        hidden=None,
        _id=None,
        inputmode=None,
        itemprop=None,
        lang=None,
        maxlength=None,
        minlength=None,
        name=None,
        placeholder=None,
        readonly=None,
        required=None,
        rows=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "autocomplete": autocomplete,
            "autofocus": autofocus,
            "_class": _class,
            "cols": cols,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "dirname": dirname,
            "disabled": disabled,
            "enterkeyhit": enterkeyhit,
            "form": form,
            "hidden": hidden,
            "_id": _id,
            "inputmode": inputmode,
            "itemprop": itemprop,
            "lang": lang,
            "maxlength": maxlength,
            "minlength": minlength,
            "name": name,
            "placeholder": placeholder,
            "readonly": readonly,
            "required": required,
            "rows": rows,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Tfoot(Element):
    tag = 'tfoot'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        autocapitalize=None,
        bgcolor=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "autocapitalize": autocapitalize,
            "bgcolor": bgcolor,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Th(Element):
    tag = 'th'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        autocapitalize=None,
        background=None,
        bgcolor=None,
        _class=None,
        colspan=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        headers=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        rowspan=None,
        scope=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "autocapitalize": autocapitalize,
            "background": background,
            "bgcolor": bgcolor,
            "_class": _class,
            "colspan": colspan,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "headers": headers,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "rowspan": rowspan,
            "scope": scope,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Thead(Element):
    tag = 'thead'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Time(Element):
    tag = 'time'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        datetime=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "datetime": datetime,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Title(Element):
    tag = 'title'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Tr(Element):
    tag = 'tr'

    def __init__(
        self,
        *args,
        accesskey=None,
        align=None,
        autocapitalize=None,
        bgcolor=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "align": align,
            "autocapitalize": autocapitalize,
            "bgcolor": bgcolor,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Track(SelfClosingElement):
    tag = 'track'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        default=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        kind=None,
        label=None,
        lang=None,
        slot=None,
        spellcheck=None,
        src=None,
        srclang=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "default": default,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "kind": kind,
            "label": label,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "src": src,
            "srclang": srclang,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(**{**props, **kwargs})


class U(Element):
    tag = 'u'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Ul(Element):
    tag = 'ul'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Var(Element):
    tag = 'var'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(*args, **{**props, **kwargs})


class Video(Element):
    tag = 'video'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        autoplay=None,
        buffered=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        controls=None,
        crossorigin=None,
        _dir=None,
        height=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        muted=None,
        poster=None,
        preload=None,
        slot=None,
        spellcheck=None,
        src=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        width=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "autoplay": autoplay,
            "buffered": buffered,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "controls": controls,
            "crossorigin": crossorigin,
            "_dir": _dir,
            "height": height,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "muted": muted,
            "poster": poster,
            "preload": preload,
            "slot": slot,
            "spellcheck": spellcheck,
            "src": src,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
            "width": width,
        }
        super().__init__(*args, **{**props, **kwargs})


class Wbr(SelfClosingElement):
    tag = 'wbr'

    def __init__(
        self,
        *args,
        accesskey=None,
        autocapitalize=None,
        _class=None,
        contenteditable=None,
        contextmenu=None,
        _dir=None,
        hidden=None,
        _id=None,
        itemprop=None,
        lang=None,
        slot=None,
        spellcheck=None,
        style=None,
        tabindex=None,
        title=None,
        translate=None,
        **kwargs
    ):
        props = {
            "accesskey": accesskey,
            "autocapitalize": autocapitalize,
            "_class": _class,
            "contenteditable": contenteditable,
            "contextmenu": contextmenu,
            "_dir": _dir,
            "hidden": hidden,
            "_id": _id,
            "itemprop": itemprop,
            "lang": lang,
            "slot": slot,
            "spellcheck": spellcheck,
            "style": style,
            "tabindex": tabindex,
            "title": title,
            "translate": translate,
        }
        super().__init__(**{**props, **kwargs})

