"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsiteOrigin = exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const cloudfront_web_acl_1 = require("./cloudfront-web-acl");
const DEFAULT_RUNTIME_CONFIG_FILENAME = "runtime-config.json";
/**
 * Deploys a Static Website using by default a private S3 bucket as an origin and Cloudfront as the entrypoint.
 *
 * This construct configures a webAcl containing rules that are generally applicable to web applications. This
 * provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk
 * and commonly occurring vulnerabilities described in OWASP publications such as OWASP Top 10.
 *
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.validateProps = (props) => {
            this.validateEncryptionSettings(props);
            props.runtimeOptions && this.validateRuntimeConfig(props.runtimeOptions);
            props.websiteBucket && this.validateBucketConfig(props.websiteBucket);
        };
        this.validateRuntimeConfig = (config) => {
            if (!config) {
                throw new Error("validateRuntimeConfig only accepts non-null RuntimeOptions.");
            }
            if (config.jsonFileName && !config.jsonFileName.endsWith(".json")) {
                throw new Error("RuntimeOptions.jsonFileName must be a json file.");
            }
        };
        this.validateBucketConfig = (bucket) => {
            if (bucket.isWebsite) {
                throw new Error("Website buckets cannot be configured as websites as this will break Cloudfront hosting!");
            }
        };
        this.validateEncryptionSettings = ({ defaultWebsiteBucketEncryption, defaultWebsiteBucketEncryptionKey, }) => {
            if (defaultWebsiteBucketEncryptionKey &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS) {
                throw new Error("Bucket encryption should be set to KMS if providing a defaultWebsiteBucketEncryptionKey.");
            }
            if (defaultWebsiteBucketEncryption &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.S3_MANAGED) {
                throw new Error("Only KMS and S3_MANAGED encryption are supported on the default bucket.");
            }
        };
        this.suppressCDKNagViolations = (props) => {
            const stack = aws_cdk_lib_1.Stack.of(this);
            !props.distributionProps?.certificate &&
                [
                    "AwsSolutions-CFR4",
                    "AwsPrototyping-CloudFrontDistributionHttpsViewerNoOutdatedSSL",
                ].forEach((RuleId) => {
                    cdk_nag_1.NagSuppressions.addResourceSuppressions(this.cloudFrontDistribution, [
                        {
                            id: RuleId,
                            reason: "Certificate is not mandatory therefore the Cloudfront certificate will be used.",
                        },
                    ]);
                });
            ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                    {
                        id: RuleId,
                        reason: "Latest runtime cannot be configured. CDK will need to upgrade the BucketDeployment construct accordingly.",
                    },
                ]);
            });
            ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                    {
                        id: RuleId,
                        reason: "All Policies have been scoped to a Bucket. Given Buckets can contain arbitrary content, wildcard resources with bucket scope are required.",
                        appliesTo: [
                            {
                                regex: "/^Action::s3:.*$/g",
                            },
                            {
                                regex: `/^Resource::.*$/g`,
                            },
                        ],
                    },
                ]);
            });
            ["AwsSolutions-IAM4", "AwsPrototyping-IAMNoManagedPolicies"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                    {
                        id: RuleId,
                        reason: "Buckets can contain arbitrary content, therefore wildcard resources under a bucket are required.",
                        appliesTo: [
                            {
                                regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole$/g`,
                            },
                        ],
                    },
                ]);
            });
            ["AwsSolutions-S1", "AwsPrototyping-S3BucketLoggingEnabled"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                    {
                        id: RuleId,
                        reason: "Access Log buckets should not have s3 bucket logging",
                    },
                ]);
            });
        };
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        this.validateProps(props);
        const accessLogsBucket = new aws_s3_1.Bucket(this, "AccessLogsBucket", {
            versioned: false,
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            objectOwnership: aws_s3_1.ObjectOwnership.OBJECT_WRITER,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
        });
        // S3 Bucket to hold website files
        this.websiteBucket =
            props.websiteBucket ??
                new aws_s3_1.Bucket(this, "WebsiteBucket", {
                    versioned: true,
                    enforceSSL: true,
                    autoDeleteObjects: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                    objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_ENFORCED,
                    encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                    publicReadAccess: false,
                    blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                    serverAccessLogsPrefix: "website-access-logs",
                    serverAccessLogsBucket: accessLogsBucket,
                });
        // Web ACL
        const { distributionProps } = props;
        const webAclArn = distributionProps?.webAclId ??
            new cloudfront_web_acl_1.CloudfrontWebAcl(this, "WebsiteAcl", props.webAclProps).webAclArn;
        // Cloudfront Distribution
        const logBucket = props.distributionProps?.logBucket ||
            new aws_s3_1.Bucket(this, "DistributionLogBucket", {
                enforceSSL: true,
                autoDeleteObjects: true,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_PREFERRED,
                encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                publicReadAccess: false,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                serverAccessLogsPrefix: "distribution-access-logs",
                serverAccessLogsBucket: accessLogsBucket,
            });
        const originAccessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, "OriginAccessIdentity");
        this.websiteBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [this.websiteBucket.bucketArn],
            actions: ["s3:ListBucket"],
            principals: [originAccessIdentity.grantPrincipal],
        }));
        const defaultRootObject = distributionProps?.defaultRootObject ?? "index.html";
        this.cloudFrontDistribution = new aws_cloudfront_1.Distribution(this, "CloudfrontDistribution", {
            ...distributionProps,
            webAclId: webAclArn,
            enableLogging: true,
            logBucket: logBucket,
            defaultBehavior: {
                ...distributionProps?.defaultBehavior,
                origin: new aws_cloudfront_origins_1.S3Origin(this.websiteBucket, {
                    originAccessIdentity,
                }),
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            defaultRootObject,
            errorResponses: distributionProps?.errorResponses ?? [
                {
                    httpStatus: 404,
                    responseHttpStatus: 200,
                    responsePagePath: `/${defaultRootObject}`,
                },
            ],
        });
        // Deploy Website
        this.bucketDeployment = new aws_s3_deployment_1.BucketDeployment(this, "WebsiteDeployment", {
            sources: [
                aws_s3_deployment_1.Source.asset(props.websiteContentPath),
                ...(props.runtimeOptions
                    ? [
                        aws_s3_deployment_1.Source.jsonData(props.runtimeOptions?.jsonFileName ||
                            DEFAULT_RUNTIME_CONFIG_FILENAME, props.runtimeOptions?.jsonPayload),
                    ]
                    : []),
            ],
            destinationBucket: this.websiteBucket,
            // Files in the distribution's edge caches will be invalidated after files are uploaded to the destination bucket.
            distribution: this.cloudFrontDistribution,
        });
        this.suppressCDKNagViolations(props);
    }
}
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsite", version: "0.18.7" };
exports.StaticWebsite = StaticWebsite;
/**
 * If passing in distributionProps, the default behaviour.origin is a required parameter. An instance of this class can be passed in
 * to make the compiler happy.
 */
class StaticWebsiteOrigin {
    bind(_scope, _options) {
        throw new Error("This should never be called");
    }
}
_b = JSII_RTTI_SYMBOL_1;
StaticWebsiteOrigin[_b] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsiteOrigin", version: "0.18.7" };
exports.StaticWebsiteOrigin = StaticWebsiteOrigin;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGljLXdlYnNpdGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvc3RhdGljLXdlYnNpdGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTtzQ0FDc0M7QUFDdEMsMERBQXNEO0FBQ3RELDZDQUFtRDtBQUNuRCwrREFRb0M7QUFDcEMsK0VBQThEO0FBQzlELGlEQUFzRDtBQUV0RCwrQ0FNNEI7QUFDNUIscUVBQXlFO0FBQ3pFLHFDQUEwQztBQUMxQywyQ0FBdUM7QUFDdkMsNkRBQStFO0FBRS9FLE1BQU0sK0JBQStCLEdBQUcscUJBQXFCLENBQUM7QUFvRjlEOzs7Ozs7O0dBT0c7QUFDSCxNQUFhLGFBQWMsU0FBUSxzQkFBUztJQUsxQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXlCO1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUEySFgsa0JBQWEsR0FBRyxDQUFDLEtBQXlCLEVBQUUsRUFBRTtZQUNwRCxJQUFJLENBQUMsMEJBQTBCLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDdkMsS0FBSyxDQUFDLGNBQWMsSUFBSSxJQUFJLENBQUMscUJBQXFCLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBQ3pFLEtBQUssQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLG9CQUFvQixDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUN4RSxDQUFDLENBQUM7UUFFTSwwQkFBcUIsR0FBRyxDQUFDLE1BQXNCLEVBQUUsRUFBRTtZQUN6RCxJQUFJLENBQUMsTUFBTSxFQUFFO2dCQUNYLE1BQU0sSUFBSSxLQUFLLENBQ2IsNkRBQTZELENBQzlELENBQUM7YUFDSDtZQUVELElBQUksTUFBTSxDQUFDLFlBQVksSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFO2dCQUNqRSxNQUFNLElBQUksS0FBSyxDQUFDLGtEQUFrRCxDQUFDLENBQUM7YUFDckU7UUFDSCxDQUFDLENBQUM7UUFFTSx5QkFBb0IsR0FBRyxDQUFDLE1BQWUsRUFBRSxFQUFFO1lBQ2pELElBQUksTUFBTSxDQUFDLFNBQVMsRUFBRTtnQkFDcEIsTUFBTSxJQUFJLEtBQUssQ0FDYix5RkFBeUYsQ0FDMUYsQ0FBQzthQUNIO1FBQ0gsQ0FBQyxDQUFDO1FBRU0sK0JBQTBCLEdBQUcsQ0FBQyxFQUNwQyw4QkFBOEIsRUFDOUIsaUNBQWlDLEdBQ2QsRUFBRSxFQUFFO1lBQ3ZCLElBQ0UsaUNBQWlDO2dCQUNqQyw4QkFBOEIsS0FBSyx5QkFBZ0IsQ0FBQyxHQUFHLEVBQ3ZEO2dCQUNBLE1BQU0sSUFBSSxLQUFLLENBQ2IsMEZBQTBGLENBQzNGLENBQUM7YUFDSDtZQUVELElBQ0UsOEJBQThCO2dCQUM5Qiw4QkFBOEIsS0FBSyx5QkFBZ0IsQ0FBQyxHQUFHO2dCQUN2RCw4QkFBOEIsS0FBSyx5QkFBZ0IsQ0FBQyxVQUFVLEVBQzlEO2dCQUNBLE1BQU0sSUFBSSxLQUFLLENBQ2IseUVBQXlFLENBQzFFLENBQUM7YUFDSDtRQUNILENBQUMsQ0FBQztRQUVNLDZCQUF3QixHQUFHLENBQUMsS0FBeUIsRUFBRSxFQUFFO1lBQy9ELE1BQU0sS0FBSyxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdCLENBQUMsS0FBSyxDQUFDLGlCQUFpQixFQUFFLFdBQVc7Z0JBQ25DO29CQUNFLG1CQUFtQjtvQkFDbkIsK0RBQStEO2lCQUNoRSxDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFO29CQUNuQix5QkFBZSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxzQkFBc0IsRUFBRTt3QkFDbkU7NEJBQ0UsRUFBRSxFQUFFLE1BQU07NEJBQ1YsTUFBTSxFQUNKLGlGQUFpRjt5QkFDcEY7cUJBQ0YsQ0FBQyxDQUFDO2dCQUNMLENBQUMsQ0FBQyxDQUFDO1lBRUwsQ0FBQyxpQkFBaUIsRUFBRSxvQ0FBb0MsQ0FBQyxDQUFDLE9BQU8sQ0FDL0QsQ0FBQyxNQUFNLEVBQUUsRUFBRTtnQkFDVCx5QkFBZSxDQUFDLG9CQUFvQixDQUFDLEtBQUssRUFBRTtvQkFDMUM7d0JBQ0UsRUFBRSxFQUFFLE1BQU07d0JBQ1YsTUFBTSxFQUNKLDJHQUEyRztxQkFDOUc7aUJBQ0YsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUNGLENBQUM7WUFFRixDQUFDLG1CQUFtQixFQUFFLHlDQUF5QyxDQUFDLENBQUMsT0FBTyxDQUN0RSxDQUFDLE1BQU0sRUFBRSxFQUFFO2dCQUNULHlCQUFlLENBQUMsb0JBQW9CLENBQUMsS0FBSyxFQUFFO29CQUMxQzt3QkFDRSxFQUFFLEVBQUUsTUFBTTt3QkFDVixNQUFNLEVBQ0osNElBQTRJO3dCQUM5SSxTQUFTLEVBQUU7NEJBQ1Q7Z0NBQ0UsS0FBSyxFQUFFLG9CQUFvQjs2QkFDNUI7NEJBQ0Q7Z0NBQ0UsS0FBSyxFQUFFLG1CQUFtQjs2QkFDM0I7eUJBQ0Y7cUJBQ0Y7aUJBQ0YsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUNGLENBQUM7WUFFRixDQUFDLG1CQUFtQixFQUFFLHFDQUFxQyxDQUFDLENBQUMsT0FBTyxDQUNsRSxDQUFDLE1BQU0sRUFBRSxFQUFFO2dCQUNULHlCQUFlLENBQUMsb0JBQW9CLENBQUMsS0FBSyxFQUFFO29CQUMxQzt3QkFDRSxFQUFFLEVBQUUsTUFBTTt3QkFDVixNQUFNLEVBQ0osa0dBQWtHO3dCQUNwRyxTQUFTLEVBQUU7NEJBQ1Q7Z0NBQ0UsS0FBSyxFQUFFLGlCQUFpQixnQkFBTSxDQUFDLHNCQUFzQixDQUNuRCxLQUFLLENBQ04sOERBQThEOzZCQUNoRTt5QkFDRjtxQkFDRjtpQkFDRixDQUFDLENBQUM7WUFDTCxDQUFDLENBQ0YsQ0FBQztZQUVGLENBQUMsaUJBQWlCLEVBQUUsdUNBQXVDLENBQUMsQ0FBQyxPQUFPLENBQ2xFLENBQUMsTUFBTSxFQUFFLEVBQUU7Z0JBQ1QseUJBQWUsQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUU7b0JBQzFDO3dCQUNFLEVBQUUsRUFBRSxNQUFNO3dCQUNWLE1BQU0sRUFBRSxzREFBc0Q7cUJBQy9EO2lCQUNGLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FDRixDQUFDO1FBQ0osQ0FBQyxDQUFDO1FBeFBBLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNsQixpREFBaUQsRUFDakQsSUFBSSxDQUNMLENBQUM7UUFFRixJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRTFCLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxlQUFNLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQzVELFNBQVMsRUFBRSxLQUFLO1lBQ2hCLFVBQVUsRUFBRSxJQUFJO1lBQ2hCLGlCQUFpQixFQUFFLElBQUk7WUFDdkIsYUFBYSxFQUFFLDJCQUFhLENBQUMsT0FBTztZQUNwQyxVQUFVLEVBQUUseUJBQWdCLENBQUMsVUFBVTtZQUN2QyxlQUFlLEVBQUUsd0JBQWUsQ0FBQyxhQUFhO1lBQzlDLGdCQUFnQixFQUFFLEtBQUs7WUFDdkIsaUJBQWlCLEVBQUUsMEJBQWlCLENBQUMsU0FBUztTQUMvQyxDQUFDLENBQUM7UUFFSCxrQ0FBa0M7UUFDbEMsSUFBSSxDQUFDLGFBQWE7WUFDaEIsS0FBSyxDQUFDLGFBQWE7Z0JBQ25CLElBQUksZUFBTSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7b0JBQ2hDLFNBQVMsRUFBRSxJQUFJO29CQUNmLFVBQVUsRUFBRSxJQUFJO29CQUNoQixpQkFBaUIsRUFBRSxJQUFJO29CQUN2QixhQUFhLEVBQUUsMkJBQWEsQ0FBQyxPQUFPO29CQUNwQyxVQUFVLEVBQ1IsS0FBSyxDQUFDLDhCQUE4QixJQUFJLHlCQUFnQixDQUFDLFVBQVU7b0JBQ3JFLGVBQWUsRUFBRSx3QkFBZSxDQUFDLHFCQUFxQjtvQkFDdEQsYUFBYSxFQUFFLEtBQUssQ0FBQyxpQ0FBaUM7b0JBQ3RELGdCQUFnQixFQUFFLEtBQUs7b0JBQ3ZCLGlCQUFpQixFQUFFLDBCQUFpQixDQUFDLFNBQVM7b0JBQzlDLHNCQUFzQixFQUFFLHFCQUFxQjtvQkFDN0Msc0JBQXNCLEVBQUUsZ0JBQWdCO2lCQUN6QyxDQUFDLENBQUM7UUFFTCxVQUFVO1FBQ1YsTUFBTSxFQUFFLGlCQUFpQixFQUFFLEdBQUcsS0FBSyxDQUFDO1FBQ3BDLE1BQU0sU0FBUyxHQUNiLGlCQUFpQixFQUFFLFFBQVE7WUFDM0IsSUFBSSxxQ0FBZ0IsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFFeEUsMEJBQTBCO1FBQzFCLE1BQU0sU0FBUyxHQUNiLEtBQUssQ0FBQyxpQkFBaUIsRUFBRSxTQUFTO1lBQ2xDLElBQUksZUFBTSxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtnQkFDeEMsVUFBVSxFQUFFLElBQUk7Z0JBQ2hCLGlCQUFpQixFQUFFLElBQUk7Z0JBQ3ZCLGFBQWEsRUFBRSwyQkFBYSxDQUFDLE9BQU87Z0JBQ3BDLFVBQVUsRUFDUixLQUFLLENBQUMsOEJBQThCLElBQUkseUJBQWdCLENBQUMsVUFBVTtnQkFDckUsZUFBZSxFQUFFLHdCQUFlLENBQUMsc0JBQXNCO2dCQUN2RCxhQUFhLEVBQUUsS0FBSyxDQUFDLGlDQUFpQztnQkFDdEQsZ0JBQWdCLEVBQUUsS0FBSztnQkFDdkIsaUJBQWlCLEVBQUUsMEJBQWlCLENBQUMsU0FBUztnQkFDOUMsc0JBQXNCLEVBQUUsMEJBQTBCO2dCQUNsRCxzQkFBc0IsRUFBRSxnQkFBZ0I7YUFDekMsQ0FBQyxDQUFDO1FBRUwsTUFBTSxvQkFBb0IsR0FBRyxJQUFJLHFDQUFvQixDQUNuRCxJQUFJLEVBQ0osc0JBQXNCLENBQ3ZCLENBQUM7UUFDRixJQUFJLENBQUMsYUFBYSxDQUFDLG1CQUFtQixDQUNwQyxJQUFJLHlCQUFlLENBQUM7WUFDbEIsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUM7WUFDekMsT0FBTyxFQUFFLENBQUMsZUFBZSxDQUFDO1lBQzFCLFVBQVUsRUFBRSxDQUFDLG9CQUFvQixDQUFDLGNBQWMsQ0FBQztTQUNsRCxDQUFDLENBQ0gsQ0FBQztRQUVGLE1BQU0saUJBQWlCLEdBQ3JCLGlCQUFpQixFQUFFLGlCQUFpQixJQUFJLFlBQVksQ0FBQztRQUN2RCxJQUFJLENBQUMsc0JBQXNCLEdBQUcsSUFBSSw2QkFBWSxDQUM1QyxJQUFJLEVBQ0osd0JBQXdCLEVBQ3hCO1lBQ0UsR0FBRyxpQkFBaUI7WUFDcEIsUUFBUSxFQUFFLFNBQVM7WUFDbkIsYUFBYSxFQUFFLElBQUk7WUFDbkIsU0FBUyxFQUFFLFNBQVM7WUFDcEIsZUFBZSxFQUFFO2dCQUNmLEdBQUcsaUJBQWlCLEVBQUUsZUFBZTtnQkFDckMsTUFBTSxFQUFFLElBQUksaUNBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFO29CQUN2QyxvQkFBb0I7aUJBQ3JCLENBQUM7Z0JBQ0Ysb0JBQW9CLEVBQUUscUNBQW9CLENBQUMsaUJBQWlCO2FBQzdEO1lBQ0QsaUJBQWlCO1lBQ2pCLGNBQWMsRUFBRSxpQkFBaUIsRUFBRSxjQUFjLElBQUk7Z0JBQ25EO29CQUNFLFVBQVUsRUFBRSxHQUFHO29CQUNmLGtCQUFrQixFQUFFLEdBQUc7b0JBQ3ZCLGdCQUFnQixFQUFFLElBQUksaUJBQWlCLEVBQUU7aUJBQzFDO2FBQ0Y7U0FDRixDQUNGLENBQUM7UUFFRixpQkFBaUI7UUFDakIsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksb0NBQWdCLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQ3RFLE9BQU8sRUFBRTtnQkFDUCwwQkFBTSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUM7Z0JBQ3RDLEdBQUcsQ0FBQyxLQUFLLENBQUMsY0FBYztvQkFDdEIsQ0FBQyxDQUFDO3dCQUNFLDBCQUFNLENBQUMsUUFBUSxDQUNiLEtBQUssQ0FBQyxjQUFjLEVBQUUsWUFBWTs0QkFDaEMsK0JBQStCLEVBQ2pDLEtBQUssQ0FBQyxjQUFjLEVBQUUsV0FBVyxDQUNsQztxQkFDRjtvQkFDSCxDQUFDLENBQUMsRUFBRSxDQUFDO2FBQ1I7WUFDRCxpQkFBaUIsRUFBRSxJQUFJLENBQUMsYUFBYTtZQUNyQyxrSEFBa0g7WUFDbEgsWUFBWSxFQUFFLElBQUksQ0FBQyxzQkFBc0I7U0FDMUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHdCQUF3QixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7Ozs7QUEvSFUsc0NBQWE7QUFtUTFCOzs7R0FHRztBQUNILE1BQWEsbUJBQW1CO0lBQzlCLElBQUksQ0FBQyxNQUFpQixFQUFFLFFBQTJCO1FBQ2pELE1BQU0sSUFBSSxLQUFLLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNqRCxDQUFDOzs7O0FBSFUsa0RBQW1CIiwic291cmNlc0NvbnRlbnQiOlsiLyohIENvcHlyaWdodCBbQW1hem9uLmNvbV0oaHR0cDovL2FtYXpvbi5jb20vKSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wICovXG5pbXBvcnQgeyBQREtOYWcgfSBmcm9tIFwiQGF3cy1wcm90b3R5cGluZy1zZGsvcGRrLW5hZ1wiO1xuaW1wb3J0IHsgUmVtb3ZhbFBvbGljeSwgU3RhY2sgfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7XG4gIERpc3RyaWJ1dGlvbixcbiAgRGlzdHJpYnV0aW9uUHJvcHMsXG4gIElPcmlnaW4sXG4gIE9yaWdpbkFjY2Vzc0lkZW50aXR5LFxuICBPcmlnaW5CaW5kQ29uZmlnLFxuICBPcmlnaW5CaW5kT3B0aW9ucyxcbiAgVmlld2VyUHJvdG9jb2xQb2xpY3ksXG59IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtY2xvdWRmcm9udFwiO1xuaW1wb3J0IHsgUzNPcmlnaW4gfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWNsb3VkZnJvbnQtb3JpZ2luc1wiO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50IH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1pYW1cIjtcbmltcG9ydCB7IEtleSB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3Mta21zXCI7XG5pbXBvcnQge1xuICBCbG9ja1B1YmxpY0FjY2VzcyxcbiAgQnVja2V0LFxuICBCdWNrZXRFbmNyeXB0aW9uLFxuICBJQnVja2V0LFxuICBPYmplY3RPd25lcnNoaXAsXG59IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtczNcIjtcbmltcG9ydCB7IEJ1Y2tldERlcGxveW1lbnQsIFNvdXJjZSB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtczMtZGVwbG95bWVudFwiO1xuaW1wb3J0IHsgTmFnU3VwcHJlc3Npb25zIH0gZnJvbSBcImNkay1uYWdcIjtcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gXCJjb25zdHJ1Y3RzXCI7XG5pbXBvcnQgeyBDbG91ZGZyb250V2ViQWNsLCBDbG91ZEZyb250V2ViQWNsUHJvcHMgfSBmcm9tIFwiLi9jbG91ZGZyb250LXdlYi1hY2xcIjtcblxuY29uc3QgREVGQVVMVF9SVU5USU1FX0NPTkZJR19GSUxFTkFNRSA9IFwicnVudGltZS1jb25maWcuanNvblwiO1xuXG4vKipcbiAqIER5bmFtaWMgY29uZmlndXJhdGlvbiB3aGljaCBnZXRzIHJlc29sdmVkIG9ubHkgZHVyaW5nIGRlcGxveW1lbnQuXG4gKlxuICogQGV4YW1wbGVcbiAqXG4gKiAvLyBXaWxsIHN0b3JlIGEgSlNPTiBmaWxlIGNhbGxlZCBydW50aW1lLWNvbmZpZy5qc29uIGluIHRoZSByb290IG9mIHRoZSBTdGF0aWNXZWJzaXRlIFMzIGJ1Y2tldCBjb250YWluaW5nIGFueVxuICogLy8gYW5kIGFsbCByZXNvbHZlZCB2YWx1ZXMuXG4gKiBjb25zdCBydW50aW1lQ29uZmlnID0ge2pzb25QYXlsb2FkOiB7YnVja2V0QXJuOiBzM0J1Y2tldC5idWNrZXRBcm59fTtcbiAqIG5ldyBTdGF0aWNXZWJzaXRlKHNjb3BlLCAnU3RhdGljV2Vic2l0ZScsIHt3ZWJzaXRlQ29udGVudFBhdGg6ICdwYXRoL3RvL3dlYnNpdGUnLCBydW50aW1lQ29uZmlnfSk7XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUnVudGltZU9wdGlvbnMge1xuICAvKipcbiAgICogRmlsZSBuYW1lIHRvIHN0b3JlIHJ1bnRpbWUgY29uZmlndXJhdGlvbiAoanNvblBheWxvYWQpLlxuICAgKlxuICAgKiBNdXN0IGZvbGxvdyBwYXR0ZXJuOiAnKi5qc29uJ1xuICAgKlxuICAgKiBAZGVmYXVsdCBcInJ1bnRpbWUtY29uZmlnLmpzb25cIlxuICAgKi9cbiAgcmVhZG9ubHkganNvbkZpbGVOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBcmJpdHJhcnkgSlNPTiBwYXlsb2FkIGNvbnRhaW5pbmcgcnVudGltZSB2YWx1ZXMgdG8gZGVwbG95LiBUeXBpY2FsbHkgdGhpcyBjb250YWlucyByZXNvdXJjZUFybnMsIGV0YyB3aGljaFxuICAgKiBhcmUgb25seSBrbm93biBhdCBkZXBsb3kgdGltZS5cbiAgICpcbiAgICogQGV4YW1wbGUgeyB1c2VyUG9vbElkOiBzb21lLnVzZXJQb29sLnVzZXJQb29sSWQsIHNvbWVSZXNvdXJjZUFybjogc29tZS5yZXNvdXJjZS5Bcm4gfVxuICAgKi9cbiAgcmVhZG9ubHkganNvblBheWxvYWQ6IGFueTtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBjb25maWd1cmluZyB0aGUgU3RhdGljV2Vic2l0ZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTdGF0aWNXZWJzaXRlUHJvcHMge1xuICAvKipcbiAgICogUGF0aCB0byB0aGUgZGlyZWN0b3J5IGNvbnRhaW5pbmcgdGhlIHN0YXRpYyB3ZWJzaXRlIGZpbGVzIGFuZCBhc3NldHMuIFRoaXMgZGlyZWN0b3J5IG11c3QgY29udGFpbiBhbiBpbmRleC5odG1sIGZpbGUuXG4gICAqL1xuICByZWFkb25seSB3ZWJzaXRlQ29udGVudFBhdGg6IHN0cmluZztcblxuICAvKipcbiAgICogRHluYW1pYyBjb25maWd1cmF0aW9uIHdoaWNoIGdldHMgcmVzb2x2ZWQgb25seSBkdXJpbmcgZGVwbG95bWVudC5cbiAgICovXG4gIHJlYWRvbmx5IHJ1bnRpbWVPcHRpb25zPzogUnVudGltZU9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEJ1Y2tldCBlbmNyeXB0aW9uIHRvIHVzZSBmb3IgdGhlIGRlZmF1bHQgYnVja2V0LlxuICAgKlxuICAgKiBTdXBwb3J0ZWQgb3B0aW9ucyBhcmUgS01TIG9yIFMzTUFOQUdFRC5cbiAgICpcbiAgICogTm90ZTogSWYgcGxhbm5pbmcgdG8gdXNlIEtNUywgZW5zdXJlIHlvdSBhc3NvY2lhdGUgYSBMYW1iZGEgRWRnZSBmdW5jdGlvbiB0byBzaWduIHJlcXVlc3RzIHRvIFMzIGFzIE9BSSBkb2VzIG5vdCBjdXJyZW50bHkgc3VwcG9ydCBLTVMgZW5jcnlwdGlvbi4gUmVmZXIgdG8ge0BsaW5rIGh0dHBzOi8vYXdzLmFtYXpvbi5jb20vYmxvZ3MvbmV0d29ya2luZy1hbmQtY29udGVudC1kZWxpdmVyeS9zZXJ2aW5nLXNzZS1rbXMtZW5jcnlwdGVkLWNvbnRlbnQtZnJvbS1zMy11c2luZy1jbG91ZGZyb250L31cbiAgICpcbiAgICogQGRlZmF1bHQgLSBcIlMzTUFOQUdFRFwiXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0V2Vic2l0ZUJ1Y2tldEVuY3J5cHRpb24/OiBCdWNrZXRFbmNyeXB0aW9uO1xuXG4gIC8qKlxuICAgKiBBIHByZWRlZmluZWQgS01TIGN1c3RvbWVyIGVuY3J5cHRpb24ga2V5IHRvIHVzZSBmb3IgdGhlIGRlZmF1bHQgYnVja2V0IHRoYXQgZ2V0cyBjcmVhdGVkLlxuICAgKlxuICAgKiBOb3RlOiBUaGlzIGlzIG9ubHkgdXNlZCBpZiB0aGUgd2Vic2l0ZUJ1Y2tldCBpcyBsZWZ0IHVuZGVmaW5lZCwgb3RoZXJ3aXNlIGFsbCBzZXR0aW5ncyBmcm9tIHRoZSBwcm92aWRlZCB3ZWJzaXRlQnVja2V0IHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRXZWJzaXRlQnVja2V0RW5jcnlwdGlvbktleT86IEtleTtcblxuICAvKipcbiAgICogUHJlZGVmaW5lZCBidWNrZXQgdG8gZGVwbG95IHRoZSB3ZWJzaXRlIGludG8uXG4gICAqL1xuICByZWFkb25seSB3ZWJzaXRlQnVja2V0PzogSUJ1Y2tldDtcblxuICAvKipcbiAgICogQ3VzdG9tIGRpc3RyaWJ1dGlvbiBwcm9wZXJ0aWVzLlxuICAgKlxuICAgKiBOb3RlOiBkZWZhdWx0QmVoYXZpb3VyLm9yaWdpbiBpcyBhIHJlcXVpcmVkIHBhcmFtZXRlciwgaG93ZXZlciBpdCB3aWxsIG5vdCBiZSB1c2VkIGFzIHRoaXMgY29uc3RydWN0IHdpbGwgd2lyZSBpdCBvbiB5b3VyIGJlaGFsZi5cbiAgICogWW91IHdpbGwgbmVlZCB0byBwYXNzIGluIGFuIGluc3RhbmNlIG9mIFN0YXRpY1dlYnNpdGVPcmlnaW4gKE5vT3ApIHRvIGtlZXAgdGhlIGNvbXBpbGVyIGhhcHB5LlxuICAgKi9cbiAgcmVhZG9ubHkgZGlzdHJpYnV0aW9uUHJvcHM/OiBEaXN0cmlidXRpb25Qcm9wcztcblxuICAvKipcbiAgICogTGltaXRlZCBjb25maWd1cmF0aW9uIHNldHRpbmdzIGZvciB0aGUgZ2VuZXJhdGVkIHdlYkFjbC4gRm9yIG1vcmUgYWR2YW5jZWQgc2V0dGluZ3MsIGNyZWF0ZSB5b3VyIG93biBBQ0wgYW5kIHBhc3MgaW4gdGhlIHdlYkFjbElkIGFzIGEgcGFyYW0gdG8gZGlzdHJpYnV0aW9uUHJvcHMuXG4gICAqXG4gICAqIE5vdGU6IElmIHBhc3MgaW4geW91ciBvd24gQUNMLCBtYWtlIHN1cmUgdGhlIFNDT1BFIGlzIENMT1VERlJPTlQgYW5kIGl0IGlzIGNyZWF0ZWQgaW4gdXMtZWFzdC0xLlxuICAgKi9cbiAgcmVhZG9ubHkgd2ViQWNsUHJvcHM/OiBDbG91ZEZyb250V2ViQWNsUHJvcHM7XG59XG5cbi8qKlxuICogRGVwbG95cyBhIFN0YXRpYyBXZWJzaXRlIHVzaW5nIGJ5IGRlZmF1bHQgYSBwcml2YXRlIFMzIGJ1Y2tldCBhcyBhbiBvcmlnaW4gYW5kIENsb3VkZnJvbnQgYXMgdGhlIGVudHJ5cG9pbnQuXG4gKlxuICogVGhpcyBjb25zdHJ1Y3QgY29uZmlndXJlcyBhIHdlYkFjbCBjb250YWluaW5nIHJ1bGVzIHRoYXQgYXJlIGdlbmVyYWxseSBhcHBsaWNhYmxlIHRvIHdlYiBhcHBsaWNhdGlvbnMuIFRoaXNcbiAqIHByb3ZpZGVzIHByb3RlY3Rpb24gYWdhaW5zdCBleHBsb2l0YXRpb24gb2YgYSB3aWRlIHJhbmdlIG9mIHZ1bG5lcmFiaWxpdGllcywgaW5jbHVkaW5nIHNvbWUgb2YgdGhlIGhpZ2ggcmlza1xuICogYW5kIGNvbW1vbmx5IG9jY3VycmluZyB2dWxuZXJhYmlsaXRpZXMgZGVzY3JpYmVkIGluIE9XQVNQIHB1YmxpY2F0aW9ucyBzdWNoIGFzIE9XQVNQIFRvcCAxMC5cbiAqXG4gKi9cbmV4cG9ydCBjbGFzcyBTdGF0aWNXZWJzaXRlIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgcHVibGljIHJlYWRvbmx5IHdlYnNpdGVCdWNrZXQ6IElCdWNrZXQ7XG4gIHB1YmxpYyByZWFkb25seSBjbG91ZEZyb250RGlzdHJpYnV0aW9uOiBEaXN0cmlidXRpb247XG4gIHB1YmxpYyByZWFkb25seSBidWNrZXREZXBsb3ltZW50OiBCdWNrZXREZXBsb3ltZW50O1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTdGF0aWNXZWJzaXRlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5ub2RlLnNldENvbnRleHQoXG4gICAgICBcIkBhd3MtY2RrL2F3cy1zMzpzZXJ2ZXJBY2Nlc3NMb2dzVXNlQnVja2V0UG9saWN5XCIsXG4gICAgICB0cnVlXG4gICAgKTtcblxuICAgIHRoaXMudmFsaWRhdGVQcm9wcyhwcm9wcyk7XG5cbiAgICBjb25zdCBhY2Nlc3NMb2dzQnVja2V0ID0gbmV3IEJ1Y2tldCh0aGlzLCBcIkFjY2Vzc0xvZ3NCdWNrZXRcIiwge1xuICAgICAgdmVyc2lvbmVkOiBmYWxzZSxcbiAgICAgIGVuZm9yY2VTU0w6IHRydWUsXG4gICAgICBhdXRvRGVsZXRlT2JqZWN0czogdHJ1ZSxcbiAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIGVuY3J5cHRpb246IEJ1Y2tldEVuY3J5cHRpb24uUzNfTUFOQUdFRCxcbiAgICAgIG9iamVjdE93bmVyc2hpcDogT2JqZWN0T3duZXJzaGlwLk9CSkVDVF9XUklURVIsXG4gICAgICBwdWJsaWNSZWFkQWNjZXNzOiBmYWxzZSxcbiAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBCbG9ja1B1YmxpY0FjY2Vzcy5CTE9DS19BTEwsXG4gICAgfSk7XG5cbiAgICAvLyBTMyBCdWNrZXQgdG8gaG9sZCB3ZWJzaXRlIGZpbGVzXG4gICAgdGhpcy53ZWJzaXRlQnVja2V0ID1cbiAgICAgIHByb3BzLndlYnNpdGVCdWNrZXQgPz9cbiAgICAgIG5ldyBCdWNrZXQodGhpcywgXCJXZWJzaXRlQnVja2V0XCIsIHtcbiAgICAgICAgdmVyc2lvbmVkOiB0cnVlLFxuICAgICAgICBlbmZvcmNlU1NMOiB0cnVlLFxuICAgICAgICBhdXRvRGVsZXRlT2JqZWN0czogdHJ1ZSxcbiAgICAgICAgcmVtb3ZhbFBvbGljeTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgICAgICBlbmNyeXB0aW9uOlxuICAgICAgICAgIHByb3BzLmRlZmF1bHRXZWJzaXRlQnVja2V0RW5jcnlwdGlvbiA/PyBCdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQsXG4gICAgICAgIG9iamVjdE93bmVyc2hpcDogT2JqZWN0T3duZXJzaGlwLkJVQ0tFVF9PV05FUl9FTkZPUkNFRCxcbiAgICAgICAgZW5jcnlwdGlvbktleTogcHJvcHMuZGVmYXVsdFdlYnNpdGVCdWNrZXRFbmNyeXB0aW9uS2V5LFxuICAgICAgICBwdWJsaWNSZWFkQWNjZXNzOiBmYWxzZSxcbiAgICAgICAgYmxvY2tQdWJsaWNBY2Nlc3M6IEJsb2NrUHVibGljQWNjZXNzLkJMT0NLX0FMTCxcbiAgICAgICAgc2VydmVyQWNjZXNzTG9nc1ByZWZpeDogXCJ3ZWJzaXRlLWFjY2Vzcy1sb2dzXCIsXG4gICAgICAgIHNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQ6IGFjY2Vzc0xvZ3NCdWNrZXQsXG4gICAgICB9KTtcblxuICAgIC8vIFdlYiBBQ0xcbiAgICBjb25zdCB7IGRpc3RyaWJ1dGlvblByb3BzIH0gPSBwcm9wcztcbiAgICBjb25zdCB3ZWJBY2xBcm4gPVxuICAgICAgZGlzdHJpYnV0aW9uUHJvcHM/LndlYkFjbElkID8/XG4gICAgICBuZXcgQ2xvdWRmcm9udFdlYkFjbCh0aGlzLCBcIldlYnNpdGVBY2xcIiwgcHJvcHMud2ViQWNsUHJvcHMpLndlYkFjbEFybjtcblxuICAgIC8vIENsb3VkZnJvbnQgRGlzdHJpYnV0aW9uXG4gICAgY29uc3QgbG9nQnVja2V0ID1cbiAgICAgIHByb3BzLmRpc3RyaWJ1dGlvblByb3BzPy5sb2dCdWNrZXQgfHxcbiAgICAgIG5ldyBCdWNrZXQodGhpcywgXCJEaXN0cmlidXRpb25Mb2dCdWNrZXRcIiwge1xuICAgICAgICBlbmZvcmNlU1NMOiB0cnVlLFxuICAgICAgICBhdXRvRGVsZXRlT2JqZWN0czogdHJ1ZSxcbiAgICAgICAgcmVtb3ZhbFBvbGljeTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgICAgICBlbmNyeXB0aW9uOlxuICAgICAgICAgIHByb3BzLmRlZmF1bHRXZWJzaXRlQnVja2V0RW5jcnlwdGlvbiA/PyBCdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQsXG4gICAgICAgIG9iamVjdE93bmVyc2hpcDogT2JqZWN0T3duZXJzaGlwLkJVQ0tFVF9PV05FUl9QUkVGRVJSRUQsXG4gICAgICAgIGVuY3J5cHRpb25LZXk6IHByb3BzLmRlZmF1bHRXZWJzaXRlQnVja2V0RW5jcnlwdGlvbktleSxcbiAgICAgICAgcHVibGljUmVhZEFjY2VzczogZmFsc2UsXG4gICAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBCbG9ja1B1YmxpY0FjY2Vzcy5CTE9DS19BTEwsXG4gICAgICAgIHNlcnZlckFjY2Vzc0xvZ3NQcmVmaXg6IFwiZGlzdHJpYnV0aW9uLWFjY2Vzcy1sb2dzXCIsXG4gICAgICAgIHNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQ6IGFjY2Vzc0xvZ3NCdWNrZXQsXG4gICAgICB9KTtcblxuICAgIGNvbnN0IG9yaWdpbkFjY2Vzc0lkZW50aXR5ID0gbmV3IE9yaWdpbkFjY2Vzc0lkZW50aXR5KFxuICAgICAgdGhpcyxcbiAgICAgIFwiT3JpZ2luQWNjZXNzSWRlbnRpdHlcIlxuICAgICk7XG4gICAgdGhpcy53ZWJzaXRlQnVja2V0LmFkZFRvUmVzb3VyY2VQb2xpY3koXG4gICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy53ZWJzaXRlQnVja2V0LmJ1Y2tldEFybl0sXG4gICAgICAgIGFjdGlvbnM6IFtcInMzOkxpc3RCdWNrZXRcIl0sXG4gICAgICAgIHByaW5jaXBhbHM6IFtvcmlnaW5BY2Nlc3NJZGVudGl0eS5ncmFudFByaW5jaXBhbF0sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBjb25zdCBkZWZhdWx0Um9vdE9iamVjdCA9XG4gICAgICBkaXN0cmlidXRpb25Qcm9wcz8uZGVmYXVsdFJvb3RPYmplY3QgPz8gXCJpbmRleC5odG1sXCI7XG4gICAgdGhpcy5jbG91ZEZyb250RGlzdHJpYnV0aW9uID0gbmV3IERpc3RyaWJ1dGlvbihcbiAgICAgIHRoaXMsXG4gICAgICBcIkNsb3VkZnJvbnREaXN0cmlidXRpb25cIixcbiAgICAgIHtcbiAgICAgICAgLi4uZGlzdHJpYnV0aW9uUHJvcHMsXG4gICAgICAgIHdlYkFjbElkOiB3ZWJBY2xBcm4sXG4gICAgICAgIGVuYWJsZUxvZ2dpbmc6IHRydWUsXG4gICAgICAgIGxvZ0J1Y2tldDogbG9nQnVja2V0LFxuICAgICAgICBkZWZhdWx0QmVoYXZpb3I6IHtcbiAgICAgICAgICAuLi5kaXN0cmlidXRpb25Qcm9wcz8uZGVmYXVsdEJlaGF2aW9yLFxuICAgICAgICAgIG9yaWdpbjogbmV3IFMzT3JpZ2luKHRoaXMud2Vic2l0ZUJ1Y2tldCwge1xuICAgICAgICAgICAgb3JpZ2luQWNjZXNzSWRlbnRpdHksXG4gICAgICAgICAgfSksXG4gICAgICAgICAgdmlld2VyUHJvdG9jb2xQb2xpY3k6IFZpZXdlclByb3RvY29sUG9saWN5LlJFRElSRUNUX1RPX0hUVFBTLFxuICAgICAgICB9LFxuICAgICAgICBkZWZhdWx0Um9vdE9iamVjdCxcbiAgICAgICAgZXJyb3JSZXNwb25zZXM6IGRpc3RyaWJ1dGlvblByb3BzPy5lcnJvclJlc3BvbnNlcyA/PyBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgaHR0cFN0YXR1czogNDA0LCAvLyBXZSBuZWVkIHRvIHJlZGlyZWN0IFwia2V5IG5vdCBmb3VuZCBlcnJvcnNcIiB0byBpbmRleC5odG1sIGZvciBzaW5nbGUgcGFnZSBhcHBzXG4gICAgICAgICAgICByZXNwb25zZUh0dHBTdGF0dXM6IDIwMCxcbiAgICAgICAgICAgIHJlc3BvbnNlUGFnZVBhdGg6IGAvJHtkZWZhdWx0Um9vdE9iamVjdH1gLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9XG4gICAgKTtcblxuICAgIC8vIERlcGxveSBXZWJzaXRlXG4gICAgdGhpcy5idWNrZXREZXBsb3ltZW50ID0gbmV3IEJ1Y2tldERlcGxveW1lbnQodGhpcywgXCJXZWJzaXRlRGVwbG95bWVudFwiLCB7XG4gICAgICBzb3VyY2VzOiBbXG4gICAgICAgIFNvdXJjZS5hc3NldChwcm9wcy53ZWJzaXRlQ29udGVudFBhdGgpLFxuICAgICAgICAuLi4ocHJvcHMucnVudGltZU9wdGlvbnNcbiAgICAgICAgICA/IFtcbiAgICAgICAgICAgICAgU291cmNlLmpzb25EYXRhKFxuICAgICAgICAgICAgICAgIHByb3BzLnJ1bnRpbWVPcHRpb25zPy5qc29uRmlsZU5hbWUgfHxcbiAgICAgICAgICAgICAgICAgIERFRkFVTFRfUlVOVElNRV9DT05GSUdfRklMRU5BTUUsXG4gICAgICAgICAgICAgICAgcHJvcHMucnVudGltZU9wdGlvbnM/Lmpzb25QYXlsb2FkXG4gICAgICAgICAgICAgICksXG4gICAgICAgICAgICBdXG4gICAgICAgICAgOiBbXSksXG4gICAgICBdLFxuICAgICAgZGVzdGluYXRpb25CdWNrZXQ6IHRoaXMud2Vic2l0ZUJ1Y2tldCxcbiAgICAgIC8vIEZpbGVzIGluIHRoZSBkaXN0cmlidXRpb24ncyBlZGdlIGNhY2hlcyB3aWxsIGJlIGludmFsaWRhdGVkIGFmdGVyIGZpbGVzIGFyZSB1cGxvYWRlZCB0byB0aGUgZGVzdGluYXRpb24gYnVja2V0LlxuICAgICAgZGlzdHJpYnV0aW9uOiB0aGlzLmNsb3VkRnJvbnREaXN0cmlidXRpb24sXG4gICAgfSk7XG5cbiAgICB0aGlzLnN1cHByZXNzQ0RLTmFnVmlvbGF0aW9ucyhwcm9wcyk7XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlUHJvcHMgPSAocHJvcHM6IFN0YXRpY1dlYnNpdGVQcm9wcykgPT4ge1xuICAgIHRoaXMudmFsaWRhdGVFbmNyeXB0aW9uU2V0dGluZ3MocHJvcHMpO1xuICAgIHByb3BzLnJ1bnRpbWVPcHRpb25zICYmIHRoaXMudmFsaWRhdGVSdW50aW1lQ29uZmlnKHByb3BzLnJ1bnRpbWVPcHRpb25zKTtcbiAgICBwcm9wcy53ZWJzaXRlQnVja2V0ICYmIHRoaXMudmFsaWRhdGVCdWNrZXRDb25maWcocHJvcHMud2Vic2l0ZUJ1Y2tldCk7XG4gIH07XG5cbiAgcHJpdmF0ZSB2YWxpZGF0ZVJ1bnRpbWVDb25maWcgPSAoY29uZmlnOiBSdW50aW1lT3B0aW9ucykgPT4ge1xuICAgIGlmICghY29uZmlnKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwidmFsaWRhdGVSdW50aW1lQ29uZmlnIG9ubHkgYWNjZXB0cyBub24tbnVsbCBSdW50aW1lT3B0aW9ucy5cIlxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAoY29uZmlnLmpzb25GaWxlTmFtZSAmJiAhY29uZmlnLmpzb25GaWxlTmFtZS5lbmRzV2l0aChcIi5qc29uXCIpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXCJSdW50aW1lT3B0aW9ucy5qc29uRmlsZU5hbWUgbXVzdCBiZSBhIGpzb24gZmlsZS5cIik7XG4gICAgfVxuICB9O1xuXG4gIHByaXZhdGUgdmFsaWRhdGVCdWNrZXRDb25maWcgPSAoYnVja2V0OiBJQnVja2V0KSA9PiB7XG4gICAgaWYgKGJ1Y2tldC5pc1dlYnNpdGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJXZWJzaXRlIGJ1Y2tldHMgY2Fubm90IGJlIGNvbmZpZ3VyZWQgYXMgd2Vic2l0ZXMgYXMgdGhpcyB3aWxsIGJyZWFrIENsb3VkZnJvbnQgaG9zdGluZyFcIlxuICAgICAgKTtcbiAgICB9XG4gIH07XG5cbiAgcHJpdmF0ZSB2YWxpZGF0ZUVuY3J5cHRpb25TZXR0aW5ncyA9ICh7XG4gICAgZGVmYXVsdFdlYnNpdGVCdWNrZXRFbmNyeXB0aW9uLFxuICAgIGRlZmF1bHRXZWJzaXRlQnVja2V0RW5jcnlwdGlvbktleSxcbiAgfTogU3RhdGljV2Vic2l0ZVByb3BzKSA9PiB7XG4gICAgaWYgKFxuICAgICAgZGVmYXVsdFdlYnNpdGVCdWNrZXRFbmNyeXB0aW9uS2V5ICYmXG4gICAgICBkZWZhdWx0V2Vic2l0ZUJ1Y2tldEVuY3J5cHRpb24gIT09IEJ1Y2tldEVuY3J5cHRpb24uS01TXG4gICAgKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiQnVja2V0IGVuY3J5cHRpb24gc2hvdWxkIGJlIHNldCB0byBLTVMgaWYgcHJvdmlkaW5nIGEgZGVmYXVsdFdlYnNpdGVCdWNrZXRFbmNyeXB0aW9uS2V5LlwiXG4gICAgICApO1xuICAgIH1cblxuICAgIGlmIChcbiAgICAgIGRlZmF1bHRXZWJzaXRlQnVja2V0RW5jcnlwdGlvbiAmJlxuICAgICAgZGVmYXVsdFdlYnNpdGVCdWNrZXRFbmNyeXB0aW9uICE9PSBCdWNrZXRFbmNyeXB0aW9uLktNUyAmJlxuICAgICAgZGVmYXVsdFdlYnNpdGVCdWNrZXRFbmNyeXB0aW9uICE9PSBCdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRURcbiAgICApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJPbmx5IEtNUyBhbmQgUzNfTUFOQUdFRCBlbmNyeXB0aW9uIGFyZSBzdXBwb3J0ZWQgb24gdGhlIGRlZmF1bHQgYnVja2V0LlwiXG4gICAgICApO1xuICAgIH1cbiAgfTtcblxuICBwcml2YXRlIHN1cHByZXNzQ0RLTmFnVmlvbGF0aW9ucyA9IChwcm9wczogU3RhdGljV2Vic2l0ZVByb3BzKSA9PiB7XG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcbiAgICAhcHJvcHMuZGlzdHJpYnV0aW9uUHJvcHM/LmNlcnRpZmljYXRlICYmXG4gICAgICBbXG4gICAgICAgIFwiQXdzU29sdXRpb25zLUNGUjRcIixcbiAgICAgICAgXCJBd3NQcm90b3R5cGluZy1DbG91ZEZyb250RGlzdHJpYnV0aW9uSHR0cHNWaWV3ZXJOb091dGRhdGVkU1NMXCIsXG4gICAgICBdLmZvckVhY2goKFJ1bGVJZCkgPT4ge1xuICAgICAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnModGhpcy5jbG91ZEZyb250RGlzdHJpYnV0aW9uLCBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgaWQ6IFJ1bGVJZCxcbiAgICAgICAgICAgIHJlYXNvbjpcbiAgICAgICAgICAgICAgXCJDZXJ0aWZpY2F0ZSBpcyBub3QgbWFuZGF0b3J5IHRoZXJlZm9yZSB0aGUgQ2xvdWRmcm9udCBjZXJ0aWZpY2F0ZSB3aWxsIGJlIHVzZWQuXCIsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSk7XG4gICAgICB9KTtcblxuICAgIFtcIkF3c1NvbHV0aW9ucy1MMVwiLCBcIkF3c1Byb3RvdHlwaW5nLUxhbWJkYUxhdGVzdFZlcnNpb25cIl0uZm9yRWFjaChcbiAgICAgIChSdWxlSWQpID0+IHtcbiAgICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFN0YWNrU3VwcHJlc3Npb25zKHN0YWNrLCBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgaWQ6IFJ1bGVJZCxcbiAgICAgICAgICAgIHJlYXNvbjpcbiAgICAgICAgICAgICAgXCJMYXRlc3QgcnVudGltZSBjYW5ub3QgYmUgY29uZmlndXJlZC4gQ0RLIHdpbGwgbmVlZCB0byB1cGdyYWRlIHRoZSBCdWNrZXREZXBsb3ltZW50IGNvbnN0cnVjdCBhY2NvcmRpbmdseS5cIixcbiAgICAgICAgICB9LFxuICAgICAgICBdKTtcbiAgICAgIH1cbiAgICApO1xuXG4gICAgW1wiQXdzU29sdXRpb25zLUlBTTVcIiwgXCJBd3NQcm90b3R5cGluZy1JQU1Ob1dpbGRjYXJkUGVybWlzc2lvbnNcIl0uZm9yRWFjaChcbiAgICAgIChSdWxlSWQpID0+IHtcbiAgICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFN0YWNrU3VwcHJlc3Npb25zKHN0YWNrLCBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgaWQ6IFJ1bGVJZCxcbiAgICAgICAgICAgIHJlYXNvbjpcbiAgICAgICAgICAgICAgXCJBbGwgUG9saWNpZXMgaGF2ZSBiZWVuIHNjb3BlZCB0byBhIEJ1Y2tldC4gR2l2ZW4gQnVja2V0cyBjYW4gY29udGFpbiBhcmJpdHJhcnkgY29udGVudCwgd2lsZGNhcmQgcmVzb3VyY2VzIHdpdGggYnVja2V0IHNjb3BlIGFyZSByZXF1aXJlZC5cIixcbiAgICAgICAgICAgIGFwcGxpZXNUbzogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgcmVnZXg6IFwiL15BY3Rpb246OnMzOi4qJC9nXCIsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICByZWdleDogYC9eUmVzb3VyY2U6Oi4qJC9nYCxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSk7XG4gICAgICB9XG4gICAgKTtcblxuICAgIFtcIkF3c1NvbHV0aW9ucy1JQU00XCIsIFwiQXdzUHJvdG90eXBpbmctSUFNTm9NYW5hZ2VkUG9saWNpZXNcIl0uZm9yRWFjaChcbiAgICAgIChSdWxlSWQpID0+IHtcbiAgICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFN0YWNrU3VwcHJlc3Npb25zKHN0YWNrLCBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgaWQ6IFJ1bGVJZCxcbiAgICAgICAgICAgIHJlYXNvbjpcbiAgICAgICAgICAgICAgXCJCdWNrZXRzIGNhbiBjb250YWluIGFyYml0cmFyeSBjb250ZW50LCB0aGVyZWZvcmUgd2lsZGNhcmQgcmVzb3VyY2VzIHVuZGVyIGEgYnVja2V0IGFyZSByZXF1aXJlZC5cIixcbiAgICAgICAgICAgIGFwcGxpZXNUbzogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgcmVnZXg6IGAvXlBvbGljeTo6YXJuOiR7UERLTmFnLmdldFN0YWNrUGFydGl0aW9uUmVnZXgoXG4gICAgICAgICAgICAgICAgICBzdGFja1xuICAgICAgICAgICAgICAgICl9OmlhbTo6YXdzOnBvbGljeS9zZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlJC9nYCxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSk7XG4gICAgICB9XG4gICAgKTtcblxuICAgIFtcIkF3c1NvbHV0aW9ucy1TMVwiLCBcIkF3c1Byb3RvdHlwaW5nLVMzQnVja2V0TG9nZ2luZ0VuYWJsZWRcIl0uZm9yRWFjaChcbiAgICAgIChSdWxlSWQpID0+IHtcbiAgICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFN0YWNrU3VwcHJlc3Npb25zKHN0YWNrLCBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgaWQ6IFJ1bGVJZCxcbiAgICAgICAgICAgIHJlYXNvbjogXCJBY2Nlc3MgTG9nIGJ1Y2tldHMgc2hvdWxkIG5vdCBoYXZlIHMzIGJ1Y2tldCBsb2dnaW5nXCIsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSk7XG4gICAgICB9XG4gICAgKTtcbiAgfTtcbn1cblxuLyoqXG4gKiBJZiBwYXNzaW5nIGluIGRpc3RyaWJ1dGlvblByb3BzLCB0aGUgZGVmYXVsdCBiZWhhdmlvdXIub3JpZ2luIGlzIGEgcmVxdWlyZWQgcGFyYW1ldGVyLiBBbiBpbnN0YW5jZSBvZiB0aGlzIGNsYXNzIGNhbiBiZSBwYXNzZWQgaW5cbiAqIHRvIG1ha2UgdGhlIGNvbXBpbGVyIGhhcHB5LlxuICovXG5leHBvcnQgY2xhc3MgU3RhdGljV2Vic2l0ZU9yaWdpbiBpbXBsZW1lbnRzIElPcmlnaW4ge1xuICBiaW5kKF9zY29wZTogQ29uc3RydWN0LCBfb3B0aW9uczogT3JpZ2luQmluZE9wdGlvbnMpOiBPcmlnaW5CaW5kQ29uZmlnIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXCJUaGlzIHNob3VsZCBuZXZlciBiZSBjYWxsZWRcIik7XG4gIH1cbn1cbiJdfQ==