"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const typescript_1 = require("projen/lib/typescript");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: false,
            jest: false,
            package: false,
            prettier: true,
            projenrcTs: true,
            release: false,
            sampleCode: false,
            defaultReleaseBranch: "mainline",
        });
        this.implicitDependencies = {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(options.nxIgnorePatterns || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(options.targetDependencies || {}),
                },
                affected: {
                    defaultBase: "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    // Remove this hack once subProjects is made public in Projen
    get subProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    preSynthesize() {
        super.preSynthesize();
        // Add workspaces for each subproject
        this.package.addField("workspaces", this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)));
        // Disable default task on subprojects as this isn't supported in a monorepo
        this.subProjects.forEach((subProject) => { var _b; return (_b = subProject.defaultTask) === null || _b === void 0 ? void 0 : _b.reset(); });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Check to see if a new subProject was added
        const newSubProject = this.subProjects.find((subProject) => !fs.existsSync(subProject.outdir));
        // Need to synth before generating the package.json otherwise the subdirectory won't exist
        newSubProject && super.synth();
        this.subProjects
            .filter((subProject) => !subProject.tryFindObjectFile("package.json") ||
            (fs.existsSync(`${subProject.outdir}/package.json`) &&
                JSON.parse(fs.readFileSync(`${subProject.outdir}/package.json`).toString()).__pdk__))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            manifest.name = subProject.name;
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
        super.synth();
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.pdk_projen.NxMonorepoProject", version: "0.0.65" };
function getPluginPath() {
    const prefix = process.env.NODE_ENV === "test"
        ? "./lib/" // If unit test from within this package
        : "./node_modules/aws-prototyping-sdk/lib/";
    return `${prefix}pdk_projen/nx_monorepo/plugin/nx-monorepo-plugin.js`;
}
//# sourceMappingURL=data:application/json;base64,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