"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoAttributeValue = exports.DynamoProjectionExpression = exports.DynamoReturnValues = exports.DynamoItemCollectionMetrics = exports.DynamoConsumedCapacity = void 0;
const utils_1 = require("./private/utils");
/**
 * Determines the level of detail about provisioned throughput consumption that is returned.
 *
 * @stability stable
 */
var DynamoConsumedCapacity;
(function (DynamoConsumedCapacity) {
    DynamoConsumedCapacity["INDEXES"] = "INDEXES";
    DynamoConsumedCapacity["TOTAL"] = "TOTAL";
    DynamoConsumedCapacity["NONE"] = "NONE";
})(DynamoConsumedCapacity = exports.DynamoConsumedCapacity || (exports.DynamoConsumedCapacity = {}));
/**
 * Determines whether item collection metrics are returned.
 *
 * @stability stable
 */
var DynamoItemCollectionMetrics;
(function (DynamoItemCollectionMetrics) {
    DynamoItemCollectionMetrics["SIZE"] = "SIZE";
    DynamoItemCollectionMetrics["NONE"] = "NONE";
})(DynamoItemCollectionMetrics = exports.DynamoItemCollectionMetrics || (exports.DynamoItemCollectionMetrics = {}));
/**
 * Use ReturnValues if you want to get the item attributes as they appear before or after they are changed.
 *
 * @stability stable
 */
var DynamoReturnValues;
(function (DynamoReturnValues) {
    DynamoReturnValues["NONE"] = "NONE";
    DynamoReturnValues["ALL_OLD"] = "ALL_OLD";
    DynamoReturnValues["UPDATED_OLD"] = "UPDATED_OLD";
    DynamoReturnValues["ALL_NEW"] = "ALL_NEW";
    DynamoReturnValues["UPDATED_NEW"] = "UPDATED_NEW";
})(DynamoReturnValues = exports.DynamoReturnValues || (exports.DynamoReturnValues = {}));
/**
 * Class to generate projection expression.
 *
 * @stability stable
 */
class DynamoProjectionExpression {
    constructor() {
        this.expression = [];
    }
    /**
     * Adds the passed attribute to the chain.
     *
     * @param attr Attribute name.
     * @stability stable
     */
    withAttribute(attr) {
        if (this.expression.length) {
            this.expression.push(`.${attr}`);
        }
        else {
            this.expression.push(attr);
        }
        return this;
    }
    /**
     * Adds the array literal access for passed index.
     *
     * @param index array index.
     * @stability stable
     */
    atIndex(index) {
        if (!this.expression.length) {
            throw new Error('Expression must start with an attribute');
        }
        this.expression.push(`[${index}]`);
        return this;
    }
    /**
     * converts and return the string expression.
     *
     * @stability stable
     */
    toString() {
        return this.expression.join('');
    }
}
exports.DynamoProjectionExpression = DynamoProjectionExpression;
/**
 * Represents the data for an attribute.
 *
 * Each attribute value is described as a name-value pair.
 * The name is the data type, and the value is the data itself.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_AttributeValue.html
 * @stability stable
 */
class DynamoAttributeValue {
    constructor(value) {
        this.attributeValue = value;
    }
    /**
     * Sets an attribute of type String.
     *
     * For example:  "S": "Hello"
     * Strings may be literal values or as JsonPath
     *
     * @stability stable
     */
    static fromString(value) {
        return new DynamoAttributeValue({ S: value });
    }
    /**
     * Sets a literal number.
     *
     * For example: 1234
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * @stability stable
     */
    static fromNumber(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * Sets an attribute of type Number.
     *
     * For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     *
     * @stability stable
     */
    static numberFromString(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * Sets an attribute of type Binary.
     *
     * For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     *
     * @param value base-64 encoded string.
     * @stability stable
     */
    static fromBinary(value) {
        return new DynamoAttributeValue({ B: value });
    }
    /**
     * Sets an attribute of type String Set.
     *
     * For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     *
     * @stability stable
     */
    static fromStringSet(value) {
        return new DynamoAttributeValue({ SS: value });
    }
    /**
     * Sets an attribute of type Number Set.
     *
     * For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * @stability stable
     */
    static fromNumberSet(value) {
        return new DynamoAttributeValue({ NS: value.map(String) });
    }
    /**
     * Sets an attribute of type Number Set.
     *
     * For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     *
     * @stability stable
     */
    static numberSetFromStrings(value) {
        return new DynamoAttributeValue({ NS: value });
    }
    /**
     * Sets an attribute of type Binary Set.
     *
     * For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     *
     * @stability stable
     */
    static fromBinarySet(value) {
        return new DynamoAttributeValue({ BS: value });
    }
    /**
     * Sets an attribute of type Map.
     *
     * For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     *
     * @stability stable
     */
    static fromMap(value) {
        return new DynamoAttributeValue({ M: utils_1.transformAttributeValueMap(value) });
    }
    /**
     * Sets an attribute of type Map.
     *
     * For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     *
     * @param value Json path that specifies state input to be used.
     * @stability stable
     */
    static mapFromJsonPath(value) {
        utils_1.validateJsonPath(value);
        return new DynamoAttributeValue({ 'M.$': value });
    }
    /**
     * Sets an attribute of type List.
     *
     * For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     *
     * @stability stable
     */
    static fromList(value) {
        return new DynamoAttributeValue({ L: value.map((val) => val.toObject()) });
    }
    /**
     * Sets an attribute of type Null.
     *
     * For example:  "NULL": true
     *
     * @stability stable
     */
    static fromNull(value) {
        return new DynamoAttributeValue({ NULL: value });
    }
    /**
     * Sets an attribute of type Boolean.
     *
     * For example:  "BOOL": true
     *
     * @stability stable
     */
    static fromBoolean(value) {
        return new DynamoAttributeValue({ BOOL: value });
    }
    /**
     * Sets an attribute of type Boolean from state input through Json path.
     *
     * For example:  "BOOL": true
     *
     * @param value Json path that specifies state input to be used.
     * @stability stable
     */
    static booleanFromJsonPath(value) {
        utils_1.validateJsonPath(value);
        return new DynamoAttributeValue({ BOOL: value.toString() });
    }
    /**
     * Returns the DynamoDB attribute value.
     *
     * @stability stable
     */
    toObject() {
        return this.attributeValue;
    }
}
exports.DynamoAttributeValue = DynamoAttributeValue;
//# sourceMappingURL=data:application/json;base64,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