"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateEndpointConfig = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * (experimental) A Step Functions Task to create a SageMaker endpoint configuration.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html
 * @experimental
 */
class SageMakerCreateEndpointConfig extends sfn.TaskStateBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateEndpointConfig.SUPPORTED_INTEGRATION_PATTERNS);
        this.validateProductionVariants();
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createEndpointConfig', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        var _a, _b;
        return {
            EndpointConfigName: this.props.endpointConfigName,
            Tags: (_a = this.props.tags) === null || _a === void 0 ? void 0 : _a.value,
            KmsKeyId: (_b = this.props.kmsKey) === null || _b === void 0 ? void 0 : _b.keyId,
            ProductionVariants: this.props.productionVariants.map((variant) => ({
                InitialInstanceCount: variant.initialInstanceCount ? variant.initialInstanceCount : 1,
                InstanceType: sfn.JsonPath.isEncodedJsonPath(variant.instanceType.toString())
                    ? variant.instanceType.toString() : `ml.${variant.instanceType}`,
                ModelName: variant.modelName,
                VariantName: variant.variantName,
                AcceleratorType: variant.acceleratorType,
                InitialVariantWeight: variant.initialVariantWeight,
            })),
        };
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        // https://docs.aws.amazon.com/sagemaker/latest/dg/api-permissions-reference.html
        return [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateEndpointConfig'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'endpoint-config',
                        // If the endpoint configuration name comes from input, we cannot target the policy to a particular ARN prefix reliably.
                        // SageMaker uses lowercase for resource name in the arn
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.endpointConfigName) ? '*' : `${this.props.endpointConfigName.toLowerCase()}`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
                resources: ['*'],
            }),
        ];
    }
    validateProductionVariants() {
        if (this.props.productionVariants.length < 1 || this.props.productionVariants.length > 10) {
            throw new Error('Must specify from 1 to 10 production variants per endpoint configuration');
        }
        this.props.productionVariants.forEach((variant) => {
            if (variant.initialInstanceCount && variant.initialInstanceCount < 1)
                throw new Error('Must define at least one instance');
            if (variant.initialVariantWeight && variant.initialVariantWeight <= 0) {
                throw new Error('InitialVariantWeight has minimum value of 0');
            }
        });
    }
}
exports.SageMakerCreateEndpointConfig = SageMakerCreateEndpointConfig;
SageMakerCreateEndpointConfig.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
];
//# sourceMappingURL=data:application/json;base64,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