"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvocationType = exports.LambdaInvoke = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Invoke a Lambda function as a Task.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 * @stability stable
 */
class LambdaInvoke extends sfn.TaskStateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is required in `payload` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (props.payloadResponseOnly &&
            (props.integrationPattern || props.invocationType || props.clientContext || props.qualifier)) {
            throw new Error("The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified.");
        }
        this.taskMetrics = {
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: {
                LambdaFunctionArn: this.props.lambdaFunction.functionArn,
                ...(this.props.qualifier && { Qualifier: this.props.qualifier }),
            },
        };
        this.taskPolicies = [
            new iam.PolicyStatement({
                resources: [this.props.lambdaFunction.functionArn],
                actions: ['lambda:InvokeFunction'],
            }),
        ];
        if ((_b = props.retryOnServiceExceptions) !== null && _b !== void 0 ? _b : true) {
            // Best practice from https://docs.aws.amazon.com/step-functions/latest/dg/bp-lambda-serviceexception.html
            this.addRetry({
                errors: ['Lambda.ServiceException', 'Lambda.AWSLambdaException', 'Lambda.SdkClientException'],
                interval: cdk.Duration.seconds(2),
                maxAttempts: 6,
                backoffRate: 2,
            });
        }
    }
    /**
     * Provides the Lambda Invoke service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        if (this.props.payloadResponseOnly) {
            return {
                Resource: this.props.lambdaFunction.functionArn,
                ...this.props.payload && { Parameters: sfn.FieldUtils.renderObject(this.props.payload.value) },
            };
        }
        else {
            return {
                Resource: task_utils_1.integrationResourceArn('lambda', 'invoke', this.integrationPattern),
                Parameters: sfn.FieldUtils.renderObject({
                    FunctionName: this.props.lambdaFunction.functionArn,
                    Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                    InvocationType: this.props.invocationType,
                    ClientContext: this.props.clientContext,
                    Qualifier: this.props.qualifier,
                }),
            };
        }
    }
}
exports.LambdaInvoke = LambdaInvoke;
LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
/**
 * Invocation type of a Lambda.
 *
 * @stability stable
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    LambdaInvocationType["EVENT"] = "Event";
    LambdaInvocationType["DRY_RUN"] = "DryRun";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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