"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const secretsmanager = require("../lib");
module.exports = {
    'default secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {},
        }));
        test.done();
    },
    'set removalPolicy to secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
            DeletionPolicy: 'Retain',
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'secret with kms'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        // WHEN
        new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {},
                    {
                        Effect: 'Allow',
                        Resource: '*',
                        Action: [
                            'kms:Decrypt',
                            'kms:Encrypt',
                            'kms:ReEncrypt*',
                            'kms:GenerateDataKey*',
                        ],
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Condition: {
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'secretsmanager.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                            },
                        },
                    },
                    {
                        Effect: 'Allow',
                        Resource: '*',
                        Action: [
                            'kms:CreateGrant',
                            'kms:DescribeKey',
                        ],
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Condition: {
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'secretsmanager.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                            },
                        },
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'secret with generate secret string options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                excludeUppercase: true,
                passwordLength: 20,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                ExcludeUppercase: true,
                PasswordLength: 20,
            },
        }));
        test.done();
    },
    'templated secret string'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                secretStringTemplate: JSON.stringify({ username: 'username' }),
                generateStringKey: 'password',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                SecretStringTemplate: '{"username":"username"}',
                GenerateStringKey: 'password',
            },
        }));
        test.done();
    },
    'grantRead'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                    }],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {},
                    {},
                    {},
                    {
                        Action: 'kms:Decrypt',
                        Condition: {
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'secretsmanager.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::GetAtt': [
                                    'Role1ABCC5F0',
                                    'Arn',
                                ],
                            },
                        },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'grantRead with version label constraint'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantRead(role, ['FOO', 'bar']);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                        Condition: {
                            'ForAnyValue:StringEquals': {
                                'secretsmanager:VersionStage': ['FOO', 'bar'],
                            },
                        },
                    }],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {},
                    {},
                    {},
                    {
                        Action: 'kms:Decrypt',
                        Condition: {
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'secretsmanager.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::GetAtt': [
                                    'Role1ABCC5F0',
                                    'Arn',
                                ],
                            },
                        },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'grantWrite'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret', {});
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantWrite(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'secretsmanager:PutSecretValue',
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                    }],
            },
        }));
        test.done();
    },
    'grantWrite with kms'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantWrite(role);
        // THEN
        const expectStack = assert_1.expect(stack);
        expectStack.to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'secretsmanager:PutSecretValue',
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                    }],
            },
        }));
        expectStack.to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {},
                    {},
                    {},
                    {
                        Action: [
                            'kms:Encrypt',
                            'kms:ReEncrypt*',
                            'kms:GenerateDataKey*',
                        ],
                        Condition: {
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'secretsmanager.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::GetAtt': [
                                    'Role1ABCC5F0',
                                    'Arn',
                                ],
                            },
                        },
                        Resource: '*',
                    },
                ],
            },
        }));
        test.done();
    },
    'secretValue'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        // WHEN
        new cdk.CfnResource(stack, 'FakeResource', {
            type: 'CDK::Phony::Resource',
            properties: {
                value: secret.secretValue,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('CDK::Phony::Resource', {
            value: {
                'Fn::Join': ['', [
                        '{{resolve:secretsmanager:',
                        { Ref: 'SecretA720EF05' },
                        ':SecretString:::}}',
                    ]],
            },
        }));
        test.done();
    },
    'import'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const encryptionKey = new kms.Key(stack, 'KMS');
        const secretArn = 'arn::of::a::secret';
        // WHEN
        const secret = secretsmanager.Secret.fromSecretAttributes(stack, 'Secret', {
            secretArn, encryptionKey,
        });
        // THEN
        test.equals(secret.secretArn, secretArn);
        test.same(secret.encryptionKey, encryptionKey);
        test.deepEqual(stack.resolve(secret.secretValue), '{{resolve:secretsmanager:arn::of::a::secret:SecretString:::}}');
        test.deepEqual(stack.resolve(secret.secretValueFromJson('password')), '{{resolve:secretsmanager:arn::of::a::secret:SecretString:password::}}');
        test.done();
    },
    'can attach a secret with attach()'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        // WHEN
        secret.attach({
            asSecretAttachmentTarget: () => ({
                targetId: 'target-id',
                targetType: 'target-type',
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::SecretTargetAttachment', {
            SecretId: {
                Ref: 'SecretA720EF05',
            },
            TargetId: 'target-id',
            TargetType: 'target-type',
        }));
        test.done();
    },
    'throws when trying to attach a target multiple times to a secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const target = {
            asSecretAttachmentTarget: () => ({
                targetId: 'target-id',
                targetType: 'target-type',
            }),
        };
        secret.attach(target);
        // THEN
        test.throws(() => secret.attach(target), /Secret is already attached to a target/);
        test.done();
    },
    'add a rotation schedule to an attached secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const attachedSecret = secret.attach({
            asSecretAttachmentTarget: () => ({
                targetId: 'target-id',
                targetType: 'target-type',
            }),
        });
        const rotationLambda = new lambda.Function(stack, 'Lambda', {
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('export.handler = event => event;'),
            handler: 'index.handler',
        });
        // WHEN
        attachedSecret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::RotationSchedule', {
            SecretId: {
                Ref: 'SecretAttachment2E1B7C3B',
            },
        }));
        test.done();
    },
    'throws when specifying secretStringTemplate but not generateStringKey'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                secretStringTemplate: JSON.stringify({ username: 'username' }),
            },
        }), /`secretStringTemplate`.+`generateStringKey`/);
        test.done();
    },
    'throws when specifying generateStringKey but not secretStringTemplate'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                generateStringKey: 'password',
            },
        }), /`secretStringTemplate`.+`generateStringKey`/);
        test.done();
    },
    'equivalence of SecretValue and Secret.fromSecretAttributes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const imported = secretsmanager.Secret.fromSecretAttributes(stack, 'Imported', { secretArn: 'my-secret-arn' }).secretValueFromJson('password');
        const value = cdk.SecretValue.secretsManager('my-secret-arn', { jsonField: 'password' });
        // THEN
        test.deepEqual(stack.resolve(imported), stack.resolve(value));
        test.done();
    },
    'can add to the resource policy of a secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        // WHEN
        secret.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:GetSecretValue'],
            resources: ['*'],
            principals: [new iam.ArnPrincipal('arn:aws:iam::123456789012:user/cool-user')],
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::ResourcePolicy', {
            ResourcePolicy: {
                Statement: [
                    {
                        Action: 'secretsmanager:GetSecretValue',
                        Effect: 'Allow',
                        Principal: {
                            AWS: 'arn:aws:iam::123456789012:user/cool-user',
                        },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
            SecretId: {
                Ref: 'SecretA720EF05',
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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