<!--
 ~ Copyright DB InfraGO AG and contributors
 ~ SPDX-License-Identifier: Apache-2.0
 -->

# polarion-rest-api-client

<!-- prettier-ignore -->
![image](https://github.com/dbinfrago/polarion-rest-api-client/actions/workflows/build-test-publish.yml/badge.svg)
![image](https://github.com/dbinfrago/polarion-rest-api-client/actions/workflows/lint.yml/badge.svg)

A client library for accessing Polarion REST API. This project consists of multiple layers. There is a high level, still incomplete
version of the client and a feature complete low level API client, which was generated using an [OpenAPI generator](https://github.com/openapi-generators/openapi-python-client).
Therefore the OpenAPI Specification of Polarion was used.

## Usage of the High Level Client
The high level client is an abstraction of the fine-grained, auto-generated client. It is non project specific, however, for most operations, you need to create a project specific one from the generic one. All created created project clients share the client session of the generic client.
To get started, create a client and check, if the project exists. In the end, to get all Work Items of a project with an empty query, you can simply run the following code and our client will automatically take care of the paging:

```python
import polarion_rest_api_client as polarion_api

client = polarion_api.PolarionClient(
   polarion_api_endpoint="http://127.0.0.1/api",
   polarion_access_token="PAT123",
)
project_client = client.generate_project_client("PROJ")
project_exists = project_client.exists() # Should be True
work_items = project_client.work_items.get_all()
```
During the initialization of the client you can define additional settings like the page size when getting items or the maximum content size when bulk creating new items.
In addition, you can define your own Work Item class with custom fields, which become available as attributes on object level instead of being part of the `additional_attributes` dictionary only.
To use this feature, inherit from our Work Item class and pass your extended class when requesting Work Items:
```python
import polarion_rest_api_client as polarion_api
import dataclasses

@dataclasses.dataclass
class CustomWorkItem(polarion_api.WorkItem):
   capella_uuid: str | None = None

client = polarion_api.PolarionClient(
   polarion_api_endpoint="http://127.0.0.1/api",
   polarion_access_token="PAT123",
)
project_client = client.generate_project_client("PROJ")
work_items = project_client.work_items.get_all(work_item_cls=CustomWorkItem)
uuid = work_items[0].capella_uuid # the value of the custom field capella_uuid can be accessed this way
```
## Usage of the autogenerated OpenAPI Client
First, create a client:

```python
from polarion_rest_api_client.open_api_client import Client

client = Client(base_url="https://api.example.com")
```

If the endpoints you're going to hit require authentication, use `AuthenticatedClient` instead:

```python
from polarion_rest_api_client.open_api_client import AuthenticatedClient

client = AuthenticatedClient(base_url="https://api.example.com", token="SuperSecretToken")
```

Now call your endpoint and use your models:

```python
from polarion_rest_api_client.open_api_client.models import MyDataModel
from polarion_rest_api_client.open_api_client.api.my_tag import get_my_data_model
from polarion_rest_api_client.open_api_client.types import Response

my_data: MyDataModel = get_my_data_model.sync(client=client)
# or if you need more info (e.g. status_code)
response: Response[MyDataModel] = get_my_data_model.sync_detailed(client=client)
```

Or do the same thing with an async version:

```python
from polarion_rest_api_client.open_api_client.models import MyDataModel
from polarion_rest_api_client.open_api_client.api.my_tag import get_my_data_model
from polarion_rest_api_client.open_api_client.types import Response

my_data: MyDataModel = await get_my_data_model.asyncio(client=client)
response: Response[MyDataModel] = await get_my_data_model.asyncio_detailed(client=client)
```

By default, when you're calling an HTTPS API it will attempt to verify that SSL is working correctly. Using certificate verification is highly recommended most of the time, but sometimes you may need to authenticate to a server (especially an internal server) using a custom certificate bundle.

```python
client = AuthenticatedClient(
    base_url="https://internal_api.example.com",
    token="SuperSecretToken",
    verify_ssl="/path/to/certificate_bundle.pem",
)
```

You can also disable certificate validation altogether, but beware that **this is a security risk**.

```python
client = AuthenticatedClient(
    base_url="https://internal_api.example.com",
    token="SuperSecretToken",
    verify_ssl=False
)
```

There are more settings on the generated `Client` class which let you control more runtime behavior, check out the docstring on that class for more info.

Things to know:
1. Every path/method combo becomes a Python module with four functions:
    1. `sync`: Blocking request that returns parsed data (if successful) or `None`
    1. `sync_detailed`: Blocking request that always returns a `Request`, optionally with `parsed` set if the request was successful.
    1. `asyncio`: Like `sync` but async instead of blocking
    1. `asyncio_detailed`: Like `sync_detailed` but async instead of blocking

1. All path/query params, and bodies become method arguments.
1. If your endpoint had any tags on it, the first tag will be used as a module name for the function (my_tag above)
1. Any endpoint which did not have a tag will be in `polarion_rest_api_client.open_api_client.api.default`

# Documentation

<!-- prettier-ignore -->
Read the [full documentation on Github pages](https://dbinfrago.github.io/polarion-rest-api-client).

# Installation

You can install the latest released version directly from [PyPI](https://pypi.org/project/polarion-rest-api-client/).

```zsh
pip install polarion-rest-api-client
```

To set up a development environment, clone the project and install it into a
virtual environment.

```zsh
git clone https://github.com/dbinfrago/polarion-rest-api-client
cd polarion-rest-api-client
python -m venv .venv

source .venv/bin/activate.sh  # for Linux / Mac
.venv\Scripts\activate  # for Windows

pip install -U pip pre-commit
pip install -e '.[docs,test]'
pre-commit install
```

## Updating the auto generated part
To update the auto generated part of the code, execute the `open_api_client_build/build_client.py` script with `path` or `url` as first
arg and the path to the OpenAPI-Specification as second arg from the project root directory. The publicly available [specification](https://developer.siemens.com/polarion/polarion-rest-apispec.json)
from the Polarion developer Portal was used.

Please note that as of Polarion 24.10 there is a mistake in the specification. In the relationships of a work items the type of approvals
has to be "workrecords" according to the specification, but in fact it is "workitem_approvals". For that reason we added a fixed version of
the specification to this repository and the issue will be reported to SIEMENS.

# Contributing

We'd love to see your bug reports and improvement suggestions! Please take a
look at our [guidelines for contributors](CONTRIBUTING.md) for details.

# Licenses

This project is compliant with the
[REUSE Specification Version 3.0](https://git.fsfe.org/reuse/docs/src/commit/d173a27231a36e1a2a3af07421f5e557ae0fec46/spec.md).

Copyright DB InfraGO AG, licensed under Apache 2.0 (see full text in
[LICENSES/Apache-2.0.txt](LICENSES/Apache-2.0.txt))

Dot-files are licensed under CC0-1.0 (see full text in
[LICENSES/CC0-1.0.txt](LICENSES/CC0-1.0.txt))
