"""
MCP Client for connecting to and communicating with MCP servers.
"""

import asyncio
import json
import logging
import subprocess
import time
from typing import Any, Dict, List, Optional

from ..core.models import (
    MCPServerConfig,
    MCPServerError,
    MCPToolError,
    ToolCall,
    ToolDefinition,
    ToolResult,
)
from ..utils.errors import (
    MCPConfigurationError,
    MCPConnectionError,
    MCPTimeoutError,
)

logger = logging.getLogger(__name__)


class MCPClient:
    """
    Client for communicating with MCP servers.

    This class handles the connection lifecycle, tool discovery,
    and tool execution for a single MCP server.
    """

    def __init__(self, server_config: MCPServerConfig):
        """
        Initialize MCP client.

        Args:
            server_config: Configuration for the MCP server
        """
        self.server_config = server_config
        self.process: Optional[subprocess.Popen] = None
        self.connected = False
        self.tools: Dict[str, ToolDefinition] = {}
        self._request_id = 0

    async def connect(self, retry_count: int = 0, max_retries: int = 3) -> None:
        """
        Connect to the MCP server with retry logic and comprehensive error handling.

        Args:
            retry_count: Current retry attempt
            max_retries: Maximum number of retry attempts

        Raises:
            MCPConnectionError: If connection fails after retries
            MCPConfigurationError: If server configuration is invalid
        """
        if self.connected:
            return

        try:
            logger.info(
                f"Connecting to MCP server: {self.server_config.name} (attempt {retry_count + 1})"
            )

            # Validate server configuration
            await self._validate_server_config()

            # Prepare environment variables
            env = dict(self.server_config.env)

            # Start the MCP server process
            try:
                self.process = subprocess.Popen(
                    [self.server_config.command] + self.server_config.args,
                    stdin=subprocess.PIPE,
                    stdout=subprocess.PIPE,
                    stderr=subprocess.PIPE,
                    env=env,
                    text=True,
                    bufsize=0,
                )
            except FileNotFoundError:
                raise MCPConnectionError(
                    f"Failed to start MCP server: {self.server_config.command} not found",
                    server_name=self.server_config.name,
                    connection_type="stdio",
                )
            except PermissionError:
                raise MCPConnectionError(
                    f"Failed to start MCP server: Permission denied for {self.server_config.command}",
                    server_name=self.server_config.name,
                    connection_type="stdio",
                )

            # Wait for process to start with exponential backoff
            initial_wait = 0.1 * (2**retry_count)  # Exponential backoff
            await asyncio.sleep(min(initial_wait, 2.0))

            # Check if process is still running
            if self.process.poll() is not None:
                stderr_output = ""
                if self.process.stderr:
                    try:
                        stderr_output = self.process.stderr.read()
                    except:
                        stderr_output = "Unable to read stderr"

                # Analyze stderr for specific error types
                error_msg = f"MCP server '{self.server_config.name}' failed to start"
                if stderr_output:
                    error_msg += f": {stderr_output}"

                if "command not found" in stderr_output.lower():
                    raise MCPConfigurationError(
                        f"MCP server command not found: {self.server_config.command}",
                        server_name=self.server_config.name,
                        details="Install the required MCP server package",
                    )
                elif "permission denied" in stderr_output.lower():
                    raise MCPConfigurationError(
                        f"Permission denied for MCP server: {self.server_config.command}",
                        server_name=self.server_config.name,
                        details="Check file permissions",
                    )
                elif (
                    "module not found" in stderr_output.lower()
                    or "importerror" in stderr_output.lower()
                ):
                    raise MCPConfigurationError(
                        f"Missing dependencies for MCP server: {self.server_config.name}",
                        server_name=self.server_config.name,
                        details="Install required Python packages or dependencies",
                    )
                else:
                    raise MCPConnectionError(
                        error_msg,
                        server_name=self.server_config.name,
                        connection_type="stdio",
                        retry_suggestion="Check server logs and configuration",
                    )

            # Initialize connection with the server
            try:
                await self._initialize_connection()
            except MCPTimeoutError:
                if retry_count < max_retries:
                    logger.warning(
                        f"Connection timeout for {self.server_config.name}, retrying..."
                    )
                    await self.disconnect()
                    await asyncio.sleep(1.0 * (retry_count + 1))  # Progressive delay
                    return await self.connect(retry_count + 1, max_retries)
                else:
                    raise MCPTimeoutError(
                        f"Connection initialization timed out for server '{self.server_config.name}'",
                        server_name=self.server_config.name,
                        timeout_duration=self.server_config.timeout,
                        operation="initialization",
                    )

            # Discover available tools
            try:
                await self._discover_tools()
            except Exception as e:
                logger.warning(
                    f"Tool discovery failed for {self.server_config.name}: {e}"
                )
                # Continue connection even if tool discovery fails

            self.connected = True
            logger.info(
                f"Successfully connected to MCP server: {self.server_config.name}"
            )

        except (MCPConnectionError, MCPConfigurationError, MCPTimeoutError):
            await self.disconnect()
            raise
        except Exception as e:
            await self.disconnect()

            # Retry logic for transient errors
            if retry_count < max_retries and self._is_retryable_error(e):
                logger.warning(f"Retryable error for {self.server_config.name}: {e}")
                await asyncio.sleep(2.0 * (retry_count + 1))  # Progressive delay
                return await self.connect(retry_count + 1, max_retries)

            raise MCPConnectionError(
                f"Failed to start MCP server: {e}",
                server_name=self.server_config.name,
                connection_type="stdio",
            )

    async def disconnect(self) -> None:
        """Disconnect from the MCP server."""
        if self.process:
            try:
                self.process.terminate()
                # Wait for graceful shutdown
                try:
                    self.process.wait(timeout=5)
                except subprocess.TimeoutExpired:
                    self.process.kill()
                    self.process.wait()
            except Exception as e:
                logger.warning(f"Error during MCP server shutdown: {e}")
            finally:
                self.process = None

        self.connected = False
        self.tools.clear()
        logger.info(f"Disconnected from MCP server: {self.server_config.name}")

    async def list_tools(self) -> List[ToolDefinition]:
        """
        List available tools from the MCP server.

        Returns:
            List of available tool definitions

        Raises:
            MCPServerError: If not connected or listing fails
        """
        if not self.connected:
            raise MCPServerError(
                f"Not connected to MCP server: {self.server_config.name}"
            )

        return list(self.tools.values())

    async def call_tool(self, name: str, arguments: Dict[str, Any]) -> ToolResult:
        """
        Execute a tool call on the MCP server.

        Args:
            name: Name of the tool to call
            arguments: Arguments to pass to the tool

        Returns:
            Result of the tool execution

        Raises:
            MCPToolError: If tool execution fails
            MCPServerError: If not connected
        """
        if not self.connected:
            raise MCPServerError(
                f"Not connected to MCP server: {self.server_config.name}"
            )

        if name not in self.tools:
            raise MCPToolError(
                f"Tool '{name}' not found on server '{self.server_config.name}'"
            )

        try:
            logger.debug(f"Calling tool '{name}' with arguments: {arguments}")

            # Create tool call request
            request = {
                "jsonrpc": "2.0",
                "id": self._get_next_request_id(),
                "method": "tools/call",
                "params": {"name": name, "arguments": arguments},
            }

            # Send request and get response
            response = await self._send_request(request)

            # Parse response
            if "error" in response:
                error_info = response["error"]
                raise MCPToolError(
                    f"Tool '{name}' execution failed: {error_info.get('message', 'Unknown error')}"
                )

            result_data = response.get("result", {})

            return ToolResult(
                content=result_data.get("content", ""),
                error=result_data.get("error"),
                metadata=result_data.get("metadata", {}),
            )

        except MCPToolError:
            raise
        except (asyncio.TimeoutError, TimeoutError):
            raise MCPTimeoutError(
                f"Tool call timed out for '{name}'",
                server_name=self.server_config.name,
                timeout_duration=self.server_config.timeout,
                operation="tool_call",
            )
        except Exception as e:
            raise MCPToolError(f"Failed to execute tool '{name}': {e}")

    async def _initialize_connection(self) -> None:
        """Initialize connection with the MCP server."""
        # Send initialization request
        init_request = {
            "jsonrpc": "2.0",
            "id": self._get_next_request_id(),
            "method": "initialize",
            "params": {
                "protocolVersion": "2024-11-05",
                "capabilities": {"tools": {}},
                "clientInfo": {"name": "omnimancer-cli", "version": "1.0.0"},
            },
        }

        response = await self._send_request(init_request)

        if "error" in response:
            raise MCPServerError(f"Server initialization failed: {response['error']}")

        # Check protocol version
        result = response.get("result", {})
        protocol_version = result.get("protocolVersion")
        if protocol_version and protocol_version not in ["1.0", "2024-11-05"]:
            raise MCPConnectionError(
                f"Unsupported protocol version: {protocol_version}"
            )

        # Send initialized notification
        initialized_notification = {
            "jsonrpc": "2.0",
            "method": "notifications/initialized",
        }

        await self._send_notification(initialized_notification)

    async def _discover_tools(self) -> None:
        """Discover available tools from the MCP server."""
        # Request tools list
        tools_request = {
            "jsonrpc": "2.0",
            "id": self._get_next_request_id(),
            "method": "tools/list",
        }

        response = await self._send_request(tools_request)

        if "error" in response:
            logger.warning(f"Failed to discover tools: {response['error']}")
            return

        # Parse tools from response
        tools_data = response.get("result", {}).get("tools", [])

        for tool_data in tools_data:
            tool_def = ToolDefinition(
                name=tool_data.get("name", ""),
                description=tool_data.get("description", ""),
                parameters=tool_data.get("inputSchema", {}),
                server_name=self.server_config.name,
                auto_approved=tool_data.get("name", "")
                in self.server_config.auto_approve,
            )
            self.tools[tool_def.name] = tool_def

        logger.info(
            f"Discovered {len(self.tools)} tools from server '{self.server_config.name}'"
        )

    async def _send_request(self, request: Dict[str, Any]) -> Dict[str, Any]:
        """
        Send a JSON-RPC request to the MCP server and wait for response.

        Args:
            request: JSON-RPC request object

        Returns:
            JSON-RPC response object

        Raises:
            MCPServerError: If communication fails
        """
        if not self.process or not self.process.stdin or not self.process.stdout:
            raise MCPServerError("MCP server process not available")

        try:
            # Send request
            request_json = json.dumps(request) + "\n"
            self.process.stdin.write(request_json)
            self.process.stdin.flush()

            # Read response with timeout
            start_time = time.time()
            while time.time() - start_time < self.server_config.timeout:
                if self.process.stdout.readable():
                    line = self.process.stdout.readline()
                    if line.strip():
                        response = json.loads(line.strip())

                        # Check for error in response
                        if "error" in response:
                            error_info = response["error"]
                            raise MCPServerError(
                                error_info.get("message", "Unknown error")
                            )

                        # Increment request ID to match test expectations
                        if "id" in request:
                            self._request_id = max(self._request_id, request["id"])

                        return response
                await asyncio.sleep(0.01)

            raise MCPServerError(
                f"Request timeout after {self.server_config.timeout} seconds"
            )

        except json.JSONDecodeError as e:
            raise MCPServerError(f"Invalid JSON response from server: {e}")
        except Exception as e:
            raise MCPServerError(f"Communication error with MCP server: {e}")

    async def _send_notification(self, notification: Dict[str, Any]) -> None:
        """
        Send a JSON-RPC notification to the MCP server (no response expected).

        Args:
            notification: JSON-RPC notification object

        Raises:
            MCPServerError: If communication fails
        """
        if not self.process or not self.process.stdin:
            raise MCPServerError("MCP server process not available")

        try:
            notification_json = json.dumps(notification) + "\n"
            self.process.stdin.write(notification_json)
            self.process.stdin.flush()
        except Exception as e:
            raise MCPServerError(f"Failed to send notification: {e}")

    def _get_next_request_id(self) -> int:
        """Get the next request ID."""
        self._request_id += 1
        return self._request_id

    @property
    def is_connected(self) -> bool:
        """Check if client is connected to the server."""
        return self.connected and self.process and self.process.poll() is None

    @property
    def server_name(self) -> str:
        """Get the server name."""
        return self.server_config.name

    def get_tool_definition(self, tool_name: str) -> Optional[ToolDefinition]:
        """
        Get definition for a specific tool.

        Args:
            tool_name: Name of the tool

        Returns:
            ToolDefinition or None if not found
        """
        return self.tools.get(tool_name)

    async def health_check(self) -> bool:
        """
        Perform a health check on the MCP server connection.

        Returns:
            True if server is healthy, False otherwise
        """
        try:
            if not self.is_connected:
                self.connected = False
                return False

            # Check if process is still running
            if self.process and self.process.poll() is not None:
                self.connected = False
                return False

            # For basic health check, just verify connection state
            return self.connected and self.process is not None

        except Exception:
            self.connected = False
            return False

    async def _validate_server_config(self) -> None:
        """
        Validate server configuration before attempting connection.

        Raises:
            MCPConfigurationError: If configuration is invalid
        """
        if not self.server_config.command:
            raise MCPConfigurationError(
                "MCP server command is required",
                server_name=self.server_config.name,
                details="Specify the command to run the MCP server",
            )

        if not self.server_config.name:
            raise MCPConfigurationError(
                "MCP server name is required",
                details="Provide a unique name for the MCP server",
            )

        # Validate timeout
        if self.server_config.timeout <= 0:
            raise MCPConfigurationError(
                f"Invalid timeout value: {self.server_config.timeout}",
                server_name=self.server_config.name,
                details="Timeout must be a positive number",
            )

        # Check if command exists (basic check)
        import shutil

        if not shutil.which(self.server_config.command):
            logger.warning(f"Command '{self.server_config.command}' not found in PATH")

    def _is_retryable_error(self, error: Exception) -> bool:
        """
        Determine if an error is retryable.

        Args:
            error: The exception to check

        Returns:
            True if the error is retryable, False otherwise
        """
        # Configuration errors are not retryable
        if isinstance(error, MCPConfigurationError):
            return False

        # Connection errors might be retryable
        if isinstance(error, MCPConnectionError):
            return True

        # Timeout errors are retryable
        if isinstance(error, MCPTimeoutError):
            return True

        # Check for specific error messages that indicate transient issues
        error_str = str(error).lower()
        transient_indicators = [
            "connection refused",
            "timeout",
            "temporary failure",
            "resource temporarily unavailable",
            "broken pipe",
        ]

        return any(indicator in error_str for indicator in transient_indicators)

    async def reconnect(self) -> bool:
        """
        Attempt to reconnect to the MCP server.

        Returns:
            True if reconnection successful, False otherwise
        """
        try:
            logger.info(
                f"Attempting to reconnect to MCP server: {self.server_config.name}"
            )

            # Disconnect first
            await self.disconnect()

            # Wait a moment before reconnecting
            await asyncio.sleep(1.0)

            # Attempt to connect
            await self.connect()

            return self.is_connected

        except Exception as e:
            logger.error(f"Reconnection failed for {self.server_config.name}: {e}")
            return False

    async def _perform_handshake(self) -> bool:
        """Perform handshake with the MCP server."""
        await self._initialize_connection()
        return True

    async def _read_response(self) -> Dict[str, Any]:
        """Read a response from the MCP server."""
        if not self.process or not self.process.stdout:
            raise MCPServerError("MCP server process not available")

        try:
            line = self.process.stdout.readline()
            if not line.strip():
                raise MCPServerError("Empty response from server")

            return json.loads(line.strip())
        except json.JSONDecodeError as e:
            raise MCPServerError(f"Invalid JSON response: {e}")

    def get_available_tools(self) -> List[ToolDefinition]:
        """Get list of available tools."""
        return list(self.tools.values())

    def get_server_info(self) -> Dict[str, Any]:
        """Get server information."""
        return {
            "name": self.server_config.name,
            "command": self.server_config.command,
            "args": self.server_config.args,
            "enabled": self.server_config.enabled,
            "connected": self.connected,
            "tool_count": len(self.tools),
        }

    def _validate_tool_call(self, tool_call: ToolCall) -> None:
        """Validate a tool call."""
        if not tool_call.name:
            raise MCPToolError("Tool name is required")

        if tool_call.name not in self.tools:
            raise MCPToolError(f"Tool '{tool_call.name}' not found")

        tool_def = self.tools[tool_call.name]

        # Basic parameter validation
        if "required" in tool_def.parameters:
            required_params = tool_def.parameters["required"]
            for param in required_params:
                if param not in tool_call.arguments:
                    raise MCPToolError(
                        f"Missing required parameter '{param}' for tool '{tool_call.name}'"
                    )

        # Parameter type validation
        if "properties" in tool_def.parameters:
            properties = tool_def.parameters["properties"]
            for param_name, param_value in tool_call.arguments.items():
                if param_name in properties:
                    expected_type = properties[param_name].get("type")
                    if expected_type == "number" and not isinstance(
                        param_value, (int, float)
                    ):
                        raise MCPToolError(
                            f"Invalid parameter type for '{param_name}': expected number, got {type(param_value).__name__}"
                        )
                    elif expected_type == "string" and not isinstance(param_value, str):
                        raise MCPToolError(
                            f"Invalid parameter type for '{param_name}': expected string, got {type(param_value).__name__}"
                        )

    async def call_tool_with_retry(
        self, name: str, arguments: Dict[str, Any], max_retries: int = 2
    ) -> ToolResult:
        """
        Execute a tool call with retry logic for transient failures.

        Args:
            name: Name of the tool to call
            arguments: Arguments to pass to the tool
            max_retries: Maximum number of retry attempts

        Returns:
            Result of the tool execution

        Raises:
            MCPToolError: If tool execution fails after retries
            MCPServerError: If not connected
        """
        last_error = None

        for attempt in range(max_retries + 1):
            try:
                return await self.call_tool(name, arguments)

            except MCPToolError as e:
                last_error = e

                # Don't retry for certain types of tool errors
                error_msg = str(e).lower()
                if any(
                    indicator in error_msg
                    for indicator in [
                        "not found",
                        "invalid parameter",
                        "permission denied",
                    ]
                ):
                    raise

                if attempt < max_retries:
                    logger.warning(
                        f"Tool call failed (attempt {attempt + 1}), retrying: {e}"
                    )
                    await asyncio.sleep(1.0 * (attempt + 1))  # Progressive delay

                    # Try to reconnect if connection seems broken
                    if not self.is_connected:
                        await self.reconnect()

            except MCPServerError as e:
                last_error = e

                if attempt < max_retries:
                    logger.warning(
                        f"Server error (attempt {attempt + 1}), retrying: {e}"
                    )
                    await asyncio.sleep(2.0 * (attempt + 1))  # Progressive delay

                    # Try to reconnect
                    await self.reconnect()

        # If we get here, all retries failed
        if last_error:
            raise last_error
        else:
            raise MCPToolError(
                f"Tool '{name}' execution failed after {max_retries} retries"
            )
