"""
Documentation generation methods for the setup wizard.

This module contains all methods for generating setup guides and
documentation for various providers.
"""

from pathlib import Path
from typing import Any, Dict, List, Optional

from rich.console import Console


class SetupWizardGuideGenerator:
    """Documentation generation methods for the setup wizard."""

    def __init__(self, console: Console, provider_info: Dict[str, Dict[str, Any]]):
        """
        Initialize the guide generator component.

        Args:
            console: Rich console instance
            provider_info: Dictionary of provider information
        """
        self.console = console
        self.provider_info = provider_info

    def create_setup_guide(
        self, provider_name: str, output_path: Optional[str] = None
    ) -> str:
        """
        Create a detailed setup guide for a specific provider.

        Args:
            provider_name: Name of the provider
            output_path: Optional path to save the guide

        Returns:
            Setup guide content as markdown string
        """
        if provider_name not in self.provider_info:
            raise ValueError(f"Unknown provider: {provider_name}")

        info = self.provider_info[provider_name]

        # Generate comprehensive setup guide
        guide_content = f"""# {info['name']} Setup Guide

## Overview

{info['description']}

### Key Strengths
{chr(10).join(f"- {strength}" for strength in info['strengths'])}

## Prerequisites

### Account Setup
1. Visit {info['api_key_url']}
2. Create an account or sign in to your existing account
3. Navigate to the API keys section
4. Generate a new API key for Omnimancer

### System Requirements
- Internet connection for API calls
- Omnimancer CLI installed and configured
- Sufficient account credits/quota for API usage

## Step-by-Step Setup

### 1. Get Your API Key

1. **Visit the API Console**: Go to {info['api_key_url']}
2. **Sign In**: Use your {info['name']} account credentials
3. **Generate Key**: Create a new API key specifically for Omnimancer
4. **Copy Key**: Save the API key securely (it should start with `{info['api_key_prefix']}`)

### 2. Configure Omnimancer

#### Option A: Using Setup Wizard (Recommended)
```bash
omnimancer --setup
```
Then select "{info['name']}" from the provider list.

#### Option B: Manual Configuration
```bash
omnimancer --config set-provider {provider_name} \\
  --api-key "your-api-key-here" \\
  --model "{info['models'][0]}"
```

### 3. Test Your Configuration

```bash
omnimancer --test-provider {provider_name}
```

## Available Models

{self._generate_model_table(provider_name, info)}

## Configuration Options

### Basic Settings
- **API Key**: Your {info['name']} API key
- **Model**: Choose from available models above
- **Temperature**: Controls randomness (0.0-1.0, default: 0.7)
- **Max Tokens**: Maximum response length (default: 4096)

{self._generate_provider_specific_config(provider_name)}

## Troubleshooting

{self._generate_troubleshooting_section(provider_name)}

## Best Practices

{self._generate_best_practices(provider_name)}

## Support and Resources

- **API Documentation**: {info['api_key_url']}
{self._generate_support_links(provider_name)}

---

*Generated by Omnimancer Setup Wizard*
"""

        # Save to file if path provided
        if output_path:
            with open(output_path, "w") as f:
                f.write(guide_content)

        return guide_content

    def _generate_model_table(self, provider_name: str, info: Dict[str, Any]) -> str:
        """Generate markdown table of available models."""
        models = info["models"]

        # Model descriptions
        descriptions = {
            "claude-3-5-sonnet-20241022": "Latest Claude model with enhanced reasoning and coding capabilities",
            "claude-3-haiku-20240307": "Fast, efficient model for quick responses",
            "claude-3-opus-20240229": "Most capable Claude model for complex tasks",
            "gpt-4o": "Latest GPT-4 with vision, tool use, and enhanced capabilities",
            "gpt-4o-mini": "Efficient GPT-4 variant with good performance/cost ratio",
            "gpt-4-turbo": "Enhanced GPT-4 with 128k context window",
            "gpt-3.5-turbo": "Fast, cost-effective model for general use",
            "gemini-1.5-pro": "Advanced Gemini with 2M token context window",
            "gemini-1.5-flash": "Fast Gemini variant optimized for speed",
            "gemini-1.0-pro": "Stable Gemini model for general use",
            "sonar-pro": "Advanced search-enabled model with real-time information",
            "sonar": "Standard search-enabled model",
            "deep-research": "Specialized model for in-depth research tasks",
            "grok-3": "Latest Grok model with enhanced reasoning",
            "grok-3-fast": "Optimized Grok variant for faster responses",
            "mistral-small-3.1": "Efficient model for general tasks",
            "mistral-medium": "Balanced performance and capability",
            "mistral-large": "Most capable Mistral model",
            "command-r": "Cohere's reasoning-focused model",
            "command-r-plus": "Enhanced version of Command-R",
            "command-light": "Lightweight model for simple tasks",
            "llama3.2": "Meta's latest open-source model",
            "codellama": "Specialized for code generation and analysis",
            "mistral": "General-purpose open-source model",
            "phi3": "Microsoft's efficient small language model",
        }

        table = (
            "| Model | Description | Best For |\n|-------|-------------|----------|\n"
        )

        for model in models:
            description = descriptions.get(model, "High-quality AI model")
            # Determine best use case based on model name
            if (
                "haiku" in model
                or "flash" in model
                or "mini" in model
                or "light" in model
                or "fast" in model
            ):
                best_for = "Quick responses, cost efficiency"
            elif "opus" in model or "large" in model or "pro" in model:
                best_for = "Complex tasks, detailed analysis"
            elif "code" in model:
                best_for = "Programming, code analysis"
            elif "search" in model or "sonar" in model:
                best_for = "Research, real-time information"
            else:
                best_for = "General purpose tasks"

            table += f"| `{model}` | {description} | {best_for} |\n"

        return table

    def _generate_provider_specific_config(self, provider_name: str) -> str:
        """Generate provider-specific configuration documentation."""
        if provider_name == "claude":
            return """
### Claude-Specific Settings
- **API Version**: Uses Anthropic API version 2023-06-01
- **System Messages**: Supported through conversation context
- **Tool Calling**: Available on Claude 3 models
- **Vision**: Supported on Claude 3 models with image inputs
"""
        elif provider_name == "openai":
            return """
### OpenAI-Specific Settings
- **Organization**: Optional organization ID for team accounts
- **Project**: Optional project ID for organization management
- **Tool Calling**: Available on GPT-4 and GPT-3.5-turbo models
- **Vision**: Available on GPT-4V and GPT-4o models
- **Function Calling**: Legacy function calling support
"""
        elif provider_name == "gemini":
            return """
### Gemini-Specific Settings
- **Safety Settings**: Configure content filtering levels
- **Project ID**: Google Cloud project for billing
- **Location**: Geographic region for processing
- **Multimodal**: Native support for text, images, and other media
"""
        elif provider_name == "ollama":
            return """
### Ollama-Specific Settings
- **Base URL**: Ollama server endpoint (default: http://localhost:11434)
- **Timeout**: Request timeout for local inference
- **Model Management**: Use `ollama pull <model>` to install models
- **GPU Support**: Automatic GPU acceleration when available
"""
        elif provider_name == "perplexity":
            return """
### Perplexity-Specific Settings
- **Search Mode**: Enable real-time web search capabilities
- **Recency Filter**: Control how recent information should be
- **Citation Style**: Configure how sources are referenced
- **Pro Features**: Requires Perplexity Pro subscription
"""
        else:
            return """
### Provider-Specific Settings
- **Custom Headers**: Additional HTTP headers if required
- **Rate Limiting**: Built-in rate limit handling
- **Retry Logic**: Automatic retry on transient failures
- **Timeout**: Request timeout configuration
"""

    def _generate_troubleshooting_section(self, provider_name: str) -> str:
        """Generate troubleshooting section for the provider."""
        from .setup_wizard_core import SetupWizardCore

        # Create a temporary instance just to access the common issues method
        temp_core = SetupWizardCore(None, None)
        common_issues = temp_core.get_common_setup_issues(provider_name)

        section = "### Common Issues\n\n"

        for i, issue in enumerate(common_issues, 1):
            section += f"#### {i}. {issue['issue']}\n\n"
            section += f"**Solution**: {issue['solution']}\n\n"

        section += "### Getting Help\n\n"
        section += "If you continue to experience issues:\n\n"
        section += "1. Check the provider's status page for known outages\n"
        section += "2. Verify your account has sufficient credits/quota\n"
        section += "3. Test with a different model from the same provider\n"
        section += "4. Check Omnimancer logs for detailed error messages\n"
        section += "5. Contact the provider's support team if issues persist\n"

        return section

    def _generate_best_practices(self, provider_name: str) -> str:
        """Generate best practices section for the provider."""
        practices = [
            "**Secure API Keys**: Never share or commit API keys to version control",
            "**Monitor Usage**: Keep track of your API usage and costs",
            "**Rate Limiting**: Respect provider rate limits to avoid throttling",
            "**Error Handling**: Implement proper error handling in your applications",
            "**Model Selection**: Choose the right model for your specific use case",
        ]

        if provider_name == "claude":
            practices.extend(
                [
                    "**Context Management**: Use system messages effectively for better responses",
                    "**Safety Guidelines**: Follow Anthropic's usage policies and safety guidelines",
                ]
            )
        elif provider_name == "openai":
            practices.extend(
                [
                    "**Fine-tuning**: Consider fine-tuning for specialized use cases",
                    "**Prompt Engineering**: Invest time in crafting effective prompts",
                ]
            )
        elif provider_name == "ollama":
            practices.extend(
                [
                    "**Resource Management**: Monitor CPU/GPU usage during inference",
                    "**Model Updates**: Regularly update models for improved performance",
                ]
            )

        return "\n".join(f"- {practice}" for practice in practices)

    def _generate_support_links(self, provider_name: str) -> str:
        """Generate support links for the provider."""
        links = []

        if provider_name == "claude":
            links = [
                "- **Status Page**: https://status.anthropic.com/",
                "- **Support**: https://support.anthropic.com/",
                "- **Community**: https://discord.gg/anthropic",
            ]
        elif provider_name == "openai":
            links = [
                "- **Status Page**: https://status.openai.com/",
                "- **Support**: https://help.openai.com/",
                "- **Community**: https://community.openai.com/",
            ]
        elif provider_name == "gemini":
            links = [
                "- **Status Page**: https://status.cloud.google.com/",
                "- **Support**: https://support.google.com/",
                "- **Community**: https://developers.googleblog.com/",
            ]
        elif provider_name == "ollama":
            links = [
                "- **GitHub**: https://github.com/ollama/ollama",
                "- **Documentation**: https://ollama.ai/docs",
                "- **Discord**: https://discord.gg/ollama",
            ]
        else:
            links = [
                "- **Provider Documentation**: Check the official documentation",
                "- **Community Forums**: Look for community support channels",
                "- **GitHub Issues**: Report bugs or request features",
            ]

        return "\n".join(links)

    def generate_all_setup_guides(self, output_dir: str) -> List[str]:
        """
        Generate setup guides for all supported providers.

        Args:
            output_dir: Directory to save the guides

        Returns:
            List of generated file paths
        """
        output_path = Path(output_dir)
        output_path.mkdir(parents=True, exist_ok=True)

        generated_files = []

        for provider_name in self.provider_info.keys():
            guide_filename = f"{provider_name}_setup_guide.md"
            guide_path = output_path / guide_filename

            try:
                self.create_setup_guide(provider_name, str(guide_path))
                generated_files.append(str(guide_path))
                self.console.print(
                    f"[green]✅ Generated setup guide for {provider_name}: {guide_path}[/green]"
                )
            except Exception as e:
                self.console.print(
                    f"[red]❌ Failed to generate guide for {provider_name}: {e}[/red]"
                )

        return generated_files
