from pydantic import BaseModel, Field
from typing import Any, List, Optional

class GetTimeSeriesAdRequest(BaseModel):
    """The Accumulation/Distribution (AD) technical indicator is a volume-based tool used in chart analysis to identify potential buying or selling pressure in a security. It measures the cumulative flow of money into and out of a financial instrument by comparing its closing price to its price range, while considering trading volume. The AD line's trend can help traders gauge the strength of a price move and spot potential reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesAddRequest(BaseModel):
    """The Addition (ADD) indicator performs a simple arithmetic addition of two input data series, typically used to combine multiple technical indicators or price data."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type_1: str = Field(default='open', description='Price type used as the first part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    series_type_2: str = Field(default='close', description='Price type used as the second part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesAdOscRequest(BaseModel):
    """The Accumulation/Distribution Oscillator (ADOSC) is a momentum indicator derived from the AD line, used to identify buying or selling pressure and potential trend reversals by comparing short-term and long-term price and volume trends."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    fast_period: int = Field(default=12, description='Number of periods for fast moving average. Takes values in the range from `1` to `800`')
    slow_period: int = Field(default=26, description='Number of periods for slow moving average. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesAdxRequest(BaseModel):
    """The Average Directional Index (ADX) measures the strength of a trend, regardless of direction, helping traders determine if a market is trending or ranging."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesAdxrRequest(BaseModel):
    """The Average Directional Movement Index Rating (ADXR) is a smoothed version of ADX, providing a more stable measure of trend strength over time."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetAnalystRatingsLightRequest(BaseModel):
    """This API endpoint returns a lightweight version of ratings issued by analyst firms. Works for US and international markets. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Filter by symbol', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    rating_change: Optional[str] = Field(default=None, description='Filter by rating change action', examples=['Maintains', 'Upgrade', 'Downgrade', 'Initiates', 'Reiterates'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetAnalystRatingsUsEquitiesRequest(BaseModel):
    """This API endpoint returns complete information on ratings issued by analyst firms. Works only for US equities. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Filter by symbol', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    rating_change: Optional[str] = Field(default=None, description='Filter by rating change action', examples=['Maintains', 'Upgrade', 'Downgrade', 'Initiates', 'Reiterates'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetApiUsageRequest(BaseModel):
    """This endpoint will provide information on the current usage of Twelve Data API."""
    format: str = Field(default='JSON', description='Output format', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='Specify the delimiter used when downloading the CSV file')
    timezone: str = Field(default='UTC', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>2. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a>.</li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesApoRequest(BaseModel):
    """The Absolute Price Oscillator (APO) is a momentum indicator that measures the difference between two moving averages, helping traders identify potential price trends and reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    fast_period: int = Field(default=12, description='Number of periods for fast moving average. Takes values in the range from `1` to `800`')
    slow_period: int = Field(default=26, description='Number of periods for slow moving average. Takes values in the range from `1` to `800`')
    ma_type: str = Field(default='SMA', description='The type of moving average used', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesAroonRequest(BaseModel):
    """The Aroon indicator detects the presence and strength of trends by measuring the time elapsed since the highest high and lowest low within a specific period."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesAroonOscRequest(BaseModel):
    """The Aroon Oscillator is the difference between the Aroon Up and Aroon Down lines, indicating trend strength and potential reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesAtrRequest(BaseModel):
    """The Average True Range (ATR) is a volatility indicator that measures the average range of price movement over a specified period, helping traders assess market volatility."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesAvgRequest(BaseModel):
    """The Average (AVG) indicator calculates the arithmetic mean of a data series over a specified period, often used to smooth out data fluctuations."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesAvgPriceRequest(BaseModel):
    """The Average Price (AVGPRICE) indicator computes the average of a security's open, high, low, and close prices, providing a simplified view of price action."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetBalanceSheetRequest(BaseModel):
    """Returns complete balance sheet of a company showing the summary of assets, liabilities, and shareholders’ equity."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    period: str = Field(default='annual', description='The reporting period for the balane sheet data', examples=['annual', 'quarterly'])
    start_date: Optional[str] = Field(default=None, description='Begin date for filtering items by fiscal date.\nReturns income statements with fiscal dates on or after this date. Format `2006-01-02`', examples=['2024-01-01'])
    end_date: Optional[str] = Field(default=None, description='End date for filtering items by fiscal date.\nReturns income statements with fiscal dates on or before this date. Format `2006-01-02`', examples=['2024-05-01'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetBalanceSheetConsolidatedRequest(BaseModel):
    """Returns consolidated balance sheet of a company showing the summary of assets, liabilities, and shareholders."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    period: str = Field(default='annual', description='The reporting period for the balance sheet data.', examples=['annual', 'quarterly'])
    start_date: Optional[str] = Field(default=None, description='Begin date for filtering items by fiscal date. Returns income statements with fiscal dates on or after this date.\nFormat `2006-01-02`')
    end_date: Optional[str] = Field(default=None, description='End date for filtering items by fiscal date. Returns income statements with fiscal dates on or before this date.\nFormat `2006-01-02`')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class AdvancedRequest(BaseModel):
    """This endpoint allows you to request multiple instruments, intervals, and endpoints simultaneously.

### Request body
Only JSON `POST` requests are supported.
The request content structure consists of key-value items. The key is a unique request ID. The value is requested url.

### Response
The response contains key-value data. The key is a unique request ID. The value is returned data.

### API credits
<ul>
<li>The number of concurrent requests is limited by your subscription plan.</li>
<li>Credits are consumed per requested endpoint, with the total usage equal to the sum of individual requests in the batch.</li>
<li>If the requested data exceeds your available credits, only partial data will be returned asynchronously until your quota is exhausted.</li>
<li>If one or more requests in the batch contain errors (e.g., invalid symbols or unsupported intervals), it will not affect the successful processing of other requests. Errors are reported individually within the response, allowing you to identify and correct specific issues without impacting the entire batch.</li>
</ul>"""
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesBBandsRequest(BaseModel):
    """Bollinger Bands (BBANDS) are volatility bands placed above and below a moving average, measuring price volatility and helping traders identify potential overbought or oversold conditions."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=20, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    sd: float = Field(default=2, description='Number of standard deviations. Must be at least `1`')
    ma_type: str = Field(default='SMA', description='The type of moving average used', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesBetaRequest(BaseModel):
    """The Beta indicator measures a security's sensitivity to market movements, comparing its price changes to a benchmark index to assess systematic risk."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type_1: str = Field(default='open', description='Price type used as the first part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    series_type_2: str = Field(default='close', description='Price type used as the second part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetBondsRequest(BaseModel):
    """This API call returns an array of bonds available at Twelve Data API. This list is updated daily."""
    symbol: Optional[str] = Field(default=None, description='The ticker symbol of an instrument for which data is requested', examples=['US2Y'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NYSE'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    show_plan: bool = Field(default=False, description='Adds info on which plan symbol is available')
    page: int = Field(default=1, description='Page number of the results to fetch')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesBopRequest(BaseModel):
    """The Balance of Power (BOP) indicator measures the balance between buying and selling pressure in a security by comparing its open, high, low, and close prices, helping traders identify potential price trends."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetCashFlowRequest(BaseModel):
    """Returns complete cash flow of a company showing the net amount of cash and cash equivalents being transferred into and out of business."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    period: str = Field(default='annual', description='The reporting period for the cash flow statements', examples=['annual', 'quarterly'])
    start_date: Optional[str] = Field(default=None, description='Start date for filtering cash flow statements.\nOnly cash flow statements with fiscal dates on or after this date will be included.\nFormat `2006-01-02`', examples=['2024-01-01'])
    end_date: Optional[str] = Field(default=None, description='End date for filtering cash flow statements.\nOnly cash flow statements with fiscal dates on or before this date will be included.\nFormat `2006-01-02`', examples=['2024-12-31'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetCashFlowConsolidatedRequest(BaseModel):
    """Returns consolidated cash flow of a company showing the net amount of cash and cash equivalents being transferred into and out of business."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    period: str = Field(default='annual', description='The reporting period for the cash flow statements', examples=['annual', 'quarterly'])
    start_date: Optional[str] = Field(default=None, description='Start date for filtering cash flow statements. Only cash flow statements with fiscal dates on or after this date will be included.\nFormat `2006-01-02`', examples=['2024-01-01'])
    end_date: Optional[str] = Field(default=None, description='End date for filtering cash flow statements. Only cash flow statements with fiscal dates on or before this date will be included.\nFormat `2006-01-02`', examples=['2024-12-31'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesCciRequest(BaseModel):
    """The Commodity Channel Index (CCI) is a momentum oscillator that measures the deviation of a security's price from its average relative to its typical price range, helping traders identify overbought or oversold conditions."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=20, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesCeilRequest(BaseModel):
    """The Ceiling (CEIL) indicator rounds input data up to the nearest integer, often used in conjunction with other indicators for data analysis or calculation purposes."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesCmoRequest(BaseModel):
    """The Chande Momentum Oscillator (CMO) is a momentum indicator that measures the relative strength of up and down price movements, helping traders identify overbought or oversold conditions and potential trend reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetCommoditiesRequest(BaseModel):
    """This API call returns an array of commodity pairs available at Twelve Data API. This list is updated daily."""
    symbol: Optional[str] = Field(default=None, description='The ticker symbol of an instrument for which data is requested', examples=['XAU/USD'])
    category: Optional[str] = Field(default=None, description='Filter by category of commodity', examples=['Precious Metal'])
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesCoppockRequest(BaseModel):
    """The Coppock Curve is a momentum oscillator that measures the rate of change in a security's price, helping traders identify potential long-term trend reversals, especially in bottoming markets."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    wma_period: int = Field(default=10, description='Number of periods for weighted moving average. Takes values in the range from `1` to `800`')
    long_roc_period: int = Field(default=14, description='Number of periods for long term rate of change. Takes values in the range from `1` to `800`')
    short_roc_period: int = Field(default=11, description='Number of periods for short term rate of change. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesCorrelRequest(BaseModel):
    """The Correlation (CORREL) indicator measures the statistical relationship between two securities, helping traders identify potential diversification opportunities or pairs trading candidates."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type_1: str = Field(default='open', description='Price type used as the first part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    series_type_2: str = Field(default='close', description='Price type used as the second part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetCountriesRequest(BaseModel):
    """This API call returns an array of countries available at Twelve Data API with their ISO codes, names, capital, and currency."""
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetCrossListingsRequest(BaseModel):
    """This API call returns an array of cross listed symbols for a specified instrument. Cross listings are the same securities listed on different exchanges. This list is updated daily."""
    symbol: str = Field(..., description='The ticker symbol of an instrument for which data is requested', examples=['NVDA'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market identifier code (MIC) under ISO 10383 standard', examples=['XNGS'])
    country: Optional[str] = Field(default=None, description='Country to which stock exchange belongs to', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesCrsiRequest(BaseModel):
    """The Connors RSI is a composite indicator combining the Relative Strength Index (RSI), the Rate of Change (ROC), and the Up/Down Length, providing a more comprehensive view of momentum and potential trend reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    rsi_period: int = Field(default=3, description='Number of periods for RSI used to calculate price momentum. Takes values in the range from `1` to `800`')
    up_down_length: int = Field(default=2, description='Number of periods for RSI used to calculate up/down trend. Takes values in the range from `1` to `800`')
    percent_rank_period: int = Field(default=100, description='Number of periods used to calculate PercentRank. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetCryptocurrenciesRequest(BaseModel):
    """This API call returns an array of cryptocurrencies available at Twelve Data API. This list is updated daily."""
    symbol: Optional[str] = Field(default=None, description='The ticker symbol of an instrument for which data is requested', examples=['BTC/USD'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name. E.g. `Binance`, `Coinbase`, etc.', examples=['Binance'])
    currency_base: Optional[str] = Field(default=None, description='Filter by currency base', examples=['BTC'])
    currency_quote: Optional[str] = Field(default=None, description='Filter by currency quote', examples=['USD'])
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetCryptocurrencyExchangesRequest(BaseModel):
    """This API call returns an array of cryptocurrency exchanges available at Twelve Data API. This list is updated daily."""
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='Specify the delimiter used when downloading the CSV file')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetCurrencyConversionRequest(BaseModel):
    """This API call returns real-time exchange rate and converted amount for currency pair. Works with forex and cryptocurrency."""
    symbol: str = Field(..., description='The currency pair you want to request can be either forex or cryptocurrency. Slash(/) delimiter is used. E.g. `EUR/USD` or `BTC/ETH` will be correct', examples=['EUR/USD'])
    amount: float = Field(..., description='Amount of base currency to be converted into quote currency. Supports values in the range from `0` and above', examples=['100'])
    date: Optional[str] = Field(default=None, description='If not null, will use exchange rate from a specific date or time. Format `2006-01-02` or `2006-01-02 15:04:05`. Is set in the local exchange time zone, use timezone parameter to specify a specific time zone', examples=['2006-01-02 15:04:05'])
    format: str = Field(default='JSON', description='Value can be `JSON` or `CSV`. Default `JSON`', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='Specify the delimiter used when downloading the `CSV` file. Default semicolon `;`')
    dp: int = Field(default=5, description='The number of decimal places for the data')
    timezone: Optional[str] = Field(default=None, description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a>.</li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>', examples=['UTC'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesDemaRequest(BaseModel):
    """The Double Exponential Moving Average (DEMA) is a more responsive moving average that reduces lag by giving more weight to recent price data, helping traders identify trends and potential entry or exit points."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetDirectHoldersRequest(BaseModel):
    """Returns the amount of the stocks owned directly and recorded in the company's share registry. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesDivRequest(BaseModel):
    """The Division (DIV) indicator performs arithmetic division between two input data series, typically used to combine or normalize multiple technical indicators or price data."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type_1: str = Field(default='open', description='Price type used as the first part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    series_type_2: str = Field(default='close', description='Price type used as the second part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetDividendsRequest(BaseModel):
    """Returns the amount of dividends paid out for the last 10+ years. Available starting from the `Grow` plan."""
    symbol: str = Field(..., description='Symbol ticker of instrument. For preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['US'])
    range: str = Field(default='last', description='Specifies the time range for which to retrieve dividend data.\nAccepts values such as `last` (most recent dividend), `next` (upcoming dividend),\n`1m` - `5y` for respective periods, or `full` for all available data.\nIf provided together with `start_date` and/or `end_date`, this parameter takes precedence.', examples=['last', 'next', '1m', '3m', '6m', 'ytd', '1y', '2y', '5y', 'full'])
    start_date: Optional[str] = Field(default=None, description='Start date for the dividend data query. Only dividends with dates on or after this date will be returned. Format `2006-01-02`.\nIf provided together with `range` parameter, `range` will take precedence.', examples=['2024-01-01'])
    end_date: Optional[str] = Field(default=None, description='End date for the dividend data query. Only dividends with dates on or after this date will be returned. Format `2006-01-02`.\nIf provided together with `range` parameter, `range` will take precedence.', examples=['2024-12-31'])
    adjust: bool = Field(default=True, description='Specifies if there should be an adjustment')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetDividendsCalendarRequest(BaseModel):
    """This API method returns dividend data as a calendar for a given date range. To call custom period, use start_date and end_date parameters. Available starting from the `Grow` plan."""
    symbol: Optional[str] = Field(default=None, description='Symbol ticker of instrument. For preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['US'])
    start_date: Optional[str] = Field(default=None, description='Start date for the dividends calendar query. Only dividends with ex-dates on or after this date will be returned. Format `2006-01-02`', examples=['2024-01-01'])
    end_date: Optional[str] = Field(default=None, description='End date for the dividends calendar query. Only dividends with ex-dates on or before this date will be returned. Format `2006-01-02`', examples=['2024-12-31'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    page: int = Field(default=1, description='Page number')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesDpoRequest(BaseModel):
    """The Detrended Price Oscillator (DPO) is a momentum oscillator that removes the underlying trend from price data, helping traders identify potential cyclical patterns and overbought or oversold conditions."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    centered: bool = Field(default=False, description='Specifies if there should be a shift to match the current price')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesDxRequest(BaseModel):
    """The Directional Movement Index (DX) is a component of the ADX indicator, measuring the strength of the positive and negative directional movements in a security's price."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetEarliestTimestampRequest(BaseModel):
    """This method returns the first available DateTime for a given instrument at the specific interval."""
    symbol: str = Field(..., description='Symbol ticker of the instrument.', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI).', examples=['BBG000B9XRY4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series.', examples=['1day', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '1day', '1week', '1month'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded.', examples=['Nasdaq'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard.', examples=['XNAS'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a>.</li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetEarningsRequest(BaseModel):
    """This API call returns earnings data for a given company, including EPS estimate and EPS actual. Earnings are available for complete company history. Available starting from the `Grow` plan."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    period: Optional[str] = Field(default=None, description='Type of earning, returns only 1 record.\nWhen is not empty, dates and outputsize parameters are ignored', examples=['latest', 'next'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    start_date: Optional[str] = Field(default=None, description='The date from which the data is requested. The date format is `YYYY-MM-DD`.', examples=['2024-04-01'])
    end_date: Optional[str] = Field(default=None, description='The date to which the data is requested. The date format is `YYYY-MM-DD`.', examples=['2024-04-30'])
    dp: int = Field(default=2, description='The number of decimal places in the response data. Should be in range [0,11] inclusive')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetEarningsCalendarRequest(BaseModel):
    """This API method returns earning data as a calendar for a given date range. By default today’s earning is returned. To call custom period, use start_date and end_date parameters. Available starting from the `Grow` plan."""
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    format: str = Field(default='JSON', description='Value can be JSON or CSV', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='Specify the delimiter used when downloading the CSV file')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with end_date.\nFormat `2006-01-02` or `2006-01-02 15:04:05`', examples=['2024-04-01'])
    end_date: Optional[str] = Field(default=None, description='Can be used separately and together with start_date.\nFormat `2006-01-02` or `2006-01-02 15:04:05`', examples=['2024-04-30'])
    dp: int = Field(default=2, description='Specifies the number of decimal places for floating values.\nShould be in range [0,11] inclusive')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetEarningsEstimateRequest(BaseModel):
    """This API endpoint returns analysts' estimate for a company's future quarterly and annual earnings per share (EPS). Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Filter by symbol', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetEdgarFilingsArchiveRequest(BaseModel):
    """Real-time and historical access to all forms, filings, and exhibits directly from the SEC's EDGAR system. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='The ticker symbol of an instrument for which data is requested', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Filter by market identifier code (MIC) under ISO 10383 standard', examples=['XNGS'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    form_type: Optional[str] = Field(default=None, description='Filter by form types, example `8-K`, `EX-1.1`', examples=['8-K'])
    filled_from: Optional[str] = Field(default=None, description='Filter by filled time from', examples=['2024-01-01'])
    filled_to: Optional[str] = Field(default=None, description='Filter by filled time to', examples=['2024-01-01'])
    page: int = Field(default=1, description='Page number')
    page_size: int = Field(default=10, description='Number of records in response')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesEmaRequest(BaseModel):
    """The Exponential Moving Average (EMA) is a weighted moving average that gives more importance to recent price data, making it more responsive to new information and helping traders identify trends and potential entry or exit points."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetEodRequest(BaseModel):
    """This endpoint returns the latest End of Day (EOD) price of an instrument."""
    symbol: str = Field(..., description='Symbol ticker of the instrument', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000BHTMY7'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['ETF', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    date: Optional[str] = Field(default=None, description='If not null, then return data from a specific date', examples=['2006-01-02'])
    prepost: bool = Field(default=False, description='Parameter is optional. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    dp: int = Field(default=5, description='Specifies the number of decimal places for floating values\nShould be in range [0,11] inclusive')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetEpsRevisionsRequest(BaseModel):
    """This API endpoint returns analysts’ revisions of a company's future quarterly and annual earnings per share (EPS) over the last week and month. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Filter by symbol', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetEpsTrendRequest(BaseModel):
    """This API endpoint returns a breakdown of the estimated historical EPS changes at a given period. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Filter by symbol', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetEtfRequest(BaseModel):
    """This API call returns an array of ETFs available at Twelve Data API. This list is updated daily."""
    symbol: Optional[str] = Field(default=None, description='The ticker symbol of an instrument for which data is requested', examples=['SPY'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000BDTF76'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NYSE'])
    mic_code: Optional[str] = Field(default=None, description='Filter by market identifier code (MIC) under ISO 10383 standard', examples=['XNYS'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    show_plan: bool = Field(default=False, description='Adds info on which plan symbol is available')
    include_delisted: bool = Field(default=False, description='Include delisted identifiers')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetETFsFamilyRequest(BaseModel):
    """This API request returns a list of exchange traded funds families."""
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    fund_family: Optional[str] = Field(default=None, description='Filter by investment company that manages the fund', examples=['iShares'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetETFsListRequest(BaseModel):
    """This API request returns a list of exchange traded funds available at Twelve Data. Sorting is in descending order by total assets value. The list is updated daily."""
    symbol: Optional[str] = Field(default=None, description='Filter by symbol', examples=['IVV'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000BVZ697'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US4642872000'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['464287200'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    fund_family: Optional[str] = Field(default=None, description='Filter by investment company that manages the fund', examples=['iShares'])
    fund_type: Optional[str] = Field(default=None, description='Filter by the type of fund', examples=['Large Blend'])
    page: int = Field(default=1, description='Page number')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetETFsTypeRequest(BaseModel):
    """This API request returns a list of exchange traded funds types."""
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    fund_type: Optional[str] = Field(default=None, description='Filter by the type of fund', examples=['Large Blend'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetETFsWorldRequest(BaseModel):
    """This API request returns a complete breakdown of a etf, including summary, performance, risk and composition. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of etf', examples=['IVV'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000BVZ697'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US4642872000'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['464287200'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetETFsWorldCompositionRequest(BaseModel):
    """This API request returns portfolio composition of a etf, including sectors, holding details, weighted exposure, and others. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of etf', examples=['IVV'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000BVZ697'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US4642872000'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['464287200'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetETFsWorldPerformanceRequest(BaseModel):
    """This API request returns detailed performance of a etf, including trailing and annual returns. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of etf', examples=['IVV'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000BVZ697'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US4642872000'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['464287200'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetETFsWorldRiskRequest(BaseModel):
    """This API request returns core metrics to measure the risk of investing in a etf. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of etf', examples=['IVV'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000BVZ697'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US4642872000'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['464287200'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetETFsWorldSummaryRequest(BaseModel):
    """This API request returns a brief summary of a etf. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of etf', examples=['IVV'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000BVZ697'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US4642872000'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['464287200'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetExchangeRateRequest(BaseModel):
    """This API call returns real-time exchange rate for currency pair. Works with forex and cryptocurrency."""
    symbol: str = Field(..., description='The currency pair you want to request can be either forex or cryptocurrency. Slash(/) delimiter is used. E.g. `EUR/USD` or `BTC/ETH` will be correct', examples=['EUR/USD'])
    date: Optional[str] = Field(default=None, description='If not null, will use exchange rate from a specific date or time. Format `2006-01-02` or `2006-01-02 15:04:05`. Is set in the local exchange time zone, use timezone parameter to specify a specific time zone', examples=['2006-01-02 15:04:05'])
    format: str = Field(default='JSON', description='Value can be `JSON` or `CSV`. Default `JSON`', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='Specify the delimiter used when downloading the `CSV` file. Default semicolon `;`')
    dp: int = Field(default=5, description='The number of decimal places for the data')
    timezone: Optional[str] = Field(default=None, description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a>.</li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>', examples=['UTC'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetExchangeScheduleRequest(BaseModel):
    """This API call return exchanges details and trading hours. Available starting from the `Ultra` plan."""
    mic_name: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Filter by market identifier code (MIC) under ISO 10383 standard', examples=['XNGS'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    date: Optional[str] = Field(default=None, description='<p>\nIf a date is provided, the API returns the schedule for the specified date; otherwise, it returns the default (common) schedule.\n</p>\nThe date can be specified in one of the following formats:\n<ul>\n<li>An exact date (e.g., <code>2021-10-27</code>)</li>\n<li>A human-readable keyword: <code>today</code> or <code>yesterday</code></li>\n<li>A full datetime string in UTC (e.g., <code>2025-04-11 20:00:00</code>) to retrieve the schedule corresponding to the day in the specified time.</li>\n</ul>\nWhen using a datetime value, the resulting schedule will correspond to the local calendar day at the specified time.\nFor example, <code>2025-04-11 20:00:00 UTC</code> corresponds to:\n<ul>\n<li><code>2025-04-11</code> in the <code>America/New_York</code> timezone</li>\n<li><code>2025-04-12</code> in the <code>Australia/Sydney</code> timezone</li>\n</ul>', examples=['2021-10-27'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetExchangesRequest(BaseModel):
    """This API call returns an array of stock or ETF exchanges available at Twelve Data API. This list is updated daily."""
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['ETF', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    name: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    code: Optional[str] = Field(default=None, description='Filter by market identifier code (MIC) under ISO 10383 standard', examples=['XNGS'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    show_plan: bool = Field(default=False, description='Adds info on which plan symbol is available')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesExpRequest(BaseModel):
    """The Exponential (EXP) indicator calculates the exponential value of a given input, often used in advanced mathematical analysis or calculations in financial markets."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesFloorRequest(BaseModel):
    """The Floor (FLOOR) indicator rounds input data down to the nearest integer, often used in conjunction with other indicators for data analysis or calculation purposes."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetForexPairsRequest(BaseModel):
    """This API call returns an array of forex pairs available at Twelve Data API. This list is updated daily."""
    symbol: Optional[str] = Field(default=None, description='The ticker symbol of an instrument for which data is requested', examples=['EUR/USD'])
    currency_base: Optional[str] = Field(default=None, description='Filter by currency base', examples=['EUR'])
    currency_quote: Optional[str] = Field(default=None, description='Filter by currency quote', examples=['USD'])
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetFundHoldersRequest(BaseModel):
    """Returns the amount of the company’s available stock owned by mutual fund holders. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetFundsRequest(BaseModel):
    """This API call returns an array of funds available at Twelve Data API. This list is updated daily."""
    symbol: Optional[str] = Field(default=None, description='The ticker symbol of an instrument for which data is requested', examples=['FXAIX'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000BHTMY7'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['Nasdaq'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    show_plan: bool = Field(default=False, description='Adds info on which plan symbol is available')
    page: int = Field(default=1, description='Page number of the results to fetch')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetGrowthEstimatesRequest(BaseModel):
    """This API endpoint returns consensus analyst estimates over the company's growth rates for various periods. Calculation averages projections of numerous analysts, taking arbitrary parameters, such as earnings per share, revenue, etc. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Filter by symbol', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesHeikinashiCandlesRequest(BaseModel):
    """Heikin Ashi Candles are a modified form of Japanese candlestick charts, using averaged price data to smooth out noise and better highlight trends and potential trend reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesHlc3Request(BaseModel):
    """The High, Low, Close Average (HLC3) indicator calculates the average of a security's high, low, and close prices, providing a simplified view of price action."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesHtDcPeriodRequest(BaseModel):
    """The Hilbert Transform Dominant Cycle Period (HT_DCPERIOD) identifies the dominant market cycle length, helping traders adapt their strategies to different market conditions.

You can read more about it in the Rocket Science for Traders book by John F. Ehlers."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesHtDcPhaseRequest(BaseModel):
    """The Hilbert Transform Dominant Cycle Phase (HT_DCPHASE) measures the current phase of the dominant market cycle, helping traders identify potential entry and exit points in relation to the cycle.

You can read more about it in the Rocket Science for Traders book by John F. Ehlers."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesHtPhasorRequest(BaseModel):
    """The Hilbert Transform Phasor Components (HT_PHASOR) decomposes a price series into in-phase and quadrature components, providing insight into cyclical patterns and trend direction.

You can read more about it in the Rocket Science for Traders book by John F. Ehlers."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesHtSineRequest(BaseModel):
    """The Hilbert Transform Sine Wave (HT_SINE) indicator calculates sine and cosine wave components based on the dominant market cycle, helping traders identify potential turning points and trend direction.

You can read more about it in the Rocket Science for Traders book by John F. Ehlers."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesHtTrendlineRequest(BaseModel):
    """The Hilbert Transform Instantaneous Trendline (HT_TRENDLINE) is a smoothed moving average that follows the dominant market cycle, helping traders identify trends and potential entry or exit points.

You can read more about it in the Rocket Science for Traders book by John F. Ehlers."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesHtTrendModeRequest(BaseModel):
    """The Hilbert Transform Trend vs Cycle Mode (HT_TRENDMODE) distinguishes between trending and cyclical market phases, helping traders adapt their strategies accordingly.

You can read more about it in the Rocket Science for Traders book by John F. Ehlers."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesIchimokuRequest(BaseModel):
    """The Ichimoku Cloud (ICHIMOKU) is a comprehensive trend-following indicator that combines multiple moving averages and support/resistance levels to help traders identify potential entry and exit points, trend direction, and momentum."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    conversion_line_period: int = Field(default=9, description='The time period used for generating the conversation line. Takes values in the range from `1` to `800`')
    base_line_period: int = Field(default=26, description='The time period used for generating the base line. Takes values in the range from `1` to `800`')
    leading_span_b_period: int = Field(default=52, description='The time period used for generating the leading span B line. Takes values in the range from `1` to `800`')
    lagging_span_period: int = Field(default=26, description='The time period used for generating the lagging span line. Takes values in the range from `1` to `800`')
    include_ahead_span_period: bool = Field(default=True, description='Indicates whether to include ahead span period')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetIncomeStatementRequest(BaseModel):
    """Returns complete income statement of a company and shows the company’s revenues and expenses during a period (annual or quarter)."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    period: Optional[str] = Field(default=None, description='The reporting period for the income statement data', examples=['annual', 'annual', 'quarterly'])
    start_date: Optional[str] = Field(default=None, description='Begin date for filtering income statements by fiscal date.\nReturns income statements with fiscal dates on or after this date.\nFormat `2006-01-02`', examples=['2024-01-01'])
    end_date: Optional[str] = Field(default=None, description='End date for filtering income statements by fiscal date.\nReturns income statements with fiscal dates on or before this date.\nFormat `2006-01-02`', examples=['2024-12-31'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetIncomeStatementConsolidatedRequest(BaseModel):
    """Returns consolidated income statement of a company and expenses during a period (annual or quarter)."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    period: Optional[str] = Field(default=None, description='The reporting period for the income statement data', examples=['annual', 'quarterly'])
    start_date: Optional[str] = Field(default=None, description='Begin date for filtering income statements by fiscal date.\nReturns income statements with fiscal dates on or after this date.\nFormat `2006-01-02`', examples=['2024-01-01'])
    end_date: Optional[str] = Field(default=None, description='End date for filtering income statements by fiscal date.\nReturns income statements with fiscal dates on or before this date.\nFormat `2006-01-02`', examples=['2024-12-31'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetInsiderTransactionsRequest(BaseModel):
    """Returns trading information performed by insiders. Available starting from the `Pro` plan."""
    symbol: str = Field(..., description='The ticker symbol of an instrument for which data is requested, e.g., `AAPL`, `TSLA`.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded, e.g., `Nasdaq`, `NSE`', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., United States or US.', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetInstitutionalHoldersRequest(BaseModel):
    """Returns the amount of the company’s available stock owned by institutions (pension funds, insurance companies, investment firms, private foundations, endowments, or other large entities that manage funds on behalf of others). Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetInstrumentTypeRequest(BaseModel):
    """This API call returns a list of instrument types available at Twelve Data API."""
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetIntervalsRequest(BaseModel):
    """Returns a list of available time intervals for data requests."""
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetIpoCalendarRequest(BaseModel):
    """This endpoint returns past, today, or upcoming IPOs. Available starting from the `Grow` plan."""
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    start_date: Optional[str] = Field(default=None, description='The earliest IPO date to include in the results. Format: `2006-01-02`', examples=['2021-01-01'])
    end_date: Optional[str] = Field(default=None, description='The latest IPO date to include in the results. Format: `2006-01-02`', examples=['2021-12-31'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesKamaRequest(BaseModel):
    """The Kaufman Adaptive Moving Average (KAMA) is a moving average that adjusts its length based on market volatility, providing a balance between responsiveness and noise reduction."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesKeltnerRequest(BaseModel):
    """The Keltner Channel (KELTNER) is a volatility-based indicator that uses a combination of EMA and the ATR to create a channel around a security's price. The channel helps traders identify potential overbought or oversold conditions, as well as trend direction and potential price breakouts."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=20, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    atr_time_period: int = Field(default=10, description='The time period used for calculating the Average True Range. Takes values in the range from `1` to `800`')
    multiplier: int = Field(default=2, description="The factor used to adjust the indicator's sensitivity")
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    ma_type: str = Field(default='SMA', description='The type of moving average used', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetKeyExecutivesRequest(BaseModel):
    """Returns key executive information for a specified symbol. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesKstRequest(BaseModel):
    """The Know Sure Thing (KST) is a momentum oscillator that combines four different smoothed rates of change to generate a single trend-following indicator. By measuring the price momentum across multiple timeframes, KST helps traders identify potential trend reversals, overbought, or oversold conditions, and trading opportunities."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    roc_period_1: int = Field(default=10, description='The time period for the first Rate of Change calculation.')
    roc_period_2: int = Field(default=15, description='The time period for the second Rate of Change calculation.')
    roc_period_3: int = Field(default=20, description='The time period for the third Rate of Change calculation.')
    roc_period_4: int = Field(default=30, description='The time period for the forth Rate of Change calculation.')
    sma_period_1: int = Field(default=10, description='The time period for the first Simple Moving Average.')
    sma_period_2: int = Field(default=10, description='The time period for the second Simple Moving Average.')
    sma_period_3: int = Field(default=10, description='The time period for the third Simple Moving Average.')
    sma_period_4: int = Field(default=15, description='The time period for the forth Simple Moving Average.')
    signal_period: int = Field(default=9, description='The time period used for generating the signal line.')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetLastChangesRequest(BaseModel):
    """Returns the latest changes of fundamental data. This endpoint helps consume API credits more efficiently and request new data once it is updated."""
    endpoint: str = Field(..., description='Endpoint name', examples=['statistics', 'price_target', 'recommendations', 'statistics', 'insider_transactions', 'profile', 'mutual_funds_world_summary', 'mutual_funds_world', 'institutional_holders', 'analyst_rating', 'income_statement', 'income_statement_quarterly', 'cash_flow', 'cash_flow_quarterly', 'balance_sheet', 'balance_sheet_quarterly', 'mutual_funds_list', 'mutual_funds_world_sustainability', 'mutual_funds_world_summary', 'mutual_funds_world_risk', 'mutual_funds_world_purchase_info', 'mutual_funds_world_composition', 'mutual_funds_world_performance', 'mutual_funds_world', 'etfs_list', 'etfs_world', 'etfs_world_summary', 'etfs_world_performance', 'etfs_world_risk', 'etfs_world_composition'])
    start_date: Optional[str] = Field(default=None, description='The starting date and time for data selection, in `2006-01-02 15:04:05` format', examples=['2023-10-14 00:00:00'])
    symbol: Optional[str] = Field(default=None, description='Filter by symbol', examples=['AAPL'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Filter by market identifier code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    page: int = Field(default=1, description='Page number')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesLinearRegRequest(BaseModel):
    """The Linear Regression (LINEARREG) indicator calculates the best-fit straight line through a series of data points, helping traders identify trends and potential support or resistance levels."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesLinearRegAngleRequest(BaseModel):
    """The Linear Regression Angle (LINEARREGANGLE) measures the angle of the linear regression line, indicating the slope and direction of the underlying trend."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesLinearRegInterceptRequest(BaseModel):
    """The Linear Regression Intercept (LINEARREGINTERCEPT) calculates the point where the linear regression line intersects the vertical axis, providing a reference point for trend analysis."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesLinearRegSlopeRequest(BaseModel):
    """The Linear Regression Slope (LINEARREGSLOPE) measures the steepness of the linear regression line, indicating the rate of change of the underlying trend."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesLnRequest(BaseModel):
    """The Natural Logarithm (LN) indicator calculates the natural logarithm of a given input value, often used in advanced mathematical analysis or calculations in financial markets."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesLog10Request(BaseModel):
    """The Base-10 Logarithm (LOG10) indicator calculates the base-10 logarithm of a given input value, commonly used in advanced mathematical analysis or calculations in financial markets."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetLogoRequest(BaseModel):
    """Returns a logo of company, cryptocurrency, or forex pair."""
    symbol: str = Field(..., description='The ticker symbol of an instrument for which data is requested, e.g., AAPL, TSLA.', examples=['AAPL'])
    exchange: Optional[str] = Field(default=None, description='The exchange name where the instrument is traded, e.g., `NASDAQ`, `NSE`', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='The Market Identifier Code (MIC) of the exchange where the instrument is traded, e.g., `XNAS`, `XLON`', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMaRequest(BaseModel):
    """The Moving Average (MA) is a smoothing indicator that calculates the average price of a security over a specified period, helping traders identify trends and potential support or resistance levels."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    ma_type: str = Field(default='SMA', description='The type of moving average used', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMacdRequest(BaseModel):
    """The Moving Average Convergence Divergence (MACD) is a momentum indicator that measures the difference between two moving averages, with a signal line used to identify potential trend reversals and trading opportunities."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    fast_period: int = Field(default=12, description='Number of periods for fast moving average. Takes values in the range from `1` to `800`')
    slow_period: int = Field(default=26, description='Number of periods for slow moving average. Takes values in the range from `1` to `800`')
    signal_period: int = Field(default=9, description='The time period used for generating the signal line.')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMacdSlopeRequest(BaseModel):
    """The MACD Slope is the rate of change of the MACD line, helping traders identify the acceleration or deceleration of momentum in a security's price. Available starting from the `Pro` plan."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    fast_period: int = Field(default=12, description='Number of periods for fast moving average. Takes values in the range from `1` to `800`')
    slow_period: int = Field(default=26, description='Number of periods for slow moving average. Takes values in the range from `1` to `800`')
    signal_period: int = Field(default=9, description='The time period used for generating the signal line.')
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMacdExtRequest(BaseModel):
    """The MACD Extension (MACDEXT) is a customizable version of the MACD indicator, allowing traders to choose different moving average types and parameters for increased flexibility."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    fast_period: int = Field(default=12, description='Number of periods for fast moving average. Takes values in the range from `1` to `800`')
    fast_ma_type: str = Field(default='SMA', description='The type of fast moving average used in the calculation.', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    slow_period: int = Field(default=26, description='Number of periods for slow moving average. Takes values in the range from `1` to `800`')
    slow_ma_type: str = Field(default='SMA', description='The type of slow moving average used in the calculation.', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    signal_period: int = Field(default=9, description='The time period used for generating the signal line.')
    signal_ma_type: str = Field(default='SMA', description='The type of fast moving average used for generating the signal line.', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMamaRequest(BaseModel):
    """The MESA Adaptive Moving Average (MAMA) is a moving average that adapts to price movement based on the dominant market cycle, providing a balance between responsiveness and noise reduction.
More about MAMA can be read [here](http://www.mesasoftware.com/papers/MAMA.pdf)."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    fast_limit: float = Field(default=0.5, description='The limit for the fast moving average.')
    slow_limit: float = Field(default=0.05, description='The limit for the slow moving average.')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMarketCapRequest(BaseModel):
    """Market capitalization history. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Filter by symbol', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Filter by market identifier code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    start_date: Optional[str] = Field(default=None, description='Start date for market capitalization data retrieval.\nData will be returned from this date onwards.\nFormat `2006-01-02`', examples=['2023-01-01'])
    end_date: Optional[str] = Field(default=None, description='End date for market capitalization data retrieval.\nData will be returned up to and including this date.\nFormat `2006-01-02`', examples=['2023-12-31'])
    page: int = Field(default=1, description='Page number')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMarketMoversRequest(BaseModel):
    """Get the list of the top gaining or losing stocks today.

Top gainers are ordered by the highest rate of price increase since the previous day''s close. Top losers are ordered by the highest percentage of price decrease since the last day.

Data is available for all international equities, forex, crypto. Available starting from the `Pro` plan."""
    market: str = Field(..., description='Maket type', examples=['stocks', 'stocks', 'etf', 'mutual_funds', 'forex', 'crypto'])
    direction: str = Field(default='gainers', description='Specifies direction of the snapshot gainers or losers', examples=['gainers', 'losers'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    country: str = Field(default='USA', description='Country of the snapshot, applicable to non-currencies only.\nTakes country name or alpha code')
    price_greater_than: Optional[str] = Field(default=None, description='Takes values with price grater than specified value', examples=['175.5'])
    dp: str = Field(default='5', description='Specifies the number of decimal places for floating values.\nShould be in range [0,11] inclusive')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMarketStateRequest(BaseModel):
    """Check the state of all available exchanges, time to open, and time to close. Returns all available stock exchanges by default."""
    exchange: Optional[str] = Field(default=None, description='The exchange name where the instrument is traded.', examples=['NYSE'])
    code: Optional[str] = Field(default=None, description='The Market Identifier Code (MIC) of the exchange where the instrument is traded.', examples=['XNYS'])
    country: Optional[str] = Field(default=None, description='The country where the exchange is located. Takes country name or alpha code.', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMaxRequest(BaseModel):
    """The Maximum (MAX) indicator calculates the highest value of a data series over a specified period, often used to identify potential resistance levels or extreme price movements."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMaxIndexRequest(BaseModel):
    """The Maximum Index (MAXINDEX) returns the index of the highest value in a data series over a specified period, providing information about the timing of extreme price movements."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMcGinleyDynamicRequest(BaseModel):
    """The McGinley Dynamic (MCGINLEY_DYNAMIC) indicator is a smoothing tool designed to minimize the lag between the indicator and price action. It adjusts its speed based on the volatility of the security, providing a more accurate representation of price trends and potential support or resistance levels compared to traditional moving averages."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMedPriceRequest(BaseModel):
    """The Median Price (MEDPRICE) indicator calculates the midpoint between a security's high and low prices, providing a simplified view of price action."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMfiRequest(BaseModel):
    """The Money Flow Index (MFI) is a volume-weighted momentum oscillator that measures buying and selling pressure by comparing positive and negative money flow, helping traders identify overbought or oversold conditions."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMidPointRequest(BaseModel):
    """The Midpoint (MIDPOINT) indicator calculates the midpoint of a data series over a specified period, often used to smooth out data fluctuations and identify trends."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMidPriceRequest(BaseModel):
    """The Midprice (MIDPRICE) indicator calculates the average of a security's high and low prices over a specified period, providing a smoothed view of price action and potential support or resistance levels."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMinRequest(BaseModel):
    """The Minimum (MIN) indicator calculates the lowest value of a data series over a specified period, often used to identify potential support levels or extreme price movements."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMinIndexRequest(BaseModel):
    """The Minimum Index (MININDEX) returns the index of the lowest value in a data series over a specified period, providing information about the timing of extreme price movements."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMinMaxRequest(BaseModel):
    """The Minimum and Maximum (MINMAX) indicator calculates the lowest and highest values of a data series over a specified period, often used to identify potential support and resistance levels or extreme price movements."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMinMaxIndexRequest(BaseModel):
    """The Minimum and Maximum Index (MINMAXINDEX) returns the indices of the lowest and highest values in a data series over a specified period, providing information about the timing of extreme price movements."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMinusDIRequest(BaseModel):
    """The Minus Directional Indicator (MINUS_DI) is a component of the ADX indicator, measuring the strength of a security's downward price movement."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMinusDMRequest(BaseModel):
    """The Minus Directional Movement (MINUS_DM) is a component of the ADX indicator, measuring the extent of a security's downward price movement."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMomRequest(BaseModel):
    """The Momentum (MOM) indicator measures the rate of change of a security's price over a specified period, helping traders identify potential price trends and reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesMultRequest(BaseModel):
    """The Multiplication (MULT) indicator performs arithmetic multiplication of two input data series, typically used to combine or normalize multiple technical indicators or price data."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type_1: str = Field(default='open', description='Price type used as the first part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    series_type_2: str = Field(default='close', description='Price type used as the second part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMutualFundsFamilyRequest(BaseModel):
    """This API request returns a list of mutual funds families."""
    fund_family: Optional[str] = Field(default=None, description='Filter by investment company that manages the fund', examples=['Jackson National'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMutualFundsListRequest(BaseModel):
    """This API request returns a list of mutual funds available at Twelve Data. Sorting is in descending order by total assets value. The list is updated daily."""
    symbol: Optional[str] = Field(default=None, description='Filter by symbol', examples=['1535462D'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG00HMMLCH1'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['LU1206782309'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['120678230'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    fund_family: Optional[str] = Field(default=None, description='Filter by investment company that manages the fund', examples=['Jackson National'])
    fund_type: Optional[str] = Field(default=None, description='Filter by the type of fund', examples=['Small Blend'])
    performance_rating: Optional[int] = Field(default=None, description='Filter by performance rating from `0` to `5`', examples=[4])
    risk_rating: Optional[int] = Field(default=None, description='Filter by risk rating from `0` to `5`', examples=[4])
    page: int = Field(default=1, description='Page number')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMutualFundsTypeRequest(BaseModel):
    """This API request returns a list of mutual funds types."""
    fund_type: Optional[str] = Field(default=None, description='Filter by the type of fund', examples=['Jackson National'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMutualFundsWorldRequest(BaseModel):
    """This API request returns a complete breakdown of a mutual fund, including summary, performance, risk, ratings, composition, purchase_info, and sustainability. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of mutual fund', examples=['1535462D'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG00HMMLCH1'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['LU1206782309'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['120678230'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMutualFundsWorldCompositionRequest(BaseModel):
    """This API request returns portfolio composition of a mutual fund, including sectors, holding details, weighted exposure, and others. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of mutual fund', examples=['1535462D'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG00HMMLCH1'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['LU1206782309'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['120678230'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMutualFundsWorldPerformanceRequest(BaseModel):
    """This API request returns detailed performance of a mutual fund, including trailing, annual, quarterly, and load-adjusted returns. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of mutual fund', examples=['1535462D'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG00HMMLCH1'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['LU1206782309'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['120678230'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMutualFundsWorldPurchaseInfoRequest(BaseModel):
    """This API request returns essential information on purchasing a mutual fund, including minimums, pricing, and available brokerages. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of mutual fund', examples=['1535462D'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG00HMMLCH1'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['LU1206782309'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['120678230'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMutualFundsWorldRatingsRequest(BaseModel):
    """This API request returns ratings of a mutual fund. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of mutual fund', examples=['1535462D'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG00HMMLCH1'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['LU1206782309'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['120678230'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMutualFundsWorldRiskRequest(BaseModel):
    """This API request returns core metrics to measure the risk of investing in a mutual fund. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of mutual fund', examples=['1535462D'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG00HMMLCH1'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['LU1206782309'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['120678230'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMutualFundsWorldSummaryRequest(BaseModel):
    """This API request returns a brief summary of a mutual fund. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of mutual fund', examples=['1535462D'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG00HMMLCH1'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['LU1206782309'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['120678230'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetMutualFundsWorldSustainabilityRequest(BaseModel):
    """This API request returns brief information on mutual fund sustainability and ESG. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Symbol ticker of mutual fund', examples=['1535462D'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG00HMMLCH1'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['LU1206782309'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['120678230'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    dp: int = Field(default=5, description='Number of decimal places for floating values. Accepts value in range [0,11]')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesNatrRequest(BaseModel):
    """The Normalized Average True Range (NATR) is a volatility indicator that measures the average range of price movement over a specified period, expressed as a percentage of the security's price, allowing for comparisons across different securities."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesObvRequest(BaseModel):
    """The On Balance Volume (OBV) indicator is a cumulative volume-based tool used to measure buying and selling pressure, helping traders identify potential price trends and reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesPercent_BRequest(BaseModel):
    """The Percent B (%B) is a component of the Bollinger Bands indicator, measuring the position of a security's price relative to the bands, helping traders identify potential overbought or oversold conditions."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    sd: float = Field(default=2, description='The standard deviation applied in the calculation.  Must be at least `1`')
    ma_type: str = Field(default='SMA', description='The type of moving average used', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesPivotPointsHLRequest(BaseModel):
    """The Pivot Points High Low (PIVOT_POINTS_HL) indicator calculates support and resistance levels based on the highest high and lowest low of a security's price over a specified period, providing potential entry and exit points. Available starting from the `Pro` plan."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=10, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesPlusDIRequest(BaseModel):
    """The Plus Directional Indicator (PLUS_DI) is a component of the ADX indicator, measuring the strength of a security's upward price movement."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesPlusDMRequest(BaseModel):
    """The Plus Directional Movement (PLUS_DM) is a component of the ADX indicator, measuring the extent of a security's upward price movement."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesPpoRequest(BaseModel):
    """The Percentage Price Oscillator (PPO) is a momentum oscillator that measures the percentage difference between two moving averages, helping traders identify potential trend reversals and trading opportunities."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    fast_period: int = Field(default=12, description='Number of periods for fast moving average. Takes values in the range from `1` to `800`')
    slow_period: int = Field(default=26, description='Number of periods for slow moving average. Takes values in the range from `1` to `800`')
    ma_type: str = Field(default='SMA', description='The type of moving average used', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetPriceRequest(BaseModel):
    """This endpoint is a lightweight method that allows retrieving only the real-time price of the selected instrument."""
    symbol: str = Field(..., description='Symbol ticker of the instrument', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000BHTMY7'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['ETF', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    format: str = Field(default='JSON', description='Value can be JSON or CSV', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='Specify the delimiter used when downloading the CSV file')
    prepost: bool = Field(default=False, description='Parameter is optional. Only for Pro and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume.')
    dp: int = Field(default=5, description='Specifies the number of decimal places for floating values.\nShould be in range [0,11] inclusive')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetPriceTargetRequest(BaseModel):
    """This API endpoint returns the analysts' projection of a security's future price. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Filter by symbol', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetProfileRequest(BaseModel):
    """Returns general information about the company. Available starting from the `Grow` plan."""
    symbol: str = Field(..., description='Symbol ticker of instrument. For preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetQuoteRequest(BaseModel):
    """Quote endpoint is an efficient method to retrieve the latest quote of the selected instrument."""
    symbol: str = Field(..., description='Symbol ticker of the instrument', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000BHTMY7'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval of the quote', examples=['1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '1day', '1week', '1month'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    volume_time_period: int = Field(default=9, description='Number of periods for Average Volume')
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['ETF', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    format: str = Field(default='JSON', description='Value can be JSON or CSV\nDefault JSON', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='Specify the delimiter used when downloading the CSV file')
    prepost: bool = Field(default=False, description='Parameter is optional. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume.')
    eod: bool = Field(default=False, description='If true, then return data for closed day')
    rolling_period: int = Field(default=24, description='Number of hours for calculate rolling change at period. By default set to 24, it can be in range [1, 168].')
    dp: int = Field(default=5, description='Specifies the number of decimal places for floating values\nShould be in range [0,11] inclusive')
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a>.</li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetRecommendationsRequest(BaseModel):
    """This API endpoint returns the average of all analyst recommendations and classifies them as Strong Buy, Buy, Hold, or Sell. Also, it returns a recommendation score. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Filter by symbol', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    exchange: Optional[str] = Field(default=None, description='The exchange name where the instrument is traded, e.g., `Nasdaq`, `NSE`.', examples=['NASDAQ'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetRevenueEstimateRequest(BaseModel):
    """This API endpoint returns analysts' estimate for a company's future quarterly and annual sales (total revenue). Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='Filter by symbol', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    dp: int = Field(default=5, description='Number of decimal places for floating values.\nShould be in range [0,11] inclusive')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesRocRequest(BaseModel):
    """The Rate of Change (ROC) indicator measures the percentage change in a security's price over a specified period, helping traders identify potential price trends and reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesRocpRequest(BaseModel):
    """The Rate of Change Percentage (ROCP) indicator calculates the percentage change in a security's price over a specified period, providing insight into momentum and potential trend reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesRocrRequest(BaseModel):
    """The Rate of Change Ratio (ROCR) indicator measures the ratio of a security's current price to its price a specified number of periods ago, helping traders identify potential price trends and reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesRocr100Request(BaseModel):
    """The Rate of Change Ratio 100 (ROCR100) indicator measures the percentage change in a security's price over a specified period, expressed as a ratio to 100, providing insight into momentum and potential trend reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesRsiRequest(BaseModel):
    """The Relative Strength Index (RSI) is a momentum oscillator that measures the speed and change of price movements, helping traders identify potential overbought or oversold conditions and trend reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesRvolRequest(BaseModel):
    """The Relative Volume (RVOL) is a ratio that compares a security's current trading volume to its average trading volume over a specified period. By measuring volume activity relative to its historical norm, RVOL helps traders identify unusual market activity, potential breakouts, and the strength of price movements. Available starting from the `Grow` plan."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetSourceSanctionedEntitiesRequest(BaseModel):
    """This API endpoint returns a list of sanctioned entities for a given sanction source (ofac, uk, eu, au, etc) Available starting from the `Ultra` plan."""
    source: str = Field(..., description='Sanctions source', examples=['ofac', 'ofac', 'uk', 'eu', 'au'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesSarRequest(BaseModel):
    """The Parabolic SAR (SAR) is a trend-following indicator that calculates potential support and resistance levels based on a security's price and time, helping traders identify potential entry and exit points. Available starting from the `Grow` plan."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    acceleration: float = Field(default=0.02, description="The rate of change in the indicator's values.")
    maximum: float = Field(default=0.2, description='The maximum value considered for the indicator calculation.')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesSarExtRequest(BaseModel):
    """The Parabolic SAR Extended (SAREXT) is a customizable version of the Parabolic SAR indicator, allowing traders to choose different parameters for increased flexibility in identifying potential entry and exit points."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    start_value: float = Field(default=0, description='The initial value for the indicator calculation.')
    offset_on_reverse: float = Field(default=0, description="The adjustment applied when the indicator's direction changes.")
    acceleration_limit_long: float = Field(default=0.02, description='The maximum acceleration value for long positions.')
    acceleration_long: float = Field(default=0.02, description='The acceleration value for long positions.')
    acceleration_max_long: float = Field(default=0.2, description='The highest allowed acceleration for long positions.')
    acceleration_limit_short: float = Field(default=0.02, description='The maximum acceleration value for short positions.')
    acceleration_short: float = Field(default=0.02, description='The acceleration value for short positions.')
    acceleration_max_short: float = Field(default=0.2, description='The highest allowed acceleration for short positions.')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesSmaRequest(BaseModel):
    """The Simple Moving Average (SMA) is a smoothing indicator that calculates the average price of a security over a specified period, helping traders identify trends and potential support or resistance levels."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetSplitsRequest(BaseModel):
    """Returns the date and the split factor of shares of the company for the last 10+ years. Available starting from the `Grow` plan."""
    symbol: str = Field(..., description='Symbol ticker of instrument. For preffered stocks use dot(.) delimiter. E.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    range: str = Field(default='last', description='Range of data to be returned', examples=['last', '1m', '3m', '6m', 'ytd', '1y', '2y', '5y', 'full'])
    start_date: Optional[str] = Field(default=None, description='The starting date for data selection. Format `2006-01-02`', examples=['2020-01-01'])
    end_date: Optional[str] = Field(default=None, description='The ending date for data selection. Format `2006-01-02`', examples=['2020-12-31'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetSplitsCalendarRequest(BaseModel):
    """This API method returns split data as a calendar for a given date range. To call custom period, use start_date and end_date parameters. Available starting from the `Grow` plan."""
    symbol: Optional[str] = Field(default=None, description='Symbol ticker of instrument. For preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG01293F5X4'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    start_date: Optional[str] = Field(default=None, description='The starting date (inclusive) for filtering split events in the calendar. Format `2006-01-02`', examples=['2024-01-01'])
    end_date: Optional[str] = Field(default=None, description='The ending date (inclusive) for filtering split events in the calendar. Format `2006-01-02`', examples=['2024-12-31'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    page: str = Field(default='1', description='Page number')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesSqrtRequest(BaseModel):
    """The Square Root (SQRT) indicator calculates the square root of a given input value, often used in advanced mathematical analysis or calculations in financial markets."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetStatisticsRequest(BaseModel):
    """Returns current overview of company’s main statistics including valuation metrics and financials. Available starting from the `Pro` plan."""
    symbol: str = Field(..., description='Symbol ticker of instrument.\nFor preffered stocks use dot(.) delimiter.\nE.g. `BRK.A` or `BRK.B` will be correct', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000B9Y5X2'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='Country where instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesStdDevRequest(BaseModel):
    """The Standard Deviation (STDDEV) indicator measures the dispersion of a data series from its mean, often used as a volatility indicator to identify potential trading opportunities and risk management."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    sd: float = Field(default=2, description='The standard deviation applied in the calculation.')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesStochRequest(BaseModel):
    """The Stochastic Oscillator (STOCH) is a momentum indicator that compares a security's closing price to its price range over a specified period, helping traders identify potential overbought or oversold conditions and trend reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    fast_k_period: int = Field(default=14, description='The time period for the fast %K line in the Stochastic Oscillator. Takes values in the range from `1` to `800`')
    slow_k_period: int = Field(default=1, description='The time period for the slow %K line in the Stochastic Oscillator. Takes values in the range from `1` to `800`')
    slow_d_period: int = Field(default=3, description='The time period for the slow %D line in the Stochastic Oscillator. Takes values in the range from `1` to `800`')
    slow_kma_type: str = Field(default='SMA', description='The type of slow %K Moving Average used. Default is SMA.', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    slow_dma_type: str = Field(default='SMA', description='The type of slow Displaced Moving Average used. Default is SMA.', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesStochFRequest(BaseModel):
    """The Stochastic Fast (STOCHF) is a variation of the Stochastic Oscillator, using fewer periods for calculation, making it more responsive to price changes but also more prone to false signals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    fast_k_period: int = Field(default=14, description='The time period for the fast %K line in the Stochastic Oscillator. Takes values in the range from `1` to `800`')
    fast_d_period: int = Field(default=3, description='The time period for the fast %D line in the Stochastic Oscillator. Takes values in the range from `1` to `800`')
    fast_dma_type: str = Field(default='SMA', description='The type of fast Displaced Moving Average used.', examples=['SMA', 'EMA', 'WMA', 'DEMA', 'TEMA', 'TRIMA', 'KAMA', 'MAMA', 'T3MA'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesStochRsiRequest(BaseModel):
    """The Stochastic RSI (STOCHRSI) is an oscillator that applies the Stochastic formula to RSI values, providing a more sensitive indicator for identifying overbought or oversold conditions and potential trend reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Specifies the price data type: open, high, low, or close.', examples=['open', 'high', 'low', 'close'])
    rsi_length: int = Field(default=14, description='Length of period for calculating the RSI component. Takes values in the range from `1` to `800`')
    stoch_length: int = Field(default=14, description='Period length for computing the stochastic oscillator of the RSI. Takes values in the range from `1` to `800`')
    k_period: int = Field(default=3, description='Period for smoothing the %K line. Takes values in the range from `1` to `800`')
    slow_kma_type: Optional[str] = Field(default=None)
    d_period: int = Field(default=3, description='Period for smoothing the %D line, which is a moving average of %K. Takes values in the range from `1` to `800`')
    slow_dma_type: Optional[str] = Field(default=None)
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetStocksRequest(BaseModel):
    """This API call returns an array of symbols available at Twelve Data API. This list is updated daily."""
    symbol: Optional[str] = Field(default=None, description='The ticker symbol of an instrument for which data is requested', examples=['AAPL'])
    figi: Optional[str] = Field(default=None, description='Filter by financial instrument global identifier (FIGI)', examples=['BBG000B9Y5X2'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='Filter by exchange name', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Filter by market identifier code (MIC) under ISO 10383 standard', examples=['XNGS'])
    country: Optional[str] = Field(default=None, description='Filter by country name or alpha code, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Index', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    show_plan: bool = Field(default=False, description='Adds info on which plan symbol is available')
    include_delisted: bool = Field(default=False, description='Include delisted identifiers')
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesSubRequest(BaseModel):
    """The Subtraction (SUB) indicator performs arithmetic subtraction of two input data series, typically used to combine or normalize multiple technical indicators or price data."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type_1: str = Field(default='open', description='Price type used as the first part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    series_type_2: str = Field(default='close', description='Price type used as the second part of technical indicator', examples=['close', 'open', 'high', 'low', 'volume'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesSumRequest(BaseModel):
    """The Summation (SUM) indicator calculates the sum of a data series over a specified period, often used in conjunction with other indicators for data analysis or calculation purposes."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesSuperTrendRequest(BaseModel):
    """The Supertrend indicator is a trend-following tool that uses a combination of price, time, and volatility to generate potential entry and exit points in trending markets."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    period: int = Field(default=10, description='The period used for calculation in the indicator. Takes values in the range from `1` to `800`')
    multiplier: int = Field(default=3, description="The factor used to adjust the indicator's sensitivity.")
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesSuperTrendHeikinAshiCandlesRequest(BaseModel):
    """The Supertrend and Heikin Ashi Candles indicator delivers a combined analysis of trend-following signals and smoothed price action visualization, widely used for identifying trends and potential entry or exit points in trading. Available starting from the `Grow` plan."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    period: int = Field(default=10, description='The period used for calculation in the indicator. Takes values in the range from `1` to `800`')
    multiplier: int = Field(default=3, description="The factor used to adjust the indicator's sensitivity.")
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetSymbolSearchRequest(BaseModel):
    """This method helps to find the best matching symbol. It can be used as the base for custom lookups. The response is returned in descending order, with the most relevant instrument at the beginning."""
    symbol: str = Field(..., description='Symbol to search. Supports:\n<ul>\n<li>Ticker symbol of instrument.</li>\n<li>International securities identification number (ISIN).\n<li>Financial instrument global identifier (FIGI).\n<li>Composite FIGI.</li>\n<li>Share Class FIGI.</li>\n</ul>', examples=['AAPL'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    show_plan: bool = Field(default=False, description='Adds info on which plan symbol is available.')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesT3maRequest(BaseModel):
    """The Triple Exponential Moving Average (T3MA) is a smoothing indicator that applies three exponential moving averages to price data, reducing lag and providing a more accurate representation of trends."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    v_factor: float = Field(default=0.7, description="The factor used to adjust the indicator's volatility. Takes values in the range from `0` to `1`")
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTaxInfoRequest(BaseModel):
    """This API endpoint returns tax information for the instrument. Available starting from the `Ultra` plan."""
    symbol: str = Field(..., description='The ticker symbol of an instrument for which data is requested, e.g., `SKYQ`, `AIRE`, `ALM:BME`, `HSI:HKEX`.', examples=['SKYQ'])
    isin: Optional[str] = Field(default=None, description='The ISIN of an instrument for which data is requested', examples=['US5949181045'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG019XJT9D6'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    exchange: Optional[str] = Field(default=None, description='The exchange name where the instrument is traded, e.g., `Nasdaq`, `Euronext`', examples=['Nasdaq'])
    mic_code: Optional[str] = Field(default=None, description='The Market Identifier Code (MIC) of the exchange where the instrument is traded, e.g., `XNAS`, `XLON`', examples=['XNAS'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTechnicalIndicatorsRequest(BaseModel):
    """This API call returns an array of objects with available technical indicators. This endpoint might be used to build an abstract interface to make more convenient API calls from the application."""
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesTemaRequest(BaseModel):
    """The Triple Exponential Moving Average (TEMA) is a smoothing indicator that applies three exponential moving averages to price data, reducing lag and providing a more accurate representation of trends."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='The time period used for calculation in the indicator. Default is 9.')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesRequest(BaseModel):
    """This API call returns meta and time series for the requested instrument. Metaobject consists of general information about the requested symbol. Time series is the array of objects ordered by time descending with Open, High, Low, Close prices. Non-currency instruments also include volume information."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesCrossRequest(BaseModel):
    """This API call returns meta and cross rate time series for the requested instruments. Metaobject consists of general information about the requested symbol. Time series is the array of objects ordered by time descending with Open, High, Low, Close prices. Works with stocks, forex and cryptocurrency."""
    base: str = Field(..., description='Base currency symbol', examples=['JPY'])
    base_type: Optional[str] = Field(default=None, description='Base instrument type according to the `/instrument_type` endpoint', examples=['Physical Currency'])
    base_exchange: Optional[str] = Field(default=None, description='Base exchange', examples=['Binance'])
    base_mic_code: Optional[str] = Field(default=None, description='Base MIC code', examples=['XNGS'])
    quote: str = Field(..., description='Quote currency symbol', examples=['BTC'])
    quote_type: Optional[str] = Field(default=None, description='Quote instrument type according to the `/instrument_type` endpoint', examples=['Digital Currency'])
    quote_exchange: Optional[str] = Field(default=None, description='Quote exchange', examples=['Coinbase'])
    quote_mic_code: Optional[str] = Field(default=None, description='Quote MIC code', examples=['XNYS'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    format: str = Field(default='JSON', description='Format of the response data', examples=['JSON', 'JSON', 'CSV'])
    delimiter: str = Field(default=';', description='Delimiter used in CSV file', examples=[';'])
    prepost: bool = Field(default=False, description='Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume.')
    start_date: Optional[str] = Field(default=None, description='Start date for the time series data', examples=['2025-01-01'])
    end_date: Optional[str] = Field(default=None, description='End date for the time series data', examples=['2025-01-31'])
    adjust: bool = Field(default=True, description='Specifies if there should be an adjustment')
    dp: int = Field(default=5, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.', examples=[5])
    timezone: Optional[str] = Field(default=None, description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a>.</li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>', examples=['UTC'])
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesTRangeRequest(BaseModel):
    """The True Range (TRANGE) indicator measures the range of price movement over a specified period, providing a volatility measure that helps traders identify potential trading opportunities and risk management."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesTrimaRequest(BaseModel):
    """The Triangular Moving Average (TRIMA) is a smoothing indicator that calculates the average price of a security over a specified period, weighting prices in the middle of the range more heavily, providing a balanced view of price action."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesTsfRequest(BaseModel):
    """The Time Series Forecast (TSF) indicator projects future price levels based on linear regression analysis, helping traders identify potential support, resistance, and trend direction."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesTypPriceRequest(BaseModel):
    """The Typical Price (TYPPRICE) indicator calculates the average of a security's high, low, and close prices, providing a simplified view of price action."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesUltOscRequest(BaseModel):
    """The Ultimate Oscillator (ULTOSC) is a momentum oscillator that combines short, intermediate, and long-term price action to identify potential overbought or oversold conditions and trend reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period_1: int = Field(default=7, description='The first time period used for calculation in the indicator. Takes values in the range from `1` to `800`')
    time_period_2: int = Field(default=14, description='The second time period used for calculation in the indicator. Takes values in the range from `1` to `800`')
    time_period_3: int = Field(default=28, description='The third time period used for calculation in the indicator. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesVarRequest(BaseModel):
    """The Variance (VAR) indicator measures the dispersion of a data series from its mean, often used as a volatility indicator to identify potential trading opportunities and risk management."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesVwapRequest(BaseModel):
    """The Volume Weighted Average Price (VWAP) indicator offers an insightful measure of the average trading price weighted by volume, commonly used for trading analysis and execution evaluation."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    sd_time_period: int = Field(default=0, description='The time period for the standard deviation calculation. Must be greater than `0`. Recommended value is `9`.\nThis parameter is only used together with `sd`.')
    sd: float = Field(default=0, description='The standard deviation applied in the calculation. Must be greater than `0`. Recommended value is `2`.\nThis parameter is only used together with `sd_time_period`.')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesWclPriceRequest(BaseModel):
    """The Weighted Close Price (WCLPRICE) indicator calculates the average of a security's high, low, and close prices, with extra weight given to the close price, providing a balanced view of price action."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesWillRRequest(BaseModel):
    """The Williams %R (WILLR) is a momentum oscillator that measures the level of a security's closing price in relation to the high and low range over a specified period, helping traders identify potential overbought or oversold conditions and trend reversals."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    time_period: int = Field(default=14, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])

class GetTimeSeriesWmaRequest(BaseModel):
    """The Weighted Moving Average (WMA) is a smoothing indicator that calculates the average price of a security over a specified period, with more weight given to recent prices, providing a more responsive view of price action."""
    symbol: str = Field(..., description='Symbol ticker of the instrument. E.g. `AAPL`, `EUR/USD`, `ETH/BTC`, ...', examples=['AAPL'])
    isin: Optional[str] = Field(default=None, description='Filter by international securities identification number (ISIN)', examples=['US0378331005'])
    figi: Optional[str] = Field(default=None, description='The FIGI of an instrument for which data is requested', examples=['BBG01293F5X4'])
    cusip: Optional[str] = Field(default=None, description='The CUSIP of an instrument for which data is requested', examples=['594918104'])
    interval: str = Field(default='1day', description='Interval between two consecutive points in time series', examples=['1min', '1min', '5min', '15min', '30min', '45min', '1h', '2h', '4h', '5h', '1day', '1week', '1month'])
    outputsize: int = Field(default=10, description='Number of data points to retrieve. Supports values in the range from `1` to `5000`. Default `10` when no date parameters are set, otherwise set to maximum', examples=[10])
    exchange: Optional[str] = Field(default=None, description='Exchange where instrument is traded', examples=['NASDAQ'])
    mic_code: Optional[str] = Field(default=None, description='Market Identifier Code (MIC) under ISO 10383 standard', examples=['XNAS'])
    country: Optional[str] = Field(default=None, description='The country where the instrument is traded, e.g., `United States` or `US`', examples=['United States'])
    type: Optional[str] = Field(default=None, description='The asset class to which the instrument belongs', examples=['Common Stock', 'American Depositary Receipt', 'Bond', 'Bond Fund', 'Closed-end Fund', 'Common Stock', 'Depositary Receipt', 'Digital Currency', 'ETF', 'Exchange-Traded Note', 'Global Depositary Receipt', 'Limited Partnership', 'Mutual Fund', 'Physical Currency', 'Preferred Stock', 'REIT', 'Right', 'Structured Product', 'Trust', 'Unit', 'Warrant'])
    timezone: str = Field(default='Exchange', description='Timezone at which output datetime will be displayed. Supports:\n<ul>\n<li>1. <code>Exchange</code> for local exchange time</li>\n<li>2. <code>UTC</code> for datetime at universal UTC standard</li>\n<li>3. Timezone name according to the IANA Time Zone Database. E.g. <code>America/New_York</code>, <code>Asia/Singapore</code>. Full list of timezones can be found <a href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones" target="blank">here</a></li>\n</ul>\n<i>Take note that the IANA Timezone name is case-sensitive</i>')
    start_date: Optional[str] = Field(default=None, description='Can be used separately and together with `end_date`. Format `2006-01-02` or `2006-01-02 15:04:05`\n\nDefault location:\n<ul>\n<li>Forex and Cryptocurrencies - <code>UTC</code></li>\n<li>Stocks - where exchange is located (e.g. for AAPL it will be <code>America/New_York</code>)</li>\n</ul>\nBoth parameters take into account if <code>timezone</code> parameter is provided.<br/>\nIf <code>timezone</code> is given then, <code>start_date</code> and <code>end_date</code> will be used in the specified location\n\nExamples:\n<ul>\n<li>1. <code>&symbol=AAPL&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 New York time up to current date</li>\n<li>2. <code>&symbol=EUR/USD&timezone=Asia/Singapore&start_date=2019-08-09 15:50:00&…</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Singapore time up to current date</li>\n<li>3. <code>&symbol=ETH/BTC&timezone=Europe/Zurich&start_date=2019-08-09 15:50:00&end_date=2019-08-09 15:55:00&...</code><br/>\nReturns all records starting from 2019-08-09 15:50:00 Zurich time up to 2019-08-09 15:55:00</li>\n</ul>', examples=['2024-08-22 15:04:05'])
    end_date: Optional[str] = Field(default=None, description='The ending date and time for data selection, see `start_date` description for details.', examples=['2024-08-22 16:04:05'])
    date: Optional[str] = Field(default=None, description='Specifies the exact date to get the data for. Could be the exact date, e.g. `2021-10-27`, or in human language `today` or `yesterday`', examples=['2021-10-27'])
    order: str = Field(default='desc', description='Sorting order of the output', examples=['asc', 'desc'])
    prepost: bool = Field(default=False, description='Returns quotes that include pre-market and post-market data. Only for `Pro` and above plans.\nAvailable at the `1min`, `5min`, `15min`, and `30min` intervals for all US equities.\nOpen, high, low, close values are supplied without volume')
    format: str = Field(default='JSON', description='The format of the response data', examples=['JSON', 'CSV'])
    delimiter: str = Field(default=';', description='The separator used in the CSV response data')
    dp: int = Field(default=-1, description='Specifies the number of decimal places for floating values.\nShould be in range [0, 11] inclusive.\nBy default, the number of decimal places is automatically determined based on the values provided')
    previous_close: bool = Field(default=False, description='A boolean parameter to include the previous closing price in the time_series data. If true, adds previous bar close price value to the current object')
    adjust: str = Field(default='splits', description='Adjusting mode for prices', examples=['all', 'splits', 'dividends', 'none'])
    series_type: str = Field(default='close', description='Price type on which technical indicator is calculated', examples=['close', 'open', 'high', 'low', 'volume'])
    time_period: int = Field(default=9, description='Number of periods to average over. Takes values in the range from `1` to `800`')
    include_ohlc: bool = Field(default=False, description='Specify if OHLC values should be added in the output')
    apikey: str = Field(default='demo', description='API key', examples=['demo'])