"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.invokeOpenApiGenerator = exports.NonClientGeneratorDirectory = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const util_1 = require("projen/lib/util");
/**
 * Enum for generator directories for non-client generators
 */
var NonClientGeneratorDirectory;
(function (NonClientGeneratorDirectory) {
    NonClientGeneratorDirectory["DOCS"] = "docs";
})(NonClientGeneratorDirectory = exports.NonClientGeneratorDirectory || (exports.NonClientGeneratorDirectory = {}));
const serializeProperties = (properties) => Object.entries(properties)
    .map(([key, value]) => `${key}=${value}`)
    .join(",");
/**
 * Clean up any files that have already been generated by the openapi generator
 * @param generatedOutputPath output path for the generator
 */
const cleanPreviouslyGeneratedFiles = (generatedOutputPath) => {
    // OpenAPI generator writes a manifest called FILES which lists the files it generated.
    const openApiGeneratedFilesManifestPath = path.join(generatedOutputPath, ".openapi-generator", "FILES");
    // If the manifest exists, delete the files it lists
    if (fs.existsSync(openApiGeneratedFilesManifestPath)) {
        const previouslyGeneratedFiles = new Set(fs
            .readFileSync(openApiGeneratedFilesManifestPath, { encoding: "utf-8" })
            .split("\n")
            .filter((x) => x));
        previouslyGeneratedFiles.forEach((previouslyGeneratedFile) => {
            fs.unlinkSync(path.join(generatedOutputPath, previouslyGeneratedFile));
        });
    }
};
/**
 * Generate client code or docs by invoking the root generate script
 */
exports.invokeOpenApiGenerator = (options) => {
    // Prior to generating, clean up any previously generated files so that we don't have stale generated code from
    // previous executions.
    cleanPreviouslyGeneratedFiles(options.outputPath);
    const additionalProperties = options.additionalProperties
        ? ` --additional-properties "${serializeProperties(options.additionalProperties)}"`
        : "";
    util_1.exec(`./generate --generator ${options.generator} --spec-path ${options.specPath} --output-path ${options.outputPath} --generator-dir ${options.generatorDirectory}${additionalProperties}`, {
        cwd: path.resolve(__dirname, "..", "..", "..", "..", "scripts", "generators"),
    });
};
//# sourceMappingURL=data:application/json;base64,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