# -*- coding: utf-8 -*-
from collections import OrderedDict

import six

from ..utils.internal import QueryPath
from .base import QueryableProperty
from .mixins import AnnotationGetterMixin


class InheritanceModelProperty(AnnotationGetterMixin, QueryableProperty):
    """
    A property that returns information about the actual model class of objects
    in inheritance scenarios.
    """

    #: A shared cache that maps holds a dictionary per model class. The
    #: dictionaries contain child model classes as keys and their corresponding
    #: query paths as values.
    _child_paths = {}

    def __init__(self, value_generator, output_field, depth=None, **kwargs):
        """
        Initialize a new property that returns information about the actual
        model class of objects in inheritance scenarios.

        :param value_generator: A callable that returns the actual value to be
                                represented for a given model class. Must take
                                a model class as its first argument.
        :type value_generator: function
        :param output_field: The output field to use for this property, which
                             must be able to hold the values generated by
                             the given value generator.
        :type output_field: django.db.models.Field
        :param depth: The maximum depth of the inheritance hierarchy to follow.
                      Instances of model classes below this maximum depth will
                      be treated as objects of the maximum depth. If not
                      provided, no maximum depth will be enforced.
        :type depth: int | None
        """
        self.value_generator = value_generator
        self.output_field = output_field
        self.depth = depth
        super(InheritanceModelProperty, self).__init__(**kwargs)

    def _get_child_paths(self, model):
        """
        Get a dictionary containg child model classes and their respective
        query paths for the given model.

        :param type model: The model to get the child paths for.
        :return: A dictionary containg child model classes as keys and their
                 respective query paths as values.
        :rtype: OrderedDict[type, QueryPath]
        """
        model = model._meta.proxy_for_model or model
        child_paths = self._child_paths.get(model)
        if child_paths is None:
            from django.db.models.fields.related import ForeignObjectRel

            child_paths = OrderedDict()
            for field in model._meta.get_fields(include_parents=False, include_hidden=False):
                if isinstance(field, ForeignObjectRel) and field.parent_link:
                    path = QueryPath(field.name)
                    for sub_model, sub_path in six.iteritems(self._get_child_paths(field.related_model)):
                        child_paths[sub_model] = path + sub_path
                    child_paths[field.related_model] = path
            self._child_paths[model] = child_paths
        return child_paths

    def get_annotation(self, cls):
        from django.db.models import Case, Value, When

        cases = (
            When((query_path + 'isnull').build_filter(False), then=Value(self.value_generator(child_model)))
            for child_model, query_path in six.iteritems(self._get_child_paths(cls))
            if self.depth is None or len(query_path) <= self.depth
        )
        return Case(*cases, default=Value(self.value_generator(cls)), output_field=self.output_field)
