# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['timezone_field']

package_data = \
{'': ['*']}

install_requires = \
['pytz']

extras_require = \
{':python_version < "3.9"': ['backports.zoneinfo>=0.2.1,<0.3.0']}

setup_kwargs = {
    'name': 'django-timezone-field',
    'version': '5.0',
    'description': 'A Django app providing DB, form, and REST framework fields for zoneinfo and pytz timezone objects.',
    'long_description': '# django-timezone-field\n\n[![CI](https://github.com/mfogel/django-timezone-field/actions/workflows/ci.yml/badge.svg?branch=main)](https://github.com/mfogel/django-timezone-field/actions)\n[![codecov](https://codecov.io/gh/mfogel/django-timezone-field/branch/main/graph/badge.svg?token=Rwekzmim3l)](https://codecov.io/gh/mfogel/django-timezone-field)\n[![pypi downloads](https://img.shields.io/pypi/dm/django-timezone-field.svg)](https://pypi.python.org/pypi/django-timezone-field/)\n[![pypi python support](https://img.shields.io/pypi/pyversions/django-timezone-field.svg)](https://pypi.python.org/pypi/django-timezone-field/)\n[![pypi django support](https://img.shields.io/pypi/djversions/django-timezone-field.svg)](https://pypi.python.org/pypi/django-timezone-field/)\n\nA Django app providing DB, form, and REST framework fields for\n[`zoneinfo`](https://docs.python.org/3/library/zoneinfo.html) and [`pytz`](http://pypi.python.org/pypi/pytz/) timezone\nobjects.\n\n## The transition from `pytz` to `zoneinfo`\n\nLike Django, this app supports both `pytz` and `zoneinfo` objects while the community transitions away from `pytz` to\n`zoneinfo`. All exposed fields and functions that return a timezone object accept an optional boolean kwarg `use_pytz`.\n\nIf not explicitly specified, the default value used for `use_pytz` matches Django\'s behavior:\n\n- Django <= 3.X: `use_pytz` defaults to `True`\n- Django == 4.X: `use_pytz` defaults to the value of\n  [`django.conf.settings.USE_DEPRECATED_PYTZ`](https://docs.djangoproject.com/en/4.0/ref/settings/#use-deprecated-pytz),\n  which itself defaults to `False`\n- Django >= 5.X: django plans to\n  [drop support for `pytz` altogether](https://docs.djangoproject.com/en/4.0/releases/4.0/#zoneinfo-default-timezone-implementation),\n  and this app will likely do the same.\n\nWhen switching between `pytz` and `zoneinfo`, in general a\n[data migration](https://docs.djangoproject.com/en/4.0/topics/migrations/#data-migrations) is _not_ needed, as both\nlibraries recognize the same set of strings as valid timezones. Exceptions to that include if your local system has an\nunusual set of time zones installed, or if you are using the `Factory` timezone which `zoneinfo` recognizes but `pytz`\ndoes not.\n\n## Examples\n\n### Database Field\n\n```python\nimport zoneinfo\nimport pytz\nfrom django.db import models\nfrom timezone_field import TimeZoneField\n\nclass MyModel(models.Model):\n    tz1 = TimeZoneField(default="Asia/Dubai")               # defaults supported, in ModelForm renders like "Asia/Dubai"\n    tz2 = TimeZoneField(choices_display="WITH_GMT_OFFSET")  # in ModelForm renders like "GMT+04:00 Asia/Dubai"\n    tz3 = TimeZoneField(use_pytz=True)                      # returns pytz timezone objects\n    tz4 = TimeZoneField(use_pytz=False)                     # returns zoneinfo objects\n\nmy_model = MyModel(\n    tz2="America/Vancouver",                     # assignment of a string\n    tz3=pytz.timezone("America/Vancouver"),      # assignment of a pytz timezone\n    tz4=zoneinfo.ZoneInfo("America/Vancouver"),  # assignment of a zoneinfo\n)\nmy_model.full_clean() # validates against pytz.common_timezones by default\nmy_model.save()       # values stored in DB as strings\nmy_model.tz3          # value returned as pytz timezone: <DstTzInfo \'America/Vancouver\' LMT-1 day, 15:48:00 STD>\nmy_model.tz4          # value returned as zoneinfo: zoneinfo.ZoneInfo(key=\'America/Vancouver\')\n```\n\n### Form Field\n\n```python\nfrom django import forms\nfrom timezone_field import TimeZoneFormField\n\nclass MyForm(forms.Form):\n    tz1 = TimeZoneFormField()                                   # renders like "Asia/Dubai"\n    tz2 = TimeZoneFormField(choices_display="WITH_GMT_OFFSET")  # renders like "GMT+04:00 Asia/Dubai"\n    tz3 = TimeZoneFormField(use_pytz=True)                      # returns pytz timezone objects\n    tz4 = TimeZoneFormField(use_pytz=False)                     # returns zoneinfo objects\n\nmy_form = MyForm({"tz3": "Europe/Berlin", "tz4": "Europe/Berlin"})\nmy_form.full_clean()         # validates against pytz.common_timezones by default\nmy_form.cleaned_data["tz3"]  # value returned as pytz timezone: <DstTzInfo \'Europe/Berlin\' LMT+0:53:00 STD>\nmy_form.cleaned_data["tz4"]  # value returned as zoneinfo: zoneinfo.ZoneInfo(key=\'Europe/Berlin\')\n```\n\n### REST Framework Serializer Field\n\n```python\nfrom rest_framework import serializers\nfrom timezone_field.rest_framework import TimeZoneSerializerField\n\nclass MySerializer(serializers.Serializer):\n    tz1 = TimeZoneSerializerField(use_pytz=True)\n    tz2 = TimeZoneSerializerField(use_pytz=False)\n\nmy_serializer = MySerializer(data={\n    "tz1": "America/Argentina/Buenos_Aires",\n    "tz2": "America/Argentina/Buenos_Aires",\n})\nmy_serializer.is_valid()\nmy_serializer.validated_data["tz1"]  # <DstTzInfo \'America/Argentina/Buenos_Aires\' LMT-1 day, 20:06:00 STD>\nmy_serializer.validated_data["tz2"]  # zoneinfo.ZoneInfo(key=\'America/Argentina/Buenos_Aires\')\n```\n\n## Installation\n\nReleases are hosted on [`pypi`](https://pypi.org/project/django-timezone-field/) and can be installed using various\npython packaging tools.\n\n```bash\n# with poetry\npoetry add django-timezone-field\n\n# with pip\npip install django-timezone-field\n```\n\n## Running the tests\n\nFrom the repository root, with [`poetry`](https://python-poetry.org/):\n\n```bash\npoetry install\npoetry run pytest\n```\n\n## Changelog\n\n#### 5.0 (2022-02-08)\n\n- Add support for `zoneinfo` objects ([#79](https://github.com/mfogel/django-timezone-field/issues/79))\n- Add support for django 4.0\n- Remove `display_GMT_offset` kwarg (use `choices_display` instead)\n- Drop support for django 3.0, 3.1\n- Drop support for python 3.5, 3.6\n\n#### 4.2.3 (2022-01-13)\n\n- Fix sdist installs ([#78](https://github.com/mfogel/django-timezone-field/issues/78))\n- Officially support python 3.10\n\n#### 4.2.1 (2021-07-07)\n\n- Reinstate `TimeZoneField.default_choices` ([#76](https://github.com/mfogel/django-timezone-field/issues/76))\n\n#### 4.2 (2021-07-07)\n\n- Officially support django 3.2, python 3.9\n- Fix bug with field deconstruction ([#74](https://github.com/mfogel/django-timezone-field/issues/74))\n- Housekeeping: use poetry, github actions, pytest\n\n#### 4.1.2 (2021-03-17)\n\n- Avoid `NonExistentTimeError` during DST transition ([#70](https://github.com/mfogel/django-timezone-field/issues/70))\n\n#### 4.1.1 (2020-11-28)\n\n- Don\'t import `rest_framework` from package root ([#67](https://github.com/mfogel/django-timezone-field/issues/67))\n\n#### 4.1 (2020-11-28)\n\n- Add Django REST Framework serializer field\n- Add new `choices_display` kwarg with supported values `WITH_GMT_OFFSET` and `STANDARD`\n- Deprecate `display_GMT_offset` kwarg\n\n#### 4.0 (2019-12-03)\n\n- Add support for django 3.0, python 3.8\n- Drop support for django 1.11, 2.0, 2.1, python 2.7, 3.4\n\n#### 3.1 (2019-10-02)\n\n- Officially support django 2.2 (already worked)\n- Add option to display TZ offsets in form field ([#46](https://github.com/mfogel/django-timezone-field/issues/46))\n\n#### 3.0 (2018-09-15)\n\n- Support django 1.11, 2.0, 2.1\n- Add support for python 3.7\n- Change default human-readable timezone names to exclude underscores\n  ([#32](https://github.com/mfogel/django-timezone-field/issues/32) &\n  [#37](https://github.com/mfogel/django-timezone-field/issues/37))\n\n#### 2.1 (2018-03-01)\n\n- Add support for django 1.10, 1.11\n- Add support for python 3.6\n- Add wheel support\n- Support bytes in DB fields ([#38](https://github.com/mfogel/django-timezone-field/issues/38) &\n  [#39](https://github.com/mfogel/django-timezone-field/issues/39))\n\n#### 2.0 (2016-01-31)\n\n- Drop support for django 1.7, add support for django 1.9\n- Drop support for python 3.2, 3.3, add support for python 3.5\n- Remove tests from source distribution\n\n#### 1.3 (2015-10-12)\n\n- Drop support for django 1.6, add support for django 1.8\n- Various [bug fixes](https://github.com/mfogel/django-timezone-field/issues?q=milestone%3A1.3)\n\n#### 1.2 (2015-02-05)\n\n- For form field, changed default list of accepted timezones from `pytz.all_timezones` to `pytz.common_timezones`, to\n  match DB field behavior.\n\n#### 1.1 (2014-10-05)\n\n- Django 1.7 compatibility\n- Added support for formatting `choices` kwarg as `[[<str>, <str>], ...]`, in addition to previous format of\n  `[[<pytz.timezone>, <str>], ...]`.\n- Changed default list of accepted timezones from `pytz.all_timezones` to `pytz.common_timezones`. If you have timezones\n  in your DB that are in `pytz.all_timezones` but not in `pytz.common_timezones`, this is a backward-incompatible\n  change. Old behavior can be restored by specifying `choices=[(tz, tz) for tz in pytz.all_timezones]` in your model\n  definition.\n\n#### 1.0 (2013-08-04)\n\n- Initial release as `timezone_field`.\n\n## Credits\n\nOriginally adapted from [Brian Rosner\'s django-timezones](https://github.com/brosner/django-timezones).\n\nMade possible thanks to the work of the\n[contributors](https://github.com/mfogel/django-timezone-field/graphs/contributors).\n',
    'author': 'Mike Fogel',
    'author_email': 'mike@fogel.ca',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mfogel/django-timezone-field/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
