"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const typescript_1 = require("projen/lib/typescript");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: false,
            jest: false,
            package: false,
            prettier: true,
            projenrcTs: true,
            release: false,
            sampleCode: false,
            defaultReleaseBranch: "mainline",
        });
        this.implicitDependencies = {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(options.nxIgnorePatterns || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(options.targetDependencies || {}),
                },
                affected: {
                    defaultBase: "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    // Remove this hack once subProjects is made public in Projen
    get subProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    preSynthesize() {
        super.preSynthesize();
        // Add workspaces for each subproject
        this.package.addField("workspaces", this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)));
        // Disable default task on subprojects as this isn't supported in a monorepo
        this.subProjects.forEach((subProject) => { var _b; return (_b = subProject.defaultTask) === null || _b === void 0 ? void 0 : _b.reset(); });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Check to see if a new subProject was added
        const newSubProject = this.subProjects.find((subProject) => !fs.existsSync(subProject.outdir));
        // Need to synth before generating the package.json otherwise the subdirectory won't exist
        newSubProject && super.synth();
        this.subProjects
            .filter((subProject) => !fs.existsSync(`${subProject.outdir}/package.json`) ||
            JSON.parse(fs.readFileSync(`${subProject.outdir}/package.json`).toString()).__pdk__)
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            manifest.name = subProject.name;
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
        super.synth();
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.pdk_projen.NxMonorepoProject", version: "0.0.64" };
function getPluginPath() {
    const prefix = process.env.NODE_ENV === "test"
        ? "./lib/" // If unit test from within this package
        : "./node_modules/aws-prototyping-sdk/lib/";
    return `${prefix}pdk_projen/nx_monorepo/plugin/nx-monorepo-plugin.js`;
}
//# sourceMappingURL=data:application/json;base64,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