# Stores all queries using simple key-value system, where the value is a Query
# instance. Each key is either user-specified name (e.g. "my_awesome_query")
# or if user doesn't specify a name, then an autogenerated name (e.g. "Q20").
# This is a singleton class implemented using a module (rather than a class).

from string import uppercase
from quest.query.query import Query

class TooManyQueryVariables(Exception):
    def __init__(self, value):
        self.value = value

# The actual cache dict. query_cache is just a wrapper around this.
cache = {}
# counter_dict tracks the current "generation" of a query variable, e.g.
# if we're in the 3rd generation of Q, then counter_dict["Q"] == 3. Each
# time an operation is applied to a query, that creates a new
# generation, so e.g. Q4 == Q3.rollup(<predicate>)
# Each time INITIALIZE is called, adds a new key with the counter set to
# 0.
counter_dict = {}
uppercase_list = list(uppercase)
# most_recent_key_query is set whenever put() is called. It is a tuple
# of (key, query) like ("A", <query object>)
most_recent_key_and_query = (None, None)

def increment_counter_for(variable):
    """Increment the counter for a given variable name. Should be called each
    time a new generation of a query is created (Q1.rollup(pred) -> Q2).
    """
    global counter_dict
    counter_dict[variable] += 1

def next_variable_name(letters = uppercase_list):
    """Searches _cache_ to get the next unused variable name. Checks
    "A"-"Z", then "AA"-"ZZ", etc., up to "ZZZ" inclusive, and uses first letter that hasn't
    been used yet. The `letters` argument allows for recursive calling with
    different sets of letters.
    """
    global counter_dict
    if len(letters[0]) == 4:
        # We've reached "AAAA", stop.
        raise TooManyQueryVariables("Too many query variables!")

    next_variable_name = ""
    cache_keys = cache.keys()
    unused_letters = [letter for letter in letters if letter not in cache_keys]
    if len(unused_letters) == 0:
        # All letters are used, recurse.
        # Each letter in letters grows by one, so "A" -> "AA", "B" -> "BB", etc.
        new_letters = [letter + letter[0] for letter in letters]
        next_variable_name = next_variable_name(new_letters)
    else:
        # There are unused letters, take the first one
        next_variable_name = unused_letters[0]
        # Initialize counter for new variable name to 0
        counter_dict[next_variable_name] = 0
    return next_variable_name

def put(key, query):
    """Put a (key, query) pair in the cache. If key is None, autogenerates a
    variable name using next_variable_name. Returns a (key, query) tuple.

    Autoconverts query to a Query instance, if it isn't already.
    """
    global most_recent_key_and_query
    global cache
    if key is None:
        key = next_variable_name()
    if not isinstance(query, Query):
        # Set the query variable (which is a string) to an instance of
        # the Query class initialized with the value of the query
        # variable.
        query = Query(query)
    cache[key] = query
    most_recent_key_and_query = (key, query)
    return most_recent_key_and_query

def get(key):
    """Returns the query associated with the given key. Behaves exactly like a
    dict (since that's what it uses) if the key is not in the cache. That is, it
    raises a KeyError.
    """
    return cache[key]

def delete(key):
    """Delete the key from the cache. Raises a KeyError (exactly like a dict)
    if the key is not in the cache. Returns the query associated with the key.
    """
    global cache
    query = get(key)
    del cache[key]
    return query
