"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
class FakeTarget {
    constructor() {
        this.connections = new aws_ec2_1.Connections({
            securityGroupRule: new aws_ec2_1.CidrIPv4('666.666.666.666/666')
        });
    }
    attachToClassicLB(_loadBalancer) {
        // Nothing to do. Normally we set a property on ourselves so
        // our instances know to bind to the LB on startup.
    }
}
module.exports = {
    'test specifying nonstandard port works'(test) {
        const stack = new cdk_1.Stack(undefined, undefined, { env: { account: '1234', region: 'test' } });
        stack.node.setContext('availability-zones:1234:test', ['test-1a', 'test-1b']);
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        const lb = new lib_1.LoadBalancer(stack, 'LB', { vpc });
        lb.addListener({
            externalProtocol: lib_1.LoadBalancingProtocol.Http,
            externalPort: 8080,
            internalProtocol: lib_1.LoadBalancingProtocol.Http,
            internalPort: 8080,
        });
        assert_1.expect(stack).to(assert_1.haveResource("AWS::ElasticLoadBalancing::LoadBalancer", {
            Listeners: [{
                    InstancePort: "8080",
                    InstanceProtocol: "http",
                    LoadBalancerPort: "8080",
                    Protocol: "http"
                }]
        }));
        test.done();
    },
    'add a health check'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            healthCheck: {
                interval: 60,
                path: '/ping',
                protocol: lib_1.LoadBalancingProtocol.Https,
                port: 443,
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::ElasticLoadBalancing::LoadBalancer", {
            HealthCheck: {
                HealthyThreshold: "2",
                Interval: "60",
                Target: "HTTPS:443/ping",
                Timeout: "5",
                UnhealthyThreshold: "5"
            },
        }));
        test.done();
    },
    'add a listener and load balancing target'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        const elb = new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            healthCheck: {
                interval: 60,
                path: '/ping',
                protocol: lib_1.LoadBalancingProtocol.Https,
                port: 443,
            }
        });
        // WHEN
        elb.addListener({ externalPort: 80, internalPort: 8080 });
        elb.addTarget(new FakeTarget());
        // THEN: at the very least it added a security group rule for the backend
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    Description: 'Port 8080 LB to fleet',
                    CidrIp: "666.666.666.666/666",
                    FromPort: 8080,
                    IpProtocol: "tcp",
                    ToPort: 8080
                }
            ],
        }));
        test.done();
    },
    'enable cross zone load balancing'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            crossZone: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            CrossZone: true
        }));
        test.done();
    },
    'disable cross zone load balancing'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            crossZone: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            CrossZone: false
        }));
        test.done();
    },
    'cross zone load balancing enabled by default'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            CrossZone: true
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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