"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const cdk_1 = require("@aws-cdk/cdk");
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["Tcp"] = "tcp";
    LoadBalancingProtocol["Ssl"] = "ssl";
    LoadBalancingProtocol["Http"] = "http";
    LoadBalancingProtocol["Https"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener
 *
 * Routes to a fleet of of instances in a VPC.
 */
class LoadBalancer extends cdk_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const subnets = props.internetFacing ? props.vpc.publicSubnets : props.vpc.privateSubnets;
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: subnets.map(s => s.subnetId),
            listeners: cdk_1.Lazy.anyValue({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (props.crossZone === undefined || props.crossZone) ? true : false
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(...subnets.map(s => s.internetConnectivityEstablished));
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer
     *
     * @returns A ListenerPort object that controls connections to the listener port
     */
    addListener(listener) {
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.Http : LoadBalancingProtocol.Tcp));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: listener.sslCertificateId,
            policyNames: listener.policyNames
        });
        const port = new ListenerPort(this.securityGroup, new aws_ec2_1.TcpPort(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [new aws_ec2_1.AnyIPv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    addTarget(target) {
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @attribute
     */
    get loadBalancerName() {
        return this.elb.refAsString;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @attribute
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, new aws_ec2_1.TcpPort(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *    const listener = elb.addListener(...);
 *
 *    listener.connections.allowDefaultPortFromAnyIPv4();
 *    // or
 *    instance.connections.allowToDefaultPort(listener);
 */
class ListenerPort {
    constructor(securityGroup, defaultPortRange) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPortRange });
    }
}
exports.ListenerPort = ListenerPort;
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.Http;
    }
    if (port === 443) {
        return LoadBalancingProtocol.Https;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.Https || proto === LoadBalancingProtocol.Http;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.Tcp));
    const path = protocol === LoadBalancingProtocol.Http || protocol === LoadBalancingProtocol.Https ? ifUndefined(healthCheck.path, "/") : "";
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: ifUndefined(healthCheck.interval, 30).toString(),
        target,
        timeout: ifUndefined(healthCheck.timeout, 5).toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
//# sourceMappingURL=data:application/json;base64,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