"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoAttributeValue = exports.DynamoProjectionExpression = exports.DynamoReturnValues = exports.DynamoItemCollectionMetrics = exports.DynamoConsumedCapacity = void 0;
const utils_1 = require("./private/utils");
/**
 * Determines the level of detail about provisioned throughput consumption that is returned.
 */
var DynamoConsumedCapacity;
(function (DynamoConsumedCapacity) {
    DynamoConsumedCapacity["INDEXES"] = "INDEXES";
    DynamoConsumedCapacity["TOTAL"] = "TOTAL";
    DynamoConsumedCapacity["NONE"] = "NONE";
})(DynamoConsumedCapacity = exports.DynamoConsumedCapacity || (exports.DynamoConsumedCapacity = {}));
/**
 * Determines whether item collection metrics are returned.
 */
var DynamoItemCollectionMetrics;
(function (DynamoItemCollectionMetrics) {
    DynamoItemCollectionMetrics["SIZE"] = "SIZE";
    DynamoItemCollectionMetrics["NONE"] = "NONE";
})(DynamoItemCollectionMetrics = exports.DynamoItemCollectionMetrics || (exports.DynamoItemCollectionMetrics = {}));
/**
 * Use ReturnValues if you want to get the item attributes as they appear before or after they are changed.
 */
var DynamoReturnValues;
(function (DynamoReturnValues) {
    DynamoReturnValues["NONE"] = "NONE";
    DynamoReturnValues["ALL_OLD"] = "ALL_OLD";
    DynamoReturnValues["UPDATED_OLD"] = "UPDATED_OLD";
    DynamoReturnValues["ALL_NEW"] = "ALL_NEW";
    DynamoReturnValues["UPDATED_NEW"] = "UPDATED_NEW";
})(DynamoReturnValues = exports.DynamoReturnValues || (exports.DynamoReturnValues = {}));
/**
 * Class to generate projection expression.
 */
class DynamoProjectionExpression {
    constructor() {
        this.expression = [];
    }
    /**
     * Adds the passed attribute to the chain.
     *
     * @param attr Attribute name.
     */
    withAttribute(attr) {
        if (this.expression.length) {
            this.expression.push(`.${attr}`);
        }
        else {
            this.expression.push(attr);
        }
        return this;
    }
    /**
     * Adds the array literal access for passed index.
     *
     * @param index array index.
     */
    atIndex(index) {
        if (!this.expression.length) {
            throw new Error('Expression must start with an attribute');
        }
        this.expression.push(`[${index}]`);
        return this;
    }
    /**
     * converts and return the string expression.
     */
    toString() {
        return this.expression.join('');
    }
}
exports.DynamoProjectionExpression = DynamoProjectionExpression;
/**
 * Represents the data for an attribute.
 *
 * Each attribute value is described as a name-value pair.
 * The name is the data type, and the value is the data itself.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_AttributeValue.html
 */
class DynamoAttributeValue {
    constructor(value) {
        this.attributeValue = value;
    }
    /**
     * Sets an attribute of type String.
     *
     * For example:  "S": "Hello"
     * Strings may be literal values or as JsonPath
     */
    static fromString(value) {
        return new DynamoAttributeValue({ S: value });
    }
    /**
     * Sets a literal number.
     *
     * For example: 1234
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    static fromNumber(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * Sets an attribute of type Number.
     *
     * For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     */
    static numberFromString(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * Sets an attribute of type Binary.
     *
     * For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     *
     * @param value base-64 encoded string.
     */
    static fromBinary(value) {
        return new DynamoAttributeValue({ B: value });
    }
    /**
     * Sets an attribute of type String Set.
     *
     * For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     */
    static fromStringSet(value) {
        return new DynamoAttributeValue({ SS: value });
    }
    /**
     * Sets an attribute of type Number Set.
     *
     * For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    static fromNumberSet(value) {
        return new DynamoAttributeValue({ NS: value.map(String) });
    }
    /**
     * Sets an attribute of type Number Set.
     *
     * For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     */
    static numberSetFromStrings(value) {
        return new DynamoAttributeValue({ NS: value });
    }
    /**
     * Sets an attribute of type Binary Set.
     *
     * For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     */
    static fromBinarySet(value) {
        return new DynamoAttributeValue({ BS: value });
    }
    /**
     * Sets an attribute of type Map.
     *
     * For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     */
    static fromMap(value) {
        return new DynamoAttributeValue({ M: utils_1.transformAttributeValueMap(value) });
    }
    /**
     * Sets an attribute of type Map.
     *
     * For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     *
     * @param value Json path that specifies state input to be used.
     */
    static mapFromJsonPath(value) {
        utils_1.validateJsonPath(value);
        return new DynamoAttributeValue({ 'M.$': value });
    }
    /**
     * Sets an attribute of type List.
     *
     * For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     */
    static fromList(value) {
        return new DynamoAttributeValue({ L: value.map((val) => val.toObject()) });
    }
    /**
     * Sets an attribute of type Null.
     *
     * For example:  "NULL": true
     */
    static fromNull(value) {
        return new DynamoAttributeValue({ NULL: value });
    }
    /**
     * Sets an attribute of type Boolean.
     *
     * For example:  "BOOL": true
     */
    static fromBoolean(value) {
        return new DynamoAttributeValue({ BOOL: value });
    }
    /**
     * Sets an attribute of type Boolean from state input through Json path.
     *
     * For example:  "BOOL": true
     *
     * @param value Json path that specifies state input to be used.
     */
    static booleanFromJsonPath(value) {
        utils_1.validateJsonPath(value);
        return new DynamoAttributeValue({ BOOL: value.toString() });
    }
    /**
     * Returns the DynamoDB attribute value.
     */
    toObject() {
        return this.attributeValue;
    }
}
exports.DynamoAttributeValue = DynamoAttributeValue;
//# sourceMappingURL=data:application/json;base64,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