# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['signal_processing_algorithms',
 'signal_processing_algorithms.energy_statistics']

package_data = \
{'': ['*']}

install_requires = \
['more-itertools>=8.2.0,<9.0.0',
 'numpy>=1.18.1,<2.0.0',
 'scipy>=1.3,<2.0',
 'structlog>=19.1,<20.0',
 'typing-extensions>=3.7.4,<4.0.0']

setup_kwargs = {
    'name': 'signal-processing-algorithms',
    'version': '2.0.0',
    'description': 'Signal Processing Algorithms from MongoDB',
    'long_description': '# Signal Processing Algorithms\n\nA suite of algorithms implementing [Energy Statistics](https://en.wikipedia.org/wiki/Energy_distance), \n[E-Divisive with Means](https://arxiv.org/pdf/1306.4933.pdf) and \n[Generalized ESD Test for Outliers](https://www.itl.nist.gov/div898/handbook/eda/section3/eda35h3.htm) in python.\n\n## Getting Started - Users\n```\npip install signal-processing-algorithms\n```\n\n## Getting Started - Developers\n\nGetting the code:\n\n```\n$ git clone git@github.com:mongodb/signal-processing-algorithms.git\n$ cd signal-processing-algorithms\n```\n\nInstallation\n```\n$ pip install poetry\n$ poetry install\n```\nTesting/linting:\n```\n$ poetry run pytest\n```\n\nRunning the slow tests:\n```\n$ poetry run pytest --runslow\n```\n\n**Some of the larger tests can take a significant amount of time (more than 2 hours).**\n\n## Energy statistics\n[Energy Statistics](https://en.wikipedia.org/wiki/Energy_distance) is the statistical concept of Energy Distance \nand can be used to measure how similar/different two distributions are.\n\nFor statistical samples from two random variables X and Y:\nx1, x2, ..., xn and y1, y2, ..., yn\n\nE-Statistic is defined as:\n\n<a href="https://www.codecogs.com/eqnedit.php?latex=E_{n,m}(X,Y):=2A-B-C" target="_blank"><img src="https://latex.codecogs.com/gif.latex?E_{n,m}(X,Y):=2A-B-C" title="E_{n,m}(X,Y):=2A-B-C" /></a>\n\nwhere,\n\n<a href="https://www.codecogs.com/eqnedit.php?latex=A:={\\frac&space;{1}{nm}}\\sum&space;_{i=1}^{n}\\sum&space;_{j=1}^{m}\\|x_{i}-y_{j}\\|,B:={\\frac&space;{1}{n^{2}}}\\sum&space;_{i=1}^{n}\\sum&space;_{j=1}^{n}\\|x_{i}-x_{j}\\|,C:={\\frac&space;{1}{m^{2}}}\\sum&space;_{i=1}^{m}\\sum&space;_{j=1}^{m}\\|y_{i}-y_{j}\\|" target="_blank"><img src="https://latex.codecogs.com/gif.latex?A:={\\frac&space;{1}{nm}}\\sum&space;_{i=1}^{n}\\sum&space;_{j=1}^{m}\\|x_{i}-y_{j}\\|,B:={\\frac&space;{1}{n^{2}}}\\sum&space;_{i=1}^{n}\\sum&space;_{j=1}^{n}\\|x_{i}-x_{j}\\|,C:={\\frac&space;{1}{m^{2}}}\\sum&space;_{i=1}^{m}\\sum&space;_{j=1}^{m}\\|y_{i}-y_{j}\\|" title="A:={\\frac {1}{nm}}\\sum _{i=1}^{n}\\sum _{j=1}^{m}\\|x_{i}-y_{j}\\|,B:={\\frac {1}{n^{2}}}\\sum _{i=1}^{n}\\sum _{j=1}^{n}\\|x_{i}-x_{j}\\|,C:={\\frac {1}{m^{2}}}\\sum _{i=1}^{m}\\sum _{j=1}^{m}\\|y_{i}-y_{j}\\|" /></a>\n\nT-statistic is defined as: \n\n<a href="https://www.codecogs.com/eqnedit.php?latex=T={\\frac&space;{nm}{n&plus;m}}E_{{n,m}}(X,Y)" target="_blank"><img src="https://latex.codecogs.com/gif.latex?T={\\frac&space;{nm}{n&plus;m}}E_{{n,m}}(X,Y)" title="T={\\frac {nm}{n+m}}E_{{n,m}}(X,Y)" /></a>\n\nE-coefficient of inhomogeneity is defined as:\n\n<a href="https://www.codecogs.com/eqnedit.php?latex=H=\\frac{2E||X-Y||&space;-&space;E||X-X\'||&space;-&space;E||Y-Y\'||}{2E||X-Y||}" target="_blank"><img src="https://latex.codecogs.com/gif.latex?H=\\frac{2E||X-Y||&space;-&space;E||X-X\'||&space;-&space;E||Y-Y\'||}{2E||X-Y||}" title="H=\\frac{2E||X-Y|| - E||X-X\'|| - E||Y-Y\'||}{2E||X-Y||}" /></a>\n\n\n```\nfrom signal_processing_algorithms.energy_statistics import energy_statistics\nfrom some_module import series1, series2\n\n# To get Energy Statistics of the distributions.\nes = energy_statistics.get_energy_statistics(series1, series2)\n\n# To get Energy Statistics and permutation test results of the distributions.\nes_with_probabilities = energy_statistics.get_energy_statistics_and_probabilities(series1, series2, permutations=100)\n\n```\n\n## Intro to E-Divisive\n\nDetecting distributional changes in a series of numerical values can be surprisingly difficult. Simple systems based on thresholds or\n mean values can be yield false positives due to outliers in the data, and will fail to detect changes in the noise\n profile of the series you are analyzing.\n \nOne robust way of detecting many of the changes missed by other approaches is to use [E-Divisive with Means](https://arxiv.org/pdf/1306.4933.pdf), an energy\n statistic based approach that compares the expected distance (Euclidean norm) between samples of two portions of the\n series with the expected distance between samples within those portions.\n \nThat is to say, assuming that the two portions can each be modeled as i.i.d. samples drawn from distinct random variables\n (X for the first portion, Y for the second portion), you would expect the E-statistic to be non-zero if there is a\n difference between the two portions: \n \n <a href="https://www.codecogs.com/eqnedit.php?latex=E_{n,m}(X,Y):=2A-B-C" target="_blank"><img src="https://latex.codecogs.com/gif.latex?E_{n,m}(X,Y):=2A-B-C" title="E_{n,m}(X,Y):=2A-B-C" /></a>\n where A, B and C are as defined in the Energy Statistics above.\n\nOne can prove that <a href="https://www.codecogs.com/eqnedit.php?latex={E_{n,m}(X,Y)\\geq&space;0}" target="_blank"><img src="https://latex.codecogs.com/gif.latex?{E_{n,m}(X,Y)\\geq&space;0}" title="{E_{n,m}(X,Y)\\geq 0}" /></a> and that the corresponding population value is zero if and only if X and Y have the same distribution. Under this null hypothesis the test statistic\n\n<a href="https://www.codecogs.com/eqnedit.php?latex=T={\\frac&space;{nm}{n&plus;m}}E_{{n,m}}(X,Y)" target="_blank"><img src="https://latex.codecogs.com/gif.latex?T={\\frac&space;{nm}{n&plus;m}}E_{{n,m}}(X,Y)" title="T={\\frac {nm}{n+m}}E_{{n,m}}(X,Y)" /></a>\n\nconverges in distribution to a quadratic form of independent standard normal random variables. Under the alternative hypothesis T tends to infinity. This makes it possible to construct a consistent statistical test, the energy test for equal distributions\n  \nThus for a series Z of length L,\n\n<a href="https://www.codecogs.com/eqnedit.php?latex=Z&space;=&space;\\{Z_{1},&space;...,&space;Z_{\\tau}&space;,&space;...&space;,&space;Z_{L}\\},&space;X&space;=\\{Z_{1},...,Z_{\\tau}\\},&space;Y=\\{Z_{\\tau&plus;1}\\,...,Z_{L}\\}" target="_blank"><img src="https://latex.codecogs.com/gif.latex?Z&space;=&space;\\{Z_{1},&space;...,&space;Z_{\\tau}&space;,&space;...&space;,&space;Z_{L}\\},&space;X&space;=\\{Z_{1},...,Z_{\\tau}\\},&space;Y=\\{Z_{\\tau&plus;1}\\,...,Z_{L}\\}" title="Z = \\{Z_{1}, ..., Z_{\\tau} , ... , Z_{L}\\}, X =\\{Z_{1},...,Z_{\\tau}\\}, Y=\\{Z_{\\tau+1}\\,...,Z_{L}\\}" /></a>\n\nwe find the most likely change point by solving the following for &tau; such that it has the maximum T statistic value.\n\n### Multiple Change Points\n\nThe algorithm for finding multiple change points is also simple.\n\nAssuming you have some k known change points:\n1. Partition the series into segments between/around these change points.\n2. Find the maximum value of our divergence metric _within_ each partition.\n3. Take the maximum of the maxima we have just found --> this is our k+1th change point.\n4. Return to step 1 and continue until reaching your stopping criterion.\n\n### Stopping Criterion\n\nIn this package we have implemented a permutation based test as a stopping criterion:\n\nAfter step 3 of the multiple change point procedure above, randomly permute all of the data _within_ each cluster, and\n find the most likely change point for this permuted data using the procedure laid out above. \n \nAfter performing this operation z times, count the number of\n permuted change points z\' that have higher divergence metrics than the change point you calculated with un-permuted data.\n The significance level of your change point is thus z\'/(z+1). \n\nWe allow users to configure a permutation tester with `pvalue`\n and `permutations` representing the significance cutoff for algorithm termination and permutations to perform for each\n test, respectively.\n \n### Example\n```\nfrom signal_processing_algorithms.energy_statistics import energy_statistics\nfrom some_module import series\n\nchange_points = energy_statistics.e_divisive(series, pvalue=0.01, permutations=100)\n```\n\n',
    'author': 'Alexander Costas',
    'author_email': 'alexander.costas@mongodb.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mongodb/signal-processing-algorithms',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}
from build import *
build(setup_kwargs)

setup(**setup_kwargs)
