"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const iam = require("aws-cdk-lib/aws-iam");
const assertions_1 = require("aws-cdk-lib/assertions");
test('Test construct properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const pattern = new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'api-gateway-sagemakerendpoint', {
        endpointName: 'my-endpoint',
        resourcePath: '{my_param}',
        requestMappingTemplate: 'my-request-vtl-template'
    });
    expect(pattern.apiGateway).toBeDefined();
    expect(pattern.apiGatewayRole).toBeDefined();
    expect(pattern.apiGatewayCloudWatchRole).toBeDefined();
    expect(pattern.apiGatewayLogGroup).toBeDefined();
});
test('Test deployment w/ overwritten properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        description: 'existing role for SageMaker integration',
        inlinePolicies: {
            InvokePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:sagemaker:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:endpoint/my-endpoint`],
                        actions: ['sagemaker:InvokeEndpoint']
                    })]
            })
        }
    });
    new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'api-gateway-sagemakerendpoint', {
        endpointName: 'my-endpoint',
        resourcePath: '{my_param}',
        requestMappingTemplate: 'my-request-vtl-template',
        apiGatewayProps: {
            restApiName: 'my-api',
            deployOptions: {
                methodOptions: {
                    '/*/*': {
                        throttlingRateLimit: 100,
                        throttlingBurstLimit: 25
                    }
                }
            }
        },
        apiGatewayExecutionRole: existingRole,
        resourceName: 'my-resource',
        responseMappingTemplate: 'my-response-vtl-template'
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Stage', {
        MethodSettings: [
            {
                DataTraceEnabled: false,
                HttpMethod: '*',
                LoggingLevel: 'INFO',
                ResourcePath: '/*'
            },
            {
                HttpMethod: '*',
                ResourcePath: '/*',
                ThrottlingRateLimit: 100,
                ThrottlingBurstLimit: 25
            }
        ]
    });
    template.hasResourceProperties('AWS::ApiGateway::Resource', {
        PathPart: 'my-resource'
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'application/json': 'my-response-vtl-template',
                    },
                    StatusCode: '200'
                },
                {
                    StatusCode: '500',
                    SelectionPattern: '5\\d{2}'
                },
                {
                    StatusCode: '400',
                    SelectionPattern: '4\\d{2}'
                }
            ]
        },
        MethodResponses: [
            { StatusCode: '200' },
            { StatusCode: '500' },
            { StatusCode: '400' }
        ]
    });
    template.hasResourceProperties('AWS::IAM::Role', {
        Description: 'existing role for SageMaker integration'
    });
});
test('Construct accepts additional read request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'api-gateway-sagemaker-endpoint', {
        endpointName: 'my-endpoint',
        resourcePath: '{my_param}',
        requestMappingTemplate: 'my-request-vtl-template',
        additionalRequestTemplates: {
            'text/plain': 'additional-request-template'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            RequestTemplates: {
                'application/json': 'my-request-vtl-template',
                'text/plain': 'additional-request-template'
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpZ2F0ZXdheS1zYWdlbWFrZXJlbmRwb2ludC50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXBpZ2F0ZXdheS1zYWdlbWFrZXJlbmRwb2ludC50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCxVQUFVO0FBQ1YsNkNBQXlDO0FBQ3pDLGdDQUF1RDtBQUN2RCwyQ0FBMkM7QUFDM0MsdURBQWtEO0FBRWxELElBQUksQ0FBQywyQkFBMkIsRUFBRSxHQUFHLEVBQUU7SUFDckMsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxtQ0FBNkIsQ0FBQyxLQUFLLEVBQUUsK0JBQStCLEVBQUU7UUFDeEYsWUFBWSxFQUFFLGFBQWE7UUFDM0IsWUFBWSxFQUFFLFlBQVk7UUFDMUIsc0JBQXNCLEVBQUUseUJBQXlCO0tBQ2xELENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDekMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUM3QyxNQUFNLENBQUMsT0FBTyxDQUFDLHdCQUF3QixDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDdkQsTUFBTSxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0FBQ25ELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDJDQUEyQyxFQUFFLEdBQUcsRUFBRTtJQUNyRCxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUUxQixNQUFNLFlBQVksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO1FBQzNELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQywwQkFBMEIsQ0FBQztRQUMvRCxXQUFXLEVBQUUseUNBQXlDO1FBQ3RELGNBQWMsRUFBRTtZQUNkLFlBQVksRUFBRSxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUM7Z0JBQ25DLFVBQVUsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzt3QkFDbkMsU0FBUyxFQUFFLENBQUMsT0FBTyxpQkFBRyxDQUFDLFNBQVMsY0FBYyxpQkFBRyxDQUFDLE1BQU0sSUFBSSxpQkFBRyxDQUFDLFVBQVUsdUJBQXVCLENBQUM7d0JBQ2xHLE9BQU8sRUFBRSxDQUFDLDBCQUEwQixDQUFDO3FCQUN0QyxDQUFDLENBQUM7YUFDSixDQUFDO1NBQ0g7S0FDRixDQUFDLENBQUM7SUFFSCxJQUFJLG1DQUE2QixDQUFDLEtBQUssRUFBRSwrQkFBK0IsRUFBRTtRQUN4RSxZQUFZLEVBQUUsYUFBYTtRQUMzQixZQUFZLEVBQUUsWUFBWTtRQUMxQixzQkFBc0IsRUFBRSx5QkFBeUI7UUFFakQsZUFBZSxFQUFFO1lBQ2YsV0FBVyxFQUFFLFFBQVE7WUFDckIsYUFBYSxFQUFFO2dCQUNiLGFBQWEsRUFBRTtvQkFDYixNQUFNLEVBQUU7d0JBQ04sbUJBQW1CLEVBQUUsR0FBRzt3QkFDeEIsb0JBQW9CLEVBQUUsRUFBRTtxQkFDekI7aUJBQ0Y7YUFDRjtTQUNGO1FBQ0QsdUJBQXVCLEVBQUUsWUFBWTtRQUNyQyxZQUFZLEVBQUUsYUFBYTtRQUMzQix1QkFBdUIsRUFBRSwwQkFBMEI7S0FDcEQsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHdCQUF3QixFQUFFO1FBQ3ZELGNBQWMsRUFBRTtZQUNkO2dCQUNFLGdCQUFnQixFQUFFLEtBQUs7Z0JBQ3ZCLFVBQVUsRUFBRSxHQUFHO2dCQUNmLFlBQVksRUFBRSxNQUFNO2dCQUNwQixZQUFZLEVBQUUsSUFBSTthQUNuQjtZQUNEO2dCQUNFLFVBQVUsRUFBRSxHQUFHO2dCQUNmLFlBQVksRUFBRSxJQUFJO2dCQUNsQixtQkFBbUIsRUFBRSxHQUFHO2dCQUN4QixvQkFBb0IsRUFBRSxFQUFFO2FBQ3pCO1NBQ0Y7S0FDRixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsMkJBQTJCLEVBQUU7UUFDMUQsUUFBUSxFQUFFLGFBQWE7S0FDeEIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHlCQUF5QixFQUFFO1FBQ3hELFdBQVcsRUFBRTtZQUNYLG9CQUFvQixFQUFFO2dCQUNwQjtvQkFDRSxpQkFBaUIsRUFBRTt3QkFDakIsa0JBQWtCLEVBQUUsMEJBQTBCO3FCQUMvQztvQkFDRCxVQUFVLEVBQUUsS0FBSztpQkFDbEI7Z0JBQ0Q7b0JBQ0UsVUFBVSxFQUFFLEtBQUs7b0JBQ2pCLGdCQUFnQixFQUFFLFNBQVM7aUJBQzVCO2dCQUNEO29CQUNFLFVBQVUsRUFBRSxLQUFLO29CQUNqQixnQkFBZ0IsRUFBRSxTQUFTO2lCQUM1QjthQUNGO1NBQ0Y7UUFDRCxlQUFlLEVBQUU7WUFDZixFQUFFLFVBQVUsRUFBRSxLQUFLLEVBQUU7WUFDckIsRUFBRSxVQUFVLEVBQUUsS0FBSyxFQUFFO1lBQ3JCLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRTtTQUN0QjtLQUNGLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxnQkFBZ0IsRUFBRTtRQUMvQyxXQUFXLEVBQUUseUNBQXlDO0tBQ3ZELENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFEQUFxRCxFQUFFLEdBQUcsRUFBRTtJQUMvRCxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixJQUFJLG1DQUE2QixDQUFDLEtBQUssRUFBRSxnQ0FBZ0MsRUFBRTtRQUN6RSxZQUFZLEVBQUUsYUFBYTtRQUMzQixZQUFZLEVBQUUsWUFBWTtRQUMxQixzQkFBc0IsRUFBRSx5QkFBeUI7UUFDakQsMEJBQTBCLEVBQUU7WUFDMUIsWUFBWSxFQUFFLDZCQUE2QjtTQUM1QztLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx5QkFBeUIsRUFBRTtRQUN4RCxVQUFVLEVBQUUsS0FBSztRQUNqQixXQUFXLEVBQUU7WUFDWCxnQkFBZ0IsRUFBRTtnQkFDaEIsa0JBQWtCLEVBQUUseUJBQXlCO2dCQUM3QyxZQUFZLEVBQUUsNkJBQTZCO2FBQzVDO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbi8vIEltcG9ydHNcbmltcG9ydCB7IFN0YWNrLCBBd3MgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBBcGlHYXRld2F5VG9TYWdlTWFrZXJFbmRwb2ludCB9IGZyb20gJy4uL2xpYic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgeyBUZW1wbGF0ZSB9IGZyb20gJ2F3cy1jZGstbGliL2Fzc2VydGlvbnMnO1xuXG50ZXN0KCdUZXN0IGNvbnN0cnVjdCBwcm9wZXJ0aWVzJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBjb25zdCBwYXR0ZXJuID0gbmV3IEFwaUdhdGV3YXlUb1NhZ2VNYWtlckVuZHBvaW50KHN0YWNrLCAnYXBpLWdhdGV3YXktc2FnZW1ha2VyZW5kcG9pbnQnLCB7XG4gICAgZW5kcG9pbnROYW1lOiAnbXktZW5kcG9pbnQnLFxuICAgIHJlc291cmNlUGF0aDogJ3tteV9wYXJhbX0nLFxuICAgIHJlcXVlc3RNYXBwaW5nVGVtcGxhdGU6ICdteS1yZXF1ZXN0LXZ0bC10ZW1wbGF0ZSdcbiAgfSk7XG5cbiAgZXhwZWN0KHBhdHRlcm4uYXBpR2F0ZXdheSkudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KHBhdHRlcm4uYXBpR2F0ZXdheVJvbGUpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChwYXR0ZXJuLmFwaUdhdGV3YXlDbG91ZFdhdGNoUm9sZSkudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KHBhdHRlcm4uYXBpR2F0ZXdheUxvZ0dyb3VwKS50b0JlRGVmaW5lZCgpO1xufSk7XG5cbnRlc3QoJ1Rlc3QgZGVwbG95bWVudCB3LyBvdmVyd3JpdHRlbiBwcm9wZXJ0aWVzJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIGNvbnN0IGV4aXN0aW5nUm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ2FwaS1nYXRld2F5LXJvbGUnLCB7XG4gICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2FwaWdhdGV3YXkuYW1hem9uYXdzLmNvbScpLFxuICAgIGRlc2NyaXB0aW9uOiAnZXhpc3Rpbmcgcm9sZSBmb3IgU2FnZU1ha2VyIGludGVncmF0aW9uJyxcbiAgICBpbmxpbmVQb2xpY2llczoge1xuICAgICAgSW52b2tlUG9saWN5OiBuZXcgaWFtLlBvbGljeURvY3VtZW50KHtcbiAgICAgICAgc3RhdGVtZW50czogW25ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICByZXNvdXJjZXM6IFtgYXJuOiR7QXdzLlBBUlRJVElPTn06c2FnZW1ha2VyOiR7QXdzLlJFR0lPTn06JHtBd3MuQUNDT1VOVF9JRH06ZW5kcG9pbnQvbXktZW5kcG9pbnRgXSxcbiAgICAgICAgICBhY3Rpb25zOiBbJ3NhZ2VtYWtlcjpJbnZva2VFbmRwb2ludCddXG4gICAgICAgIH0pXVxuICAgICAgfSlcbiAgICB9XG4gIH0pO1xuXG4gIG5ldyBBcGlHYXRld2F5VG9TYWdlTWFrZXJFbmRwb2ludChzdGFjaywgJ2FwaS1nYXRld2F5LXNhZ2VtYWtlcmVuZHBvaW50Jywge1xuICAgIGVuZHBvaW50TmFtZTogJ215LWVuZHBvaW50JyxcbiAgICByZXNvdXJjZVBhdGg6ICd7bXlfcGFyYW19JyxcbiAgICByZXF1ZXN0TWFwcGluZ1RlbXBsYXRlOiAnbXktcmVxdWVzdC12dGwtdGVtcGxhdGUnLFxuXG4gICAgYXBpR2F0ZXdheVByb3BzOiB7XG4gICAgICByZXN0QXBpTmFtZTogJ215LWFwaScsXG4gICAgICBkZXBsb3lPcHRpb25zOiB7XG4gICAgICAgIG1ldGhvZE9wdGlvbnM6IHtcbiAgICAgICAgICAnLyovKic6IHtcbiAgICAgICAgICAgIHRocm90dGxpbmdSYXRlTGltaXQ6IDEwMCxcbiAgICAgICAgICAgIHRocm90dGxpbmdCdXJzdExpbWl0OiAyNVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgYXBpR2F0ZXdheUV4ZWN1dGlvblJvbGU6IGV4aXN0aW5nUm9sZSxcbiAgICByZXNvdXJjZU5hbWU6ICdteS1yZXNvdXJjZScsXG4gICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6ICdteS1yZXNwb25zZS12dGwtdGVtcGxhdGUnXG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkFwaUdhdGV3YXk6OlN0YWdlJywge1xuICAgIE1ldGhvZFNldHRpbmdzOiBbXG4gICAgICB7XG4gICAgICAgIERhdGFUcmFjZUVuYWJsZWQ6IGZhbHNlLFxuICAgICAgICBIdHRwTWV0aG9kOiAnKicsXG4gICAgICAgIExvZ2dpbmdMZXZlbDogJ0lORk8nLFxuICAgICAgICBSZXNvdXJjZVBhdGg6ICcvKidcbiAgICAgIH0sXG4gICAgICB7XG4gICAgICAgIEh0dHBNZXRob2Q6ICcqJyxcbiAgICAgICAgUmVzb3VyY2VQYXRoOiAnLyonLFxuICAgICAgICBUaHJvdHRsaW5nUmF0ZUxpbWl0OiAxMDAsXG4gICAgICAgIFRocm90dGxpbmdCdXJzdExpbWl0OiAyNVxuICAgICAgfVxuICAgIF1cbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkFwaUdhdGV3YXk6OlJlc291cmNlJywge1xuICAgIFBhdGhQYXJ0OiAnbXktcmVzb3VyY2UnXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgSW50ZWdyYXRpb246IHtcbiAgICAgIEludGVncmF0aW9uUmVzcG9uc2VzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBSZXNwb25zZVRlbXBsYXRlczoge1xuICAgICAgICAgICAgJ2FwcGxpY2F0aW9uL2pzb24nOiAnbXktcmVzcG9uc2UtdnRsLXRlbXBsYXRlJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIFN0YXR1c0NvZGU6ICcyMDAnXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBTdGF0dXNDb2RlOiAnNTAwJyxcbiAgICAgICAgICBTZWxlY3Rpb25QYXR0ZXJuOiAnNVxcXFxkezJ9J1xuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgU3RhdHVzQ29kZTogJzQwMCcsXG4gICAgICAgICAgU2VsZWN0aW9uUGF0dGVybjogJzRcXFxcZHsyfSdcbiAgICAgICAgfVxuICAgICAgXVxuICAgIH0sXG4gICAgTWV0aG9kUmVzcG9uc2VzOiBbXG4gICAgICB7IFN0YXR1c0NvZGU6ICcyMDAnIH0sXG4gICAgICB7IFN0YXR1c0NvZGU6ICc1MDAnIH0sXG4gICAgICB7IFN0YXR1c0NvZGU6ICc0MDAnIH1cbiAgICBdXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgRGVzY3JpcHRpb246ICdleGlzdGluZyByb2xlIGZvciBTYWdlTWFrZXIgaW50ZWdyYXRpb24nXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0NvbnN0cnVjdCBhY2NlcHRzIGFkZGl0aW9uYWwgcmVhZCByZXF1ZXN0IHRlbXBsYXRlcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgbmV3IEFwaUdhdGV3YXlUb1NhZ2VNYWtlckVuZHBvaW50KHN0YWNrLCAnYXBpLWdhdGV3YXktc2FnZW1ha2VyLWVuZHBvaW50Jywge1xuICAgIGVuZHBvaW50TmFtZTogJ215LWVuZHBvaW50JyxcbiAgICByZXNvdXJjZVBhdGg6ICd7bXlfcGFyYW19JyxcbiAgICByZXF1ZXN0TWFwcGluZ1RlbXBsYXRlOiAnbXktcmVxdWVzdC12dGwtdGVtcGxhdGUnLFxuICAgIGFkZGl0aW9uYWxSZXF1ZXN0VGVtcGxhdGVzOiB7XG4gICAgICAndGV4dC9wbGFpbic6ICdhZGRpdGlvbmFsLXJlcXVlc3QtdGVtcGxhdGUnXG4gICAgfVxuICB9KTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgSHR0cE1ldGhvZDogJ0dFVCcsXG4gICAgSW50ZWdyYXRpb246IHtcbiAgICAgIFJlcXVlc3RUZW1wbGF0ZXM6IHtcbiAgICAgICAgJ2FwcGxpY2F0aW9uL2pzb24nOiAnbXktcmVxdWVzdC12dGwtdGVtcGxhdGUnLFxuICAgICAgICAndGV4dC9wbGFpbic6ICdhZGRpdGlvbmFsLXJlcXVlc3QtdGVtcGxhdGUnXG4gICAgICB9XG4gICAgfVxuICB9KTtcbn0pO1xuIl19