"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const secretsmanager = require("../lib");
module.exports = {
    'default secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {}
        }));
        test.done();
    },
    'secret with generate secret string options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                excludeUppercase: true,
                passwordLength: 20
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                ExcludeUppercase: true,
                PasswordLength: 20
            }
        }));
        test.done();
    },
    'templated secret string'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                secretStringTemplate: JSON.stringify({ username: 'username' }),
                generateStringKey: 'password'
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                SecretStringTemplate: '{"username":"username"}',
                GenerateStringKey: 'password'
            }
        }));
        test.done();
    },
    'grantRead'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'secretsmanager:GetSecretValue',
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                    }]
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [{
                        Action: [
                            "kms:Create*",
                            "kms:Describe*",
                            "kms:Enable*",
                            "kms:List*",
                            "kms:Put*",
                            "kms:Update*",
                            "kms:Revoke*",
                            "kms:Disable*",
                            "kms:Get*",
                            "kms:Delete*",
                            "kms:ScheduleKeyDeletion",
                            "kms:CancelKeyDeletion",
                            "kms:GenerateDataKey"
                        ],
                        Effect: "Allow",
                        Principal: {
                            AWS: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":iam::",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":root"
                                    ]
                                ]
                            }
                        },
                        Resource: "*"
                    }, {
                        Action: "kms:Decrypt",
                        Condition: {
                            StringEquals: {
                                "kms:ViaService": {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "secretsmanager.",
                                            {
                                                Ref: "AWS::Region"
                                            },
                                            ".amazonaws.com"
                                        ]
                                    ]
                                }
                            }
                        },
                        Effect: "Allow",
                        Principal: {
                            AWS: {
                                "Fn::GetAtt": [
                                    "Role1ABCC5F0",
                                    "Arn"
                                ]
                            }
                        },
                        Resource: "*"
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'grantRead with version label constraint'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantRead(role, ['FOO', 'bar']);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'secretsmanager:GetSecretValue',
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                        Condition: {
                            'ForAnyValue:StringEquals': {
                                'secretsmanager:VersionStage': ['FOO', 'bar'],
                            },
                        },
                    }]
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [{
                        Action: [
                            "kms:Create*",
                            "kms:Describe*",
                            "kms:Enable*",
                            "kms:List*",
                            "kms:Put*",
                            "kms:Update*",
                            "kms:Revoke*",
                            "kms:Disable*",
                            "kms:Get*",
                            "kms:Delete*",
                            "kms:ScheduleKeyDeletion",
                            "kms:CancelKeyDeletion",
                            "kms:GenerateDataKey"
                        ],
                        Effect: "Allow",
                        Principal: {
                            AWS: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":iam::",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":root"
                                    ]
                                ]
                            }
                        },
                        Resource: "*"
                    }, {
                        Action: "kms:Decrypt",
                        Condition: {
                            StringEquals: {
                                "kms:ViaService": {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "secretsmanager.",
                                            {
                                                Ref: "AWS::Region"
                                            },
                                            ".amazonaws.com"
                                        ]
                                    ]
                                }
                            }
                        },
                        Effect: "Allow",
                        Principal: {
                            AWS: {
                                "Fn::GetAtt": [
                                    "Role1ABCC5F0",
                                    "Arn"
                                ]
                            }
                        },
                        Resource: "*"
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'secretValue'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        // WHEN
        new cdk.CfnResource(stack, 'FakeResource', {
            type: 'CDK::Phony::Resource',
            properties: {
                value: secret.secretValue
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('CDK::Phony::Resource', {
            value: {
                'Fn::Join': ['', [
                        '{{resolve:secretsmanager:',
                        { Ref: 'SecretA720EF05' },
                        ':SecretString:::}}'
                    ]]
            }
        }));
        test.done();
    },
    'import'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const encryptionKey = new kms.Key(stack, 'KMS');
        const secretArn = 'arn::of::a::secret';
        // WHEN
        const secret = secretsmanager.Secret.fromSecretAttributes(stack, 'Secret', {
            secretArn, encryptionKey
        });
        // THEN
        test.equals(secret.secretArn, secretArn);
        test.same(secret.encryptionKey, encryptionKey);
        test.deepEqual(stack.resolve(secret.secretValue), '{{resolve:secretsmanager:arn::of::a::secret:SecretString:::}}');
        test.deepEqual(stack.resolve(secret.secretValueFromJson('password')), '{{resolve:secretsmanager:arn::of::a::secret:SecretString:password::}}');
        test.done();
    },
    'attached secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const target = {
            asSecretAttachmentTarget: () => ({
                targetId: 'instance',
                targetType: secretsmanager.AttachmentTargetType.INSTANCE
            })
        };
        // WHEN
        secret.addTargetAttachment('AttachedSecret', { target });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::SecretTargetAttachment', {
            SecretId: {
                Ref: 'SecretA720EF05'
            },
            TargetId: 'instance',
            TargetType: 'AWS::RDS::DBInstance'
        }));
        test.done();
    },
    'add a rotation schedule to an attached secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const target = {
            asSecretAttachmentTarget: () => ({
                targetId: 'cluster',
                targetType: secretsmanager.AttachmentTargetType.CLUSTER
            })
        };
        const attachedSecret = secret.addTargetAttachment('AttachedSecret', { target });
        const rotationLambda = new lambda.Function(stack, 'Lambda', {
            runtime: lambda.Runtime.NODEJS_8_10,
            code: lambda.Code.fromInline('export.handler = event => event;'),
            handler: 'index.handler'
        });
        // WHEN
        attachedSecret.addRotationSchedule('RotationSchedule', {
            rotationLambda
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::RotationSchedule', {
            SecretId: {
                Ref: 'SecretAttachedSecret94145316' // The secret returned by the attachment, not the secret itself.
            }
        }));
        test.done();
    },
    'throws when specifying secretStringTemplate but not generateStringKey'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                secretStringTemplate: JSON.stringify({ username: 'username' })
            }
        }), /`secretStringTemplate`.+`generateStringKey`/);
        test.done();
    },
    'throws when specifying generateStringKey but not secretStringTemplate'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                generateStringKey: 'password'
            }
        }), /`secretStringTemplate`.+`generateStringKey`/);
        test.done();
    },
    'equivalence of SecretValue and Secret.fromSecretAttributes'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const imported = secretsmanager.Secret.fromSecretAttributes(stack, 'Imported', { secretArn: 'my-secret-arn' }).secretValueFromJson('password');
        const value = core_1.SecretValue.secretsManager('my-secret-arn', { jsonField: 'password' });
        // THEN
        test.deepEqual(stack.resolve(imported), stack.resolve(value));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zZWNyZXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LnNlY3JldC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQXVEO0FBQ3ZELHdDQUF5QztBQUN6Qyx3Q0FBeUM7QUFDekMsOENBQStDO0FBQy9DLHFDQUFzQztBQUN0Qyx3Q0FBbUQ7QUFFbkQseUNBQTBDO0FBRTFDLGlCQUFTO0lBQ1AsZ0JBQWdCLENBQUMsSUFBVTtRQUN6QixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsT0FBTztRQUNQLElBQUksY0FBYyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFM0MsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyw2QkFBNkIsRUFBRTtZQUMzRCxvQkFBb0IsRUFBRSxFQUFFO1NBQ3pCLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDRDQUE0QyxDQUFDLElBQVU7UUFDckQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxJQUFJLGNBQWMsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUN6QyxvQkFBb0IsRUFBRTtnQkFDcEIsZ0JBQWdCLEVBQUUsSUFBSTtnQkFDdEIsY0FBYyxFQUFFLEVBQUU7YUFDbkI7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDZCQUE2QixFQUFFO1lBQzNELG9CQUFvQixFQUFFO2dCQUNwQixnQkFBZ0IsRUFBRSxJQUFJO2dCQUN0QixjQUFjLEVBQUUsRUFBRTthQUNuQjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHlCQUF5QixDQUFDLElBQVU7UUFDbEMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxJQUFJLGNBQWMsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUN6QyxvQkFBb0IsRUFBRTtnQkFDcEIsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxVQUFVLEVBQUUsQ0FBQztnQkFDOUQsaUJBQWlCLEVBQUUsVUFBVTthQUM5QjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsNkJBQTZCLEVBQUU7WUFDM0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLG9CQUFvQixFQUFFLHlCQUF5QjtnQkFDL0MsaUJBQWlCLEVBQUUsVUFBVTthQUM5QjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELFdBQVcsQ0FBQyxJQUFVO1FBQ3BCLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3RDLE1BQU0sTUFBTSxHQUFHLElBQUksY0FBYyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFDbEYsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsb0JBQW9CLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFeEYsT0FBTztRQUNQLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFdkIsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtZQUNoRCxjQUFjLEVBQUU7Z0JBQ2QsT0FBTyxFQUFFLFlBQVk7Z0JBQ3JCLFNBQVMsRUFBRSxDQUFDO3dCQUNWLE1BQU0sRUFBRSwrQkFBK0I7d0JBQ3ZDLE1BQU0sRUFBRSxPQUFPO3dCQUNmLFFBQVEsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTtxQkFDcEMsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZUFBZSxFQUFFO1lBQzdDLFNBQVMsRUFBRTtnQkFDVCxTQUFTLEVBQUUsQ0FBQzt3QkFDVixNQUFNLEVBQUU7NEJBQ04sYUFBYTs0QkFDYixlQUFlOzRCQUNmLGFBQWE7NEJBQ2IsV0FBVzs0QkFDWCxVQUFVOzRCQUNWLGFBQWE7NEJBQ2IsYUFBYTs0QkFDYixjQUFjOzRCQUNkLFVBQVU7NEJBQ1YsYUFBYTs0QkFDYix5QkFBeUI7NEJBQ3pCLHVCQUF1Qjs0QkFDdkIscUJBQXFCO3lCQUN0Qjt3QkFDRCxNQUFNLEVBQUUsT0FBTzt3QkFDZixTQUFTLEVBQUU7NEJBQ1QsR0FBRyxFQUFFO2dDQUNILFVBQVUsRUFBRTtvQ0FDVixFQUFFO29DQUNGO3dDQUNFLE1BQU07d0NBQ047NENBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5Q0FDdEI7d0NBQ0QsUUFBUTt3Q0FDUjs0Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3lDQUN0Qjt3Q0FDRCxPQUFPO3FDQUNSO2lDQUNGOzZCQUNGO3lCQUNGO3dCQUNELFFBQVEsRUFBRSxHQUFHO3FCQUNkLEVBQUU7d0JBQ0QsTUFBTSxFQUFFLGFBQWE7d0JBQ3JCLFNBQVMsRUFBRTs0QkFDVCxZQUFZLEVBQUU7Z0NBQ1osZ0JBQWdCLEVBQUU7b0NBQ2hCLFVBQVUsRUFBRTt3Q0FDVixFQUFFO3dDQUNGOzRDQUNFLGlCQUFpQjs0Q0FDakI7Z0RBQ0UsR0FBRyxFQUFFLGFBQWE7NkNBQ25COzRDQUNELGdCQUFnQjt5Q0FDakI7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7d0JBQ0QsTUFBTSxFQUFFLE9BQU87d0JBQ2YsU0FBUyxFQUFFOzRCQUNULEdBQUcsRUFBRTtnQ0FDSCxZQUFZLEVBQUU7b0NBQ1osY0FBYztvQ0FDZCxLQUFLO2lDQUNOOzZCQUNGO3lCQUNGO3dCQUNELFFBQVEsRUFBRSxHQUFHO3FCQUNkO2lCQUNGO2dCQUNELE9BQU8sRUFBRSxZQUFZO2FBQ3BCO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseUNBQXlDLENBQUMsSUFBVTtRQUNsRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN0QyxNQUFNLE1BQU0sR0FBRyxJQUFJLGNBQWMsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRSxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQ2xGLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLG9CQUFvQixFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRXhGLE9BQU87UUFDUCxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBRXZDLE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7WUFDaEQsY0FBYyxFQUFFO2dCQUNkLE9BQU8sRUFBRSxZQUFZO2dCQUNyQixTQUFTLEVBQUUsQ0FBQzt3QkFDVixNQUFNLEVBQUUsK0JBQStCO3dCQUN2QyxNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUUsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7d0JBQ25DLFNBQVMsRUFBRTs0QkFDVCwwQkFBMEIsRUFBRTtnQ0FDMUIsNkJBQTZCLEVBQUUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDOzZCQUM5Qzt5QkFDRjtxQkFDRixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxlQUFlLEVBQUU7WUFDN0MsU0FBUyxFQUFFO2dCQUNULFNBQVMsRUFBRSxDQUFDO3dCQUNWLE1BQU0sRUFBRTs0QkFDTixhQUFhOzRCQUNiLGVBQWU7NEJBQ2YsYUFBYTs0QkFDYixXQUFXOzRCQUNYLFVBQVU7NEJBQ1YsYUFBYTs0QkFDYixhQUFhOzRCQUNiLGNBQWM7NEJBQ2QsVUFBVTs0QkFDVixhQUFhOzRCQUNiLHlCQUF5Qjs0QkFDekIsdUJBQXVCOzRCQUN2QixxQkFBcUI7eUJBQ3RCO3dCQUNELE1BQU0sRUFBRSxPQUFPO3dCQUNmLFNBQVMsRUFBRTs0QkFDVCxHQUFHLEVBQUU7Z0NBQ0gsVUFBVSxFQUFFO29DQUNWLEVBQUU7b0NBQ0Y7d0NBQ0UsTUFBTTt3Q0FDTjs0Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3lDQUN0Qjt3Q0FDRCxRQUFRO3dDQUNSOzRDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7eUNBQ3RCO3dDQUNELE9BQU87cUNBQ1I7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7d0JBQ0QsUUFBUSxFQUFFLEdBQUc7cUJBQ2QsRUFBRTt3QkFDRCxNQUFNLEVBQUUsYUFBYTt3QkFDckIsU0FBUyxFQUFFOzRCQUNULFlBQVksRUFBRTtnQ0FDWixnQkFBZ0IsRUFBRTtvQ0FDaEIsVUFBVSxFQUFFO3dDQUNWLEVBQUU7d0NBQ0Y7NENBQ0UsaUJBQWlCOzRDQUNqQjtnREFDRSxHQUFHLEVBQUUsYUFBYTs2Q0FDbkI7NENBQ0QsZ0JBQWdCO3lDQUNqQjtxQ0FDRjtpQ0FDRjs2QkFDRjt5QkFDRjt3QkFDRCxNQUFNLEVBQUUsT0FBTzt3QkFDZixTQUFTLEVBQUU7NEJBQ1QsR0FBRyxFQUFFO2dDQUNILFlBQVksRUFBRTtvQ0FDWixjQUFjO29DQUNkLEtBQUs7aUNBQ047NkJBQ0Y7eUJBQ0Y7d0JBQ0QsUUFBUSxFQUFFLEdBQUc7cUJBQ2Q7aUJBQ0Y7Z0JBQ0QsT0FBTyxFQUFFLFlBQVk7YUFDcEI7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxhQUFhLENBQUMsSUFBVTtRQUN0QixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN0QyxNQUFNLE1BQU0sR0FBRyxJQUFJLGNBQWMsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRSxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBRWxGLE9BQU87UUFDUCxJQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtZQUN6QyxJQUFJLEVBQUUsc0JBQXNCO1lBQzVCLFVBQVUsRUFBRTtnQkFDVixLQUFLLEVBQUUsTUFBTSxDQUFDLFdBQVc7YUFDMUI7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFO1lBQ3BELEtBQUssRUFBRTtnQkFDTCxVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUU7d0JBQ2YsMkJBQTJCO3dCQUMzQixFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTt3QkFDekIsb0JBQW9CO3FCQUNyQixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxRQUFRLENBQUMsSUFBVTtRQUNqQixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNoRCxNQUFNLFNBQVMsR0FBRyxvQkFBb0IsQ0FBQztRQUV2QyxPQUFPO1FBQ1AsTUFBTSxNQUFNLEdBQUcsY0FBYyxDQUFDLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ3pFLFNBQVMsRUFBRSxhQUFhO1NBQ3pCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDekMsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLEVBQUUsK0RBQStELENBQUMsQ0FBQztRQUNuSCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLG1CQUFtQixDQUFDLFVBQVUsQ0FBQyxDQUFDLEVBQUUsdUVBQXVFLENBQUMsQ0FBQztRQUMvSSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsaUJBQWlCLENBQUMsSUFBVTtRQUMxQixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxjQUFjLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUMxRCxNQUFNLE1BQU0sR0FBMkM7WUFDckQsd0JBQXdCLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztnQkFDL0IsUUFBUSxFQUFFLFVBQVU7Z0JBQ3BCLFVBQVUsRUFBRSxjQUFjLENBQUMsb0JBQW9CLENBQUMsUUFBUTthQUN6RCxDQUFDO1NBQ0gsQ0FBQztRQUVGLE9BQU87UUFDUCxNQUFNLENBQUMsbUJBQW1CLENBQUMsZ0JBQWdCLEVBQUUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBRXpELE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsNkNBQTZDLEVBQUU7WUFDM0UsUUFBUSxFQUFFO2dCQUNSLEdBQUcsRUFBRSxnQkFBZ0I7YUFDdEI7WUFDRCxRQUFRLEVBQUUsVUFBVTtZQUNwQixVQUFVLEVBQUUsc0JBQXNCO1NBQ25DLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELCtDQUErQyxDQUFDLElBQVU7UUFDeEQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sTUFBTSxHQUFHLElBQUksY0FBYyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDMUQsTUFBTSxNQUFNLEdBQTJDO1lBQ3JELHdCQUF3QixFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7Z0JBQy9CLFFBQVEsRUFBRSxTQUFTO2dCQUNuQixVQUFVLEVBQUUsY0FBYyxDQUFDLG9CQUFvQixDQUFDLE9BQU87YUFDeEQsQ0FBQztTQUNILENBQUM7UUFDRixNQUFNLGNBQWMsR0FBRyxNQUFNLENBQUMsbUJBQW1CLENBQUMsZ0JBQWdCLEVBQUUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBQ2hGLE1BQU0sY0FBYyxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQzFELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDbkMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLGtDQUFrQyxDQUFDO1lBQ2hFLE9BQU8sRUFBRSxlQUFlO1NBQ3pCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxjQUFjLENBQUMsbUJBQW1CLENBQUMsa0JBQWtCLEVBQUU7WUFDckQsY0FBYztTQUNmLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUNBQXVDLEVBQUU7WUFDckUsUUFBUSxFQUFFO2dCQUNSLEdBQUcsRUFBRSw4QkFBOEIsQ0FBQyxnRUFBZ0U7YUFDckc7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1RUFBdUUsQ0FBQyxJQUFVO1FBQ2hGLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU5QixPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLGNBQWMsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUMzRCxvQkFBb0IsRUFBRTtnQkFDcEIsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxVQUFVLEVBQUUsQ0FBQzthQUMvRDtTQUNGLENBQUMsRUFBRSw2Q0FBNkMsQ0FBQyxDQUFDO1FBRW5ELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1RUFBdUUsQ0FBQyxJQUFVO1FBQ2hGLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU5QixPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLGNBQWMsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUMzRCxvQkFBb0IsRUFBRTtnQkFDcEIsaUJBQWlCLEVBQUUsVUFBVTthQUM5QjtTQUNGLENBQUMsRUFBRSw2Q0FBNkMsQ0FBQyxDQUFDO1FBRW5ELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0REFBNEQsQ0FBQyxJQUFVO1FBQ3JFLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxjQUFjLENBQUMsTUFBTSxDQUFDLG9CQUFvQixDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsZUFBZSxFQUFFLENBQUMsQ0FBQyxtQkFBbUIsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUMvSSxNQUFNLEtBQUssR0FBRyxrQkFBVyxDQUFDLGNBQWMsQ0FBQyxlQUFlLEVBQUUsRUFBRSxTQUFTLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQztRQUVyRixPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUM5RCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCBpYW0gPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtaWFtJyk7XG5pbXBvcnQga21zID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWttcycpO1xuaW1wb3J0IGxhbWJkYSA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnKTtcbmltcG9ydCBjZGsgPSByZXF1aXJlKCdAYXdzLWNkay9jb3JlJyk7XG5pbXBvcnQgeyBTZWNyZXRWYWx1ZSwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgc2VjcmV0c21hbmFnZXIgPSByZXF1aXJlKCcuLi9saWInKTtcblxuZXhwb3J0ID0ge1xuICAnZGVmYXVsdCBzZWNyZXQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBzZWNyZXRzbWFuYWdlci5TZWNyZXQoc3RhY2ssICdTZWNyZXQnKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpTZWNyZXRzTWFuYWdlcjo6U2VjcmV0Jywge1xuICAgICAgR2VuZXJhdGVTZWNyZXRTdHJpbmc6IHt9XG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3NlY3JldCB3aXRoIGdlbmVyYXRlIHNlY3JldCBzdHJpbmcgb3B0aW9ucycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IHNlY3JldHNtYW5hZ2VyLlNlY3JldChzdGFjaywgJ1NlY3JldCcsIHtcbiAgICAgIGdlbmVyYXRlU2VjcmV0U3RyaW5nOiB7XG4gICAgICAgIGV4Y2x1ZGVVcHBlcmNhc2U6IHRydWUsXG4gICAgICAgIHBhc3N3b3JkTGVuZ3RoOiAyMFxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OlNlY3JldHNNYW5hZ2VyOjpTZWNyZXQnLCB7XG4gICAgICBHZW5lcmF0ZVNlY3JldFN0cmluZzoge1xuICAgICAgICBFeGNsdWRlVXBwZXJjYXNlOiB0cnVlLFxuICAgICAgICBQYXNzd29yZExlbmd0aDogMjBcbiAgICAgIH1cbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndGVtcGxhdGVkIHNlY3JldCBzdHJpbmcnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBzZWNyZXRzbWFuYWdlci5TZWNyZXQoc3RhY2ssICdTZWNyZXQnLCB7XG4gICAgICBnZW5lcmF0ZVNlY3JldFN0cmluZzoge1xuICAgICAgICBzZWNyZXRTdHJpbmdUZW1wbGF0ZTogSlNPTi5zdHJpbmdpZnkoeyB1c2VybmFtZTogJ3VzZXJuYW1lJyB9KSxcbiAgICAgICAgZ2VuZXJhdGVTdHJpbmdLZXk6ICdwYXNzd29yZCdcbiAgICAgIH1cbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpTZWNyZXRzTWFuYWdlcjo6U2VjcmV0Jywge1xuICAgICAgR2VuZXJhdGVTZWNyZXRTdHJpbmc6IHtcbiAgICAgICAgU2VjcmV0U3RyaW5nVGVtcGxhdGU6ICd7XCJ1c2VybmFtZVwiOlwidXNlcm5hbWVcIn0nLFxuICAgICAgICBHZW5lcmF0ZVN0cmluZ0tleTogJ3Bhc3N3b3JkJ1xuICAgICAgfVxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdncmFudFJlYWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGtleSA9IG5ldyBrbXMuS2V5KHN0YWNrLCAnS01TJyk7XG4gICAgY29uc3Qgc2VjcmV0ID0gbmV3IHNlY3JldHNtYW5hZ2VyLlNlY3JldChzdGFjaywgJ1NlY3JldCcsIHsgZW5jcnlwdGlvbktleToga2V5IH0pO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywgeyBhc3N1bWVkQnk6IG5ldyBpYW0uQWNjb3VudFJvb3RQcmluY2lwYWwoKSB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBzZWNyZXQuZ3JhbnRSZWFkKHJvbGUpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgICAgICBTdGF0ZW1lbnQ6IFt7XG4gICAgICAgICAgQWN0aW9uOiAnc2VjcmV0c21hbmFnZXI6R2V0U2VjcmV0VmFsdWUnLFxuICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICBSZXNvdXJjZTogeyBSZWY6ICdTZWNyZXRBNzIwRUYwNScgfSxcbiAgICAgICAgfV1cbiAgICAgIH1cbiAgICB9KSk7XG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6S01TOjpLZXknLCB7XG4gICAgICBLZXlQb2xpY3k6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbe1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJrbXM6Q3JlYXRlKlwiLFxuICAgICAgICAgICAgXCJrbXM6RGVzY3JpYmUqXCIsXG4gICAgICAgICAgICBcImttczpFbmFibGUqXCIsXG4gICAgICAgICAgICBcImttczpMaXN0KlwiLFxuICAgICAgICAgICAgXCJrbXM6UHV0KlwiLFxuICAgICAgICAgICAgXCJrbXM6VXBkYXRlKlwiLFxuICAgICAgICAgICAgXCJrbXM6UmV2b2tlKlwiLFxuICAgICAgICAgICAgXCJrbXM6RGlzYWJsZSpcIixcbiAgICAgICAgICAgIFwia21zOkdldCpcIixcbiAgICAgICAgICAgIFwia21zOkRlbGV0ZSpcIixcbiAgICAgICAgICAgIFwia21zOlNjaGVkdWxlS2V5RGVsZXRpb25cIixcbiAgICAgICAgICAgIFwia21zOkNhbmNlbEtleURlbGV0aW9uXCIsXG4gICAgICAgICAgICBcImttczpHZW5lcmF0ZURhdGFLZXlcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUHJpbmNpcGFsOiB7XG4gICAgICAgICAgICBBV1M6IHtcbiAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICBcImFybjpcIixcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiBcIkFXUzo6UGFydGl0aW9uXCJcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBcIjppYW06OlwiLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpBY2NvdW50SWRcIlxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIFwiOnJvb3RcIlxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgUmVzb3VyY2U6IFwiKlwiXG4gICAgICAgIH0sIHtcbiAgICAgICAgICBBY3Rpb246IFwia21zOkRlY3J5cHRcIixcbiAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgIFN0cmluZ0VxdWFsczoge1xuICAgICAgICAgICAgICBcImttczpWaWFTZXJ2aWNlXCI6IHtcbiAgICAgICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgIFwic2VjcmV0c21hbmFnZXIuXCIsXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpSZWdpb25cIlxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBcIi5hbWF6b25hd3MuY29tXCJcbiAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgQVdTOiB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgXCJSb2xlMUFCQ0M1RjBcIixcbiAgICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIFJlc291cmNlOiBcIipcIlxuICAgICAgICB9XG4gICAgICBdLFxuICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICAgIH1cbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2dyYW50UmVhZCB3aXRoIHZlcnNpb24gbGFiZWwgY29uc3RyYWludCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3Qga2V5ID0gbmV3IGttcy5LZXkoc3RhY2ssICdLTVMnKTtcbiAgICBjb25zdCBzZWNyZXQgPSBuZXcgc2VjcmV0c21hbmFnZXIuU2VjcmV0KHN0YWNrLCAnU2VjcmV0JywgeyBlbmNyeXB0aW9uS2V5OiBrZXkgfSk7XG4gICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ1JvbGUnLCB7IGFzc3VtZWRCeTogbmV3IGlhbS5BY2NvdW50Um9vdFByaW5jaXBhbCgpIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIHNlY3JldC5ncmFudFJlYWQocm9sZSwgWydGT08nLCAnYmFyJ10pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgICAgICBTdGF0ZW1lbnQ6IFt7XG4gICAgICAgICAgQWN0aW9uOiAnc2VjcmV0c21hbmFnZXI6R2V0U2VjcmV0VmFsdWUnLFxuICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICBSZXNvdXJjZTogeyBSZWY6ICdTZWNyZXRBNzIwRUYwNScgfSxcbiAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgICdGb3JBbnlWYWx1ZTpTdHJpbmdFcXVhbHMnOiB7XG4gICAgICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpWZXJzaW9uU3RhZ2UnOiBbJ0ZPTycsICdiYXInXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfV1cbiAgICAgIH1cbiAgICB9KSk7XG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6S01TOjpLZXknLCB7XG4gICAgICBLZXlQb2xpY3k6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbe1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJrbXM6Q3JlYXRlKlwiLFxuICAgICAgICAgICAgXCJrbXM6RGVzY3JpYmUqXCIsXG4gICAgICAgICAgICBcImttczpFbmFibGUqXCIsXG4gICAgICAgICAgICBcImttczpMaXN0KlwiLFxuICAgICAgICAgICAgXCJrbXM6UHV0KlwiLFxuICAgICAgICAgICAgXCJrbXM6VXBkYXRlKlwiLFxuICAgICAgICAgICAgXCJrbXM6UmV2b2tlKlwiLFxuICAgICAgICAgICAgXCJrbXM6RGlzYWJsZSpcIixcbiAgICAgICAgICAgIFwia21zOkdldCpcIixcbiAgICAgICAgICAgIFwia21zOkRlbGV0ZSpcIixcbiAgICAgICAgICAgIFwia21zOlNjaGVkdWxlS2V5RGVsZXRpb25cIixcbiAgICAgICAgICAgIFwia21zOkNhbmNlbEtleURlbGV0aW9uXCIsXG4gICAgICAgICAgICBcImttczpHZW5lcmF0ZURhdGFLZXlcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUHJpbmNpcGFsOiB7XG4gICAgICAgICAgICBBV1M6IHtcbiAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICBcImFybjpcIixcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiBcIkFXUzo6UGFydGl0aW9uXCJcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBcIjppYW06OlwiLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpBY2NvdW50SWRcIlxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIFwiOnJvb3RcIlxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgUmVzb3VyY2U6IFwiKlwiXG4gICAgICAgIH0sIHtcbiAgICAgICAgICBBY3Rpb246IFwia21zOkRlY3J5cHRcIixcbiAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgIFN0cmluZ0VxdWFsczoge1xuICAgICAgICAgICAgICBcImttczpWaWFTZXJ2aWNlXCI6IHtcbiAgICAgICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgIFwic2VjcmV0c21hbmFnZXIuXCIsXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpSZWdpb25cIlxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBcIi5hbWF6b25hd3MuY29tXCJcbiAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgQVdTOiB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgXCJSb2xlMUFCQ0M1RjBcIixcbiAgICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIFJlc291cmNlOiBcIipcIlxuICAgICAgICB9XG4gICAgICBdLFxuICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICAgIH1cbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3NlY3JldFZhbHVlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBrZXkgPSBuZXcga21zLktleShzdGFjaywgJ0tNUycpO1xuICAgIGNvbnN0IHNlY3JldCA9IG5ldyBzZWNyZXRzbWFuYWdlci5TZWNyZXQoc3RhY2ssICdTZWNyZXQnLCB7IGVuY3J5cHRpb25LZXk6IGtleSB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgY2RrLkNmblJlc291cmNlKHN0YWNrLCAnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0NESzo6UGhvbnk6OlJlc291cmNlJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgdmFsdWU6IHNlY3JldC5zZWNyZXRWYWx1ZVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdDREs6OlBob255OjpSZXNvdXJjZScsIHtcbiAgICAgIHZhbHVlOiB7XG4gICAgICAgICdGbjo6Sm9pbic6IFsnJywgW1xuICAgICAgICAgICd7e3Jlc29sdmU6c2VjcmV0c21hbmFnZXI6JyxcbiAgICAgICAgICB7IFJlZjogJ1NlY3JldEE3MjBFRjA1JyB9LFxuICAgICAgICAgICc6U2VjcmV0U3RyaW5nOjo6fX0nXG4gICAgICAgIF1dXG4gICAgICB9XG4gICAgfSkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdpbXBvcnQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGVuY3J5cHRpb25LZXkgPSBuZXcga21zLktleShzdGFjaywgJ0tNUycpO1xuICAgIGNvbnN0IHNlY3JldEFybiA9ICdhcm46Om9mOjphOjpzZWNyZXQnO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHNlY3JldCA9IHNlY3JldHNtYW5hZ2VyLlNlY3JldC5mcm9tU2VjcmV0QXR0cmlidXRlcyhzdGFjaywgJ1NlY3JldCcsIHtcbiAgICAgIHNlY3JldEFybiwgZW5jcnlwdGlvbktleVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZXF1YWxzKHNlY3JldC5zZWNyZXRBcm4sIHNlY3JldEFybik7XG4gICAgdGVzdC5zYW1lKHNlY3JldC5lbmNyeXB0aW9uS2V5LCBlbmNyeXB0aW9uS2V5KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHNlY3JldC5zZWNyZXRWYWx1ZSksICd7e3Jlc29sdmU6c2VjcmV0c21hbmFnZXI6YXJuOjpvZjo6YTo6c2VjcmV0OlNlY3JldFN0cmluZzo6On19Jyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShzZWNyZXQuc2VjcmV0VmFsdWVGcm9tSnNvbigncGFzc3dvcmQnKSksICd7e3Jlc29sdmU6c2VjcmV0c21hbmFnZXI6YXJuOjpvZjo6YTo6c2VjcmV0OlNlY3JldFN0cmluZzpwYXNzd29yZDo6fX0nKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYXR0YWNoZWQgc2VjcmV0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBzZWNyZXQgPSBuZXcgc2VjcmV0c21hbmFnZXIuU2VjcmV0KHN0YWNrLCAnU2VjcmV0Jyk7XG4gICAgY29uc3QgdGFyZ2V0OiBzZWNyZXRzbWFuYWdlci5JU2VjcmV0QXR0YWNobWVudFRhcmdldCA9IHtcbiAgICAgIGFzU2VjcmV0QXR0YWNobWVudFRhcmdldDogKCkgPT4gKHtcbiAgICAgICAgdGFyZ2V0SWQ6ICdpbnN0YW5jZScsXG4gICAgICAgIHRhcmdldFR5cGU6IHNlY3JldHNtYW5hZ2VyLkF0dGFjaG1lbnRUYXJnZXRUeXBlLklOU1RBTkNFXG4gICAgICB9KVxuICAgIH07XG5cbiAgICAvLyBXSEVOXG4gICAgc2VjcmV0LmFkZFRhcmdldEF0dGFjaG1lbnQoJ0F0dGFjaGVkU2VjcmV0JywgeyB0YXJnZXQgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6U2VjcmV0c01hbmFnZXI6OlNlY3JldFRhcmdldEF0dGFjaG1lbnQnLCB7XG4gICAgICBTZWNyZXRJZDoge1xuICAgICAgICBSZWY6ICdTZWNyZXRBNzIwRUYwNSdcbiAgICAgIH0sXG4gICAgICBUYXJnZXRJZDogJ2luc3RhbmNlJyxcbiAgICAgIFRhcmdldFR5cGU6ICdBV1M6OlJEUzo6REJJbnN0YW5jZSdcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWRkIGEgcm90YXRpb24gc2NoZWR1bGUgdG8gYW4gYXR0YWNoZWQgc2VjcmV0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBzZWNyZXQgPSBuZXcgc2VjcmV0c21hbmFnZXIuU2VjcmV0KHN0YWNrLCAnU2VjcmV0Jyk7XG4gICAgY29uc3QgdGFyZ2V0OiBzZWNyZXRzbWFuYWdlci5JU2VjcmV0QXR0YWNobWVudFRhcmdldCA9IHtcbiAgICAgIGFzU2VjcmV0QXR0YWNobWVudFRhcmdldDogKCkgPT4gKHtcbiAgICAgICAgdGFyZ2V0SWQ6ICdjbHVzdGVyJyxcbiAgICAgICAgdGFyZ2V0VHlwZTogc2VjcmV0c21hbmFnZXIuQXR0YWNobWVudFRhcmdldFR5cGUuQ0xVU1RFUlxuICAgICAgfSlcbiAgICB9O1xuICAgIGNvbnN0IGF0dGFjaGVkU2VjcmV0ID0gc2VjcmV0LmFkZFRhcmdldEF0dGFjaG1lbnQoJ0F0dGFjaGVkU2VjcmV0JywgeyB0YXJnZXQgfSk7XG4gICAgY29uc3Qgcm90YXRpb25MYW1iZGEgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTGFtYmRhJywge1xuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzhfMTAsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tSW5saW5lKCdleHBvcnQuaGFuZGxlciA9IGV2ZW50ID0+IGV2ZW50OycpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgYXR0YWNoZWRTZWNyZXQuYWRkUm90YXRpb25TY2hlZHVsZSgnUm90YXRpb25TY2hlZHVsZScsIHtcbiAgICAgIHJvdGF0aW9uTGFtYmRhXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6U2VjcmV0c01hbmFnZXI6OlJvdGF0aW9uU2NoZWR1bGUnLCB7XG4gICAgICBTZWNyZXRJZDoge1xuICAgICAgICBSZWY6ICdTZWNyZXRBdHRhY2hlZFNlY3JldDk0MTQ1MzE2JyAvLyBUaGUgc2VjcmV0IHJldHVybmVkIGJ5IHRoZSBhdHRhY2htZW50LCBub3QgdGhlIHNlY3JldCBpdHNlbGYuXG4gICAgICB9XG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Rocm93cyB3aGVuIHNwZWNpZnlpbmcgc2VjcmV0U3RyaW5nVGVtcGxhdGUgYnV0IG5vdCBnZW5lcmF0ZVN0cmluZ0tleScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IHNlY3JldHNtYW5hZ2VyLlNlY3JldChzdGFjaywgJ1NlY3JldCcsIHtcbiAgICAgIGdlbmVyYXRlU2VjcmV0U3RyaW5nOiB7XG4gICAgICAgIHNlY3JldFN0cmluZ1RlbXBsYXRlOiBKU09OLnN0cmluZ2lmeSh7IHVzZXJuYW1lOiAndXNlcm5hbWUnIH0pXG4gICAgICB9XG4gICAgfSksIC9gc2VjcmV0U3RyaW5nVGVtcGxhdGVgLitgZ2VuZXJhdGVTdHJpbmdLZXlgLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndGhyb3dzIHdoZW4gc3BlY2lmeWluZyBnZW5lcmF0ZVN0cmluZ0tleSBidXQgbm90IHNlY3JldFN0cmluZ1RlbXBsYXRlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgc2VjcmV0c21hbmFnZXIuU2VjcmV0KHN0YWNrLCAnU2VjcmV0Jywge1xuICAgICAgZ2VuZXJhdGVTZWNyZXRTdHJpbmc6IHtcbiAgICAgICAgZ2VuZXJhdGVTdHJpbmdLZXk6ICdwYXNzd29yZCdcbiAgICAgIH1cbiAgICB9KSwgL2BzZWNyZXRTdHJpbmdUZW1wbGF0ZWAuK2BnZW5lcmF0ZVN0cmluZ0tleWAvKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdlcXVpdmFsZW5jZSBvZiBTZWNyZXRWYWx1ZSBhbmQgU2VjcmV0LmZyb21TZWNyZXRBdHRyaWJ1dGVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGltcG9ydGVkID0gc2VjcmV0c21hbmFnZXIuU2VjcmV0LmZyb21TZWNyZXRBdHRyaWJ1dGVzKHN0YWNrLCAnSW1wb3J0ZWQnLCB7IHNlY3JldEFybjogJ215LXNlY3JldC1hcm4nIH0pLnNlY3JldFZhbHVlRnJvbUpzb24oJ3Bhc3N3b3JkJyk7XG4gICAgY29uc3QgdmFsdWUgPSBTZWNyZXRWYWx1ZS5zZWNyZXRzTWFuYWdlcignbXktc2VjcmV0LWFybicsIHsganNvbkZpZWxkOiAncGFzc3dvcmQnIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUoaW1wb3J0ZWQpLCBzdGFjay5yZXNvbHZlKHZhbHVlKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH1cbn07XG4iXX0=