"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentStrategy = exports.PercentOrAbsolute = exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const container = require("./container");
const k8s = require("./imports/k8s");
const service = require("./service");
const workload = require("./workload");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends workload.Workload {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'deployments';
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.minReady = props.minReady ?? cdk8s_1.Duration.seconds(0);
        this.progressDeadline = props.progressDeadline ?? cdk8s_1.Duration.seconds(600);
        if (this.progressDeadline.toSeconds() <= this.minReady.toSeconds()) {
            throw new Error(`'progressDeadline' (${this.progressDeadline.toSeconds()}s) must be greater than 'minReady' (${this.minReady.toSeconds()}s)`);
        }
        this.replicas = props.replicas ?? 1;
        this.strategy = props.strategy ?? DeploymentStrategy.rollingUpdate();
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        const ports = options.ports ?? this.extractPorts();
        if (ports.length === 0) {
            throw new Error(`Unable to expose deployment ${this.name} via a service: `
                + 'Deployment port cannot be determined.'
                + 'Either pass \'ports\', or configure ports on the containers of the deployment');
        }
        const metadata = { namespace: this.metadata.namespace };
        if (options.name) {
            metadata.name = options.name;
        }
        return new service.Service(this, `${options.name ?? ''}Service`, {
            selector: this,
            ports,
            metadata,
            type: options.serviceType ?? service.ServiceType.CLUSTER_IP,
        });
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const ser = this.exposeViaService(options);
        return ser.exposeViaIngress(path, options);
    }
    extractPorts() {
        return container.extractContainerPorts(this).map(port => ({ targetPort: port, port }));
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            progressDeadlineSeconds: this.progressDeadline.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: this._toLabelSelector(),
            strategy: this.strategy._toKube(),
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-22.Deployment", version: "2.0.0-rc.77" };
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
class PercentOrAbsolute {
    constructor(value) {
        this.value = value;
    }
    /**
     * Percent ratio.
     */
    static percent(percent) {
        return new PercentOrAbsolute(`${percent}%`);
    }
    /**
     * Absolute number.
     */
    static absolute(num) {
        return new PercentOrAbsolute(num);
    }
    isZero() {
        return this.value === PercentOrAbsolute.absolute(0).value || this.value === PercentOrAbsolute.percent(0).value;
    }
}
exports.PercentOrAbsolute = PercentOrAbsolute;
_b = JSII_RTTI_SYMBOL_1;
PercentOrAbsolute[_b] = { fqn: "cdk8s-plus-22.PercentOrAbsolute", version: "2.0.0-rc.77" };
/**
 * Deployment strategies.
 */
class DeploymentStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate() {
        return new DeploymentStrategy({
            type: 'Recreate',
        });
    }
    static rollingUpdate(options = {}) {
        const maxSurge = options.maxSurge ?? PercentOrAbsolute.percent(25);
        const maxUnavailable = options.maxUnavailable ?? PercentOrAbsolute.percent(25);
        if (maxSurge.isZero() && maxUnavailable.isZero()) {
            throw new Error('\'maxSurge\' and \'maxUnavailable\' cannot be both zero');
        }
        return new DeploymentStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { maxSurge, maxUnavailable },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_c = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_c] = { fqn: "cdk8s-plus-22.DeploymentStrategy", version: "2.0.0-rc.77" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwbG95bWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9kZXBsb3ltZW50LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsaUNBQWtEO0FBRWxELHlDQUF5QztBQUN6QyxxQ0FBcUM7QUFFckMscUNBQXFDO0FBQ3JDLHVDQUF1QztBQWdGdkM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQTJCRztBQUNILE1BQWEsVUFBVyxTQUFRLFFBQVEsQ0FBQyxRQUFRO0lBOEIvQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXlCLEVBQUU7UUFDbkUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFIVixpQkFBWSxHQUFHLGFBQWEsQ0FBQztRQUszQyxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3hELFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtZQUN4QixJQUFJLEVBQUUsWUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQztTQUNsRCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksZ0JBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxnQkFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUV4RSxJQUFJLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxFQUFFO1lBQ2xFLE1BQU0sSUFBSSxLQUFLLENBQUMsdUJBQXVCLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsdUNBQXVDLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxDQUFDO1NBQy9JO1FBRUQsSUFBSSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxJQUFJLENBQUMsQ0FBQztRQUNwQyxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksa0JBQWtCLENBQUMsYUFBYSxFQUFFLENBQUM7SUFDdkUsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQixDQUFDLFVBQTZDLEVBQUU7UUFDckUsTUFBTSxLQUFLLEdBQUcsT0FBTyxDQUFDLEtBQUssSUFBSSxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7UUFDbkQsSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLCtCQUErQixJQUFJLENBQUMsSUFBSSxrQkFBa0I7a0JBQ3RFLHVDQUF1QztrQkFDdkMsK0VBQStFLENBQUMsQ0FBQztTQUN0RjtRQUVELE1BQU0sUUFBUSxHQUFRLEVBQUUsU0FBUyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDN0QsSUFBSSxPQUFPLENBQUMsSUFBSSxFQUFFO1lBQ2hCLFFBQVEsQ0FBQyxJQUFJLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQztTQUM5QjtRQUNELE9BQU8sSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLE9BQU8sQ0FBQyxJQUFJLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDL0QsUUFBUSxFQUFFLElBQUk7WUFDZCxLQUFLO1lBQ0wsUUFBUTtZQUNSLElBQUksRUFBRSxPQUFPLENBQUMsV0FBVyxJQUFJLE9BQU8sQ0FBQyxXQUFXLENBQUMsVUFBVTtTQUM1RCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLGdCQUFnQixDQUFDLElBQVksRUFBRSxVQUE2QyxFQUFFO1FBQ25GLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUMzQyxPQUFPLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVPLFlBQVk7UUFDbEIsT0FBTyxTQUFTLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3pGLENBQUM7SUFFRDs7T0FFRztJQUNJLE9BQU87UUFDWixPQUFPO1lBQ0wsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO1lBQ3ZCLGVBQWUsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRTtZQUMxQyx1QkFBdUIsRUFBRSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxFQUFFO1lBQzFELFFBQVEsRUFBRTtnQkFDUixRQUFRLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUU7Z0JBQ25DLElBQUksRUFBRSxJQUFJLENBQUMsVUFBVSxFQUFFO2FBQ3hCO1lBQ0QsUUFBUSxFQUFFLElBQUksQ0FBQyxnQkFBZ0IsRUFBRTtZQUNqQyxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUU7U0FDbEMsQ0FBQztJQUNKLENBQUM7O0FBNUdILGdDQThHQzs7O0FBdUNEOzs7R0FHRztBQUNILE1BQWEsaUJBQWlCO0lBZ0I1QixZQUFvQyxLQUFVO1FBQVYsVUFBSyxHQUFMLEtBQUssQ0FBSztJQUFHLENBQUM7SUFkbEQ7O09BRUc7SUFDSSxNQUFNLENBQUMsT0FBTyxDQUFDLE9BQWU7UUFDbkMsT0FBTyxJQUFJLGlCQUFpQixDQUFDLEdBQUcsT0FBTyxHQUFHLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsUUFBUSxDQUFDLEdBQVc7UUFDaEMsT0FBTyxJQUFJLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFJTSxNQUFNO1FBQ1gsT0FBTyxJQUFJLENBQUMsS0FBSyxLQUFLLGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLElBQUksSUFBSSxDQUFDLEtBQUssS0FBSyxpQkFBaUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO0lBQ2pILENBQUM7O0FBcEJILDhDQXNCQzs7O0FBRUQ7O0dBRUc7QUFDSCxNQUFhLGtCQUFrQjtJQTRCN0IsWUFBcUMsUUFBZ0M7UUFBaEMsYUFBUSxHQUFSLFFBQVEsQ0FBd0I7SUFBRyxDQUFDO0lBMUJ6RTs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFFBQVE7UUFDcEIsT0FBTyxJQUFJLGtCQUFrQixDQUFDO1lBQzVCLElBQUksRUFBRSxVQUFVO1NBQ2pCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSxNQUFNLENBQUMsYUFBYSxDQUFDLFVBQWtELEVBQUU7UUFFOUUsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLFFBQVEsSUFBSSxpQkFBaUIsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDbkUsTUFBTSxjQUFjLEdBQUcsT0FBTyxDQUFDLGNBQWMsSUFBSSxpQkFBaUIsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFL0UsSUFBSSxRQUFRLENBQUMsTUFBTSxFQUFFLElBQUksY0FBYyxDQUFDLE1BQU0sRUFBRSxFQUFFO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMseURBQXlELENBQUMsQ0FBQztTQUM1RTtRQUVELE9BQU8sSUFBSSxrQkFBa0IsQ0FBQztZQUM1QixJQUFJLEVBQUUsZUFBZTtZQUNyQixhQUFhLEVBQUUsRUFBRSxRQUFRLEVBQUUsY0FBYyxFQUFFO1NBQzVDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFJRDs7T0FFRztJQUNJLE9BQU87UUFDWixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUM7SUFDdkIsQ0FBQzs7QUFuQ0gsZ0RBcUNDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQXBpT2JqZWN0LCBMYXp5LCBEdXJhdGlvbiB9IGZyb20gJ2NkazhzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0ICogYXMgY29udGFpbmVyIGZyb20gJy4vY29udGFpbmVyJztcbmltcG9ydCAqIGFzIGs4cyBmcm9tICcuL2ltcG9ydHMvazhzJztcbmltcG9ydCAqIGFzIGluZ3Jlc3MgZnJvbSAnLi9pbmdyZXNzJztcbmltcG9ydCAqIGFzIHNlcnZpY2UgZnJvbSAnLi9zZXJ2aWNlJztcbmltcG9ydCAqIGFzIHdvcmtsb2FkIGZyb20gJy4vd29ya2xvYWQnO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGBEZXBsb3ltZW50YC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEZXBsb3ltZW50UHJvcHMgZXh0ZW5kcyB3b3JrbG9hZC5Xb3JrbG9hZFByb3BzIHtcblxuICAvKipcbiAgICogTnVtYmVyIG9mIGRlc2lyZWQgcG9kcy5cbiAgICpcbiAgICogQGRlZmF1bHQgMVxuICAgKi9cbiAgcmVhZG9ubHkgcmVwbGljYXM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgc3RyYXRlZ3kgdXNlZCB0byByZXBsYWNlIG9sZCBQb2RzIGJ5IG5ldyBvbmVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFJvbGxpbmdVcGRhdGUgd2l0aCBtYXhTdXJnZSBhbmQgbWF4VW5hdmFpbGFibGUgc2V0IHRvIDI1JS5cbiAgICovXG4gIHJlYWRvbmx5IHN0cmF0ZWd5PzogRGVwbG95bWVudFN0cmF0ZWd5O1xuXG4gIC8qKlxuICAgKiBNaW5pbXVtIGR1cmF0aW9uIGZvciB3aGljaCBhIG5ld2x5IGNyZWF0ZWQgcG9kIHNob3VsZCBiZSByZWFkeSB3aXRob3V0XG4gICAqIGFueSBvZiBpdHMgY29udGFpbmVyIGNyYXNoaW5nLCBmb3IgaXQgdG8gYmUgY29uc2lkZXJlZCBhdmFpbGFibGUuXG4gICAqXG4gICAqIFplcm8gbWVhbnMgdGhlIHBvZCB3aWxsIGJlIGNvbnNpZGVyZWQgYXZhaWxhYmxlIGFzIHNvb24gYXMgaXQgaXMgcmVhZHkuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvd29ya2xvYWRzL2NvbnRyb2xsZXJzL2RlcGxveW1lbnQvI21pbi1yZWFkeS1zZWNvbmRzXG4gICAqIEBkZWZhdWx0IER1cmF0aW9uLnNlY29uZHMoMClcbiAgICovXG4gIHJlYWRvbmx5IG1pblJlYWR5PzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIGR1cmF0aW9uIGZvciBhIGRlcGxveW1lbnQgdG8gbWFrZSBwcm9ncmVzcyBiZWZvcmUgaXRcbiAgICogaXMgY29uc2lkZXJlZCB0byBiZSBmYWlsZWQuIFRoZSBkZXBsb3ltZW50IGNvbnRyb2xsZXIgd2lsbCBjb250aW51ZVxuICAgKiB0byBwcm9jZXNzIGZhaWxlZCBkZXBsb3ltZW50cyBhbmQgYSBjb25kaXRpb24gd2l0aCBhIFByb2dyZXNzRGVhZGxpbmVFeGNlZWRlZFxuICAgKiByZWFzb24gd2lsbCBiZSBzdXJmYWNlZCBpbiB0aGUgZGVwbG95bWVudCBzdGF0dXMuXG4gICAqXG4gICAqIE5vdGUgdGhhdCBwcm9ncmVzcyB3aWxsIG5vdCBiZSBlc3RpbWF0ZWQgZHVyaW5nIHRoZSB0aW1lIGEgZGVwbG95bWVudCBpcyBwYXVzZWQuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvd29ya2xvYWRzL2NvbnRyb2xsZXJzL2RlcGxveW1lbnQvI3Byb2dyZXNzLWRlYWRsaW5lLXNlY29uZHNcbiAgICogQGRlZmF1bHQgRHVyYXRpb24uc2Vjb25kcyg2MDApXG4gICAqL1xuICByZWFkb25seSBwcm9ncmVzc0RlYWRsaW5lPzogRHVyYXRpb247XG5cbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgRGVwbG95bWVudC5leHBvc2VWaWFTZXJ2aWNlYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEZXBsb3ltZW50RXhwb3NlVmlhU2VydmljZU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHBvcnRzIHRoYXQgdGhlIHNlcnZpY2Ugc2hvdWxkIGJpbmQgdG8uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZXh0cmFjdGVkIGZyb20gdGhlIGRlcGxveW1lbnQuXG4gICAqL1xuICByZWFkb25seSBwb3J0cz86IHNlcnZpY2UuU2VydmljZVBvcnRbXTtcblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgdGhlIGV4cG9zZWQgc2VydmljZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBDbHVzdGVySVAuXG4gICAqL1xuICByZWFkb25seSBzZXJ2aWNlVHlwZT86IHNlcnZpY2UuU2VydmljZVR5cGU7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBzZXJ2aWNlIHRvIGV4cG9zZS5cbiAgICogSWYgeW91J2QgbGlrZSB0byBleHBvc2UgdGhlIGRlcGxveW1lbnQgbXVsdGlwbGUgdGltZXMsXG4gICAqIHlvdSBtdXN0IGV4cGxpY2l0bHkgc2V0IGEgbmFtZSBzdGFydGluZyBmcm9tIHRoZSBzZWNvbmQgZXhwb3NlIGNhbGwuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYXV0byBnZW5lcmF0ZWQuXG4gICAqL1xuICByZWFkb25seSBuYW1lPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGV4cG9zaW5nIGEgZGVwbG95bWVudCB2aWEgYW4gaW5ncmVzcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFeHBvc2VEZXBsb3ltZW50VmlhSW5ncmVzc09wdGlvbnMgZXh0ZW5kcyBEZXBsb3ltZW50RXhwb3NlVmlhU2VydmljZU9wdGlvbnMsIHNlcnZpY2UuRXhwb3NlU2VydmljZVZpYUluZ3Jlc3NPcHRpb25zIHt9XG5cbi8qKlxuKlxuKiBBIERlcGxveW1lbnQgcHJvdmlkZXMgZGVjbGFyYXRpdmUgdXBkYXRlcyBmb3IgUG9kcyBhbmQgUmVwbGljYVNldHMuXG4qXG4qIFlvdSBkZXNjcmliZSBhIGRlc2lyZWQgc3RhdGUgaW4gYSBEZXBsb3ltZW50LCBhbmQgdGhlIERlcGxveW1lbnQgQ29udHJvbGxlciBjaGFuZ2VzIHRoZSBhY3R1YWxcbiogc3RhdGUgdG8gdGhlIGRlc2lyZWQgc3RhdGUgYXQgYSBjb250cm9sbGVkIHJhdGUuIFlvdSBjYW4gZGVmaW5lIERlcGxveW1lbnRzIHRvIGNyZWF0ZSBuZXcgUmVwbGljYVNldHMsIG9yIHRvIHJlbW92ZVxuKiBleGlzdGluZyBEZXBsb3ltZW50cyBhbmQgYWRvcHQgYWxsIHRoZWlyIHJlc291cmNlcyB3aXRoIG5ldyBEZXBsb3ltZW50cy5cbipcbiogPiBOb3RlOiBEbyBub3QgbWFuYWdlIFJlcGxpY2FTZXRzIG93bmVkIGJ5IGEgRGVwbG95bWVudC4gQ29uc2lkZXIgb3BlbmluZyBhbiBpc3N1ZSBpbiB0aGUgbWFpbiBLdWJlcm5ldGVzIHJlcG9zaXRvcnkgaWYgeW91ciB1c2UgY2FzZSBpcyBub3QgY292ZXJlZCBiZWxvdy5cbipcbiogVXNlIENhc2VcbiogLS0tLS0tLS0tXG4qXG4qIFRoZSBmb2xsb3dpbmcgYXJlIHR5cGljYWwgdXNlIGNhc2VzIGZvciBEZXBsb3ltZW50czpcbipcbiogLSBDcmVhdGUgYSBEZXBsb3ltZW50IHRvIHJvbGxvdXQgYSBSZXBsaWNhU2V0LiBUaGUgUmVwbGljYVNldCBjcmVhdGVzIFBvZHMgaW4gdGhlIGJhY2tncm91bmQuXG4qICAgQ2hlY2sgdGhlIHN0YXR1cyBvZiB0aGUgcm9sbG91dCB0byBzZWUgaWYgaXQgc3VjY2VlZHMgb3Igbm90LlxuKiAtIERlY2xhcmUgdGhlIG5ldyBzdGF0ZSBvZiB0aGUgUG9kcyBieSB1cGRhdGluZyB0aGUgUG9kVGVtcGxhdGVTcGVjIG9mIHRoZSBEZXBsb3ltZW50LlxuKiAgIEEgbmV3IFJlcGxpY2FTZXQgaXMgY3JlYXRlZCBhbmQgdGhlIERlcGxveW1lbnQgbWFuYWdlcyBtb3ZpbmcgdGhlIFBvZHMgZnJvbSB0aGUgb2xkIFJlcGxpY2FTZXQgdG8gdGhlIG5ldyBvbmUgYXQgYSBjb250cm9sbGVkIHJhdGUuXG4qICAgRWFjaCBuZXcgUmVwbGljYVNldCB1cGRhdGVzIHRoZSByZXZpc2lvbiBvZiB0aGUgRGVwbG95bWVudC5cbiogLSBSb2xsYmFjayB0byBhbiBlYXJsaWVyIERlcGxveW1lbnQgcmV2aXNpb24gaWYgdGhlIGN1cnJlbnQgc3RhdGUgb2YgdGhlIERlcGxveW1lbnQgaXMgbm90IHN0YWJsZS5cbiogICBFYWNoIHJvbGxiYWNrIHVwZGF0ZXMgdGhlIHJldmlzaW9uIG9mIHRoZSBEZXBsb3ltZW50LlxuKiAtIFNjYWxlIHVwIHRoZSBEZXBsb3ltZW50IHRvIGZhY2lsaXRhdGUgbW9yZSBsb2FkLlxuKiAtIFBhdXNlIHRoZSBEZXBsb3ltZW50IHRvIGFwcGx5IG11bHRpcGxlIGZpeGVzIHRvIGl0cyBQb2RUZW1wbGF0ZVNwZWMgYW5kIHRoZW4gcmVzdW1lIGl0IHRvIHN0YXJ0IGEgbmV3IHJvbGxvdXQuXG4qIC0gVXNlIHRoZSBzdGF0dXMgb2YgdGhlIERlcGxveW1lbnQgYXMgYW4gaW5kaWNhdG9yIHRoYXQgYSByb2xsb3V0IGhhcyBzdHVjay5cbiogLSBDbGVhbiB1cCBvbGRlciBSZXBsaWNhU2V0cyB0aGF0IHlvdSBkb24ndCBuZWVkIGFueW1vcmUuXG4qXG4qKi9cbmV4cG9ydCBjbGFzcyBEZXBsb3ltZW50IGV4dGVuZHMgd29ya2xvYWQuV29ya2xvYWQge1xuXG4gIC8qKlxuICAgKiBOdW1iZXIgb2YgZGVzaXJlZCBwb2RzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlcGxpY2FzOiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIE1pbmltdW0gZHVyYXRpb24gZm9yIHdoaWNoIGEgbmV3bHkgY3JlYXRlZCBwb2Qgc2hvdWxkIGJlIHJlYWR5IHdpdGhvdXRcbiAgICogYW55IG9mIGl0cyBjb250YWluZXIgY3Jhc2hpbmcsIGZvciBpdCB0byBiZSBjb25zaWRlcmVkIGF2YWlsYWJsZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBtaW5SZWFkeTogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIGR1cmF0aW9uIGZvciBhIGRlcGxveW1lbnQgdG8gbWFrZSBwcm9ncmVzcyBiZWZvcmUgaXQgaXMgY29uc2lkZXJlZCB0byBiZSBmYWlsZWQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcHJvZ3Jlc3NEZWFkbGluZTogRHVyYXRpb247XG5cbiAgLypcbiAgICogVGhlIHVwZ3JhZGUgc3RyYXRlZ3kgb2YgdGhpcyBkZXBsb3ltZW50LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHN0cmF0ZWd5OiBEZXBsb3ltZW50U3RyYXRlZ3k7XG5cbiAgLyoqXG4gICAqIEBzZWUgYmFzZS5SZXNvdXJjZS5hcGlPYmplY3RcbiAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSBhcGlPYmplY3Q6IEFwaU9iamVjdDtcblxuICBwdWJsaWMgcmVhZG9ubHkgcmVzb3VyY2VUeXBlID0gJ2RlcGxveW1lbnRzJztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRGVwbG95bWVudFByb3BzID0ge30pIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcblxuICAgIHRoaXMuYXBpT2JqZWN0ID0gbmV3IGs4cy5LdWJlRGVwbG95bWVudCh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBtZXRhZGF0YTogcHJvcHMubWV0YWRhdGEsXG4gICAgICBzcGVjOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuX3RvS3ViZSgpIH0pLFxuICAgIH0pO1xuXG4gICAgdGhpcy5taW5SZWFkeSA9IHByb3BzLm1pblJlYWR5ID8/IER1cmF0aW9uLnNlY29uZHMoMCk7XG4gICAgdGhpcy5wcm9ncmVzc0RlYWRsaW5lID0gcHJvcHMucHJvZ3Jlc3NEZWFkbGluZSA/PyBEdXJhdGlvbi5zZWNvbmRzKDYwMCk7XG5cbiAgICBpZiAodGhpcy5wcm9ncmVzc0RlYWRsaW5lLnRvU2Vjb25kcygpIDw9IHRoaXMubWluUmVhZHkudG9TZWNvbmRzKCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJ3Byb2dyZXNzRGVhZGxpbmUnICgke3RoaXMucHJvZ3Jlc3NEZWFkbGluZS50b1NlY29uZHMoKX1zKSBtdXN0IGJlIGdyZWF0ZXIgdGhhbiAnbWluUmVhZHknICgke3RoaXMubWluUmVhZHkudG9TZWNvbmRzKCl9cylgKTtcbiAgICB9XG5cbiAgICB0aGlzLnJlcGxpY2FzID0gcHJvcHMucmVwbGljYXMgPz8gMTtcbiAgICB0aGlzLnN0cmF0ZWd5ID0gcHJvcHMuc3RyYXRlZ3kgPz8gRGVwbG95bWVudFN0cmF0ZWd5LnJvbGxpbmdVcGRhdGUoKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFeHBvc2UgYSBkZXBsb3ltZW50IHZpYSBhIHNlcnZpY2UuXG4gICAqXG4gICAqIFRoaXMgaXMgZXF1aXZhbGVudCB0byBydW5uaW5nIGBrdWJlY3RsIGV4cG9zZSBkZXBsb3ltZW50IDxkZXBsb3ltZW50LW5hbWU+YC5cbiAgICpcbiAgICogQHBhcmFtIG9wdGlvbnMgT3B0aW9ucyB0byBkZXRlcm1pbmUgZGV0YWlscyBvZiB0aGUgc2VydmljZSBhbmQgcG9ydCBleHBvc2VkLlxuICAgKi9cbiAgcHVibGljIGV4cG9zZVZpYVNlcnZpY2Uob3B0aW9uczogRGVwbG95bWVudEV4cG9zZVZpYVNlcnZpY2VPcHRpb25zID0ge30pOiBzZXJ2aWNlLlNlcnZpY2Uge1xuICAgIGNvbnN0IHBvcnRzID0gb3B0aW9ucy5wb3J0cyA/PyB0aGlzLmV4dHJhY3RQb3J0cygpO1xuICAgIGlmIChwb3J0cy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5hYmxlIHRvIGV4cG9zZSBkZXBsb3ltZW50ICR7dGhpcy5uYW1lfSB2aWEgYSBzZXJ2aWNlOiBgXG4gICAgICAgICsgJ0RlcGxveW1lbnQgcG9ydCBjYW5ub3QgYmUgZGV0ZXJtaW5lZC4nXG4gICAgICAgICsgJ0VpdGhlciBwYXNzIFxcJ3BvcnRzXFwnLCBvciBjb25maWd1cmUgcG9ydHMgb24gdGhlIGNvbnRhaW5lcnMgb2YgdGhlIGRlcGxveW1lbnQnKTtcbiAgICB9XG5cbiAgICBjb25zdCBtZXRhZGF0YTogYW55ID0geyBuYW1lc3BhY2U6IHRoaXMubWV0YWRhdGEubmFtZXNwYWNlIH07XG4gICAgaWYgKG9wdGlvbnMubmFtZSkge1xuICAgICAgbWV0YWRhdGEubmFtZSA9IG9wdGlvbnMubmFtZTtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBzZXJ2aWNlLlNlcnZpY2UodGhpcywgYCR7b3B0aW9ucy5uYW1lID8/ICcnfVNlcnZpY2VgLCB7XG4gICAgICBzZWxlY3RvcjogdGhpcyxcbiAgICAgIHBvcnRzLFxuICAgICAgbWV0YWRhdGEsXG4gICAgICB0eXBlOiBvcHRpb25zLnNlcnZpY2VUeXBlID8/IHNlcnZpY2UuU2VydmljZVR5cGUuQ0xVU1RFUl9JUCxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFeHBvc2UgYSBkZXBsb3ltZW50IHZpYSBhbiBpbmdyZXNzLlxuICAgKlxuICAgKiBUaGlzIHdpbGwgZmlyc3QgZXhwb3NlIHRoZSBkZXBsb3ltZW50IHdpdGggYSBzZXJ2aWNlLCBhbmQgdGhlbiBleHBvc2UgdGhlIHNlcnZpY2UgdmlhIGFuIGluZ3Jlc3MuXG4gICAqXG4gICAqIEBwYXJhbSBwYXRoIFRoZSBpbmdyZXNzIHBhdGggdG8gcmVnaXN0ZXIgdW5kZXIuXG4gICAqIEBwYXJhbSBvcHRpb25zIEFkZGl0aW9uYWwgb3B0aW9ucy5cbiAgICovXG4gIHB1YmxpYyBleHBvc2VWaWFJbmdyZXNzKHBhdGg6IHN0cmluZywgb3B0aW9uczogRXhwb3NlRGVwbG95bWVudFZpYUluZ3Jlc3NPcHRpb25zID0ge30pOiBpbmdyZXNzLkluZ3Jlc3Mge1xuICAgIGNvbnN0IHNlciA9IHRoaXMuZXhwb3NlVmlhU2VydmljZShvcHRpb25zKTtcbiAgICByZXR1cm4gc2VyLmV4cG9zZVZpYUluZ3Jlc3MocGF0aCwgb3B0aW9ucyk7XG4gIH1cblxuICBwcml2YXRlIGV4dHJhY3RQb3J0cygpOiBzZXJ2aWNlLlNlcnZpY2VQb3J0W10ge1xuICAgIHJldHVybiBjb250YWluZXIuZXh0cmFjdENvbnRhaW5lclBvcnRzKHRoaXMpLm1hcChwb3J0ID0+ICh7IHRhcmdldFBvcnQ6IHBvcnQsIHBvcnQgfSkpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogazhzLkRlcGxveW1lbnRTcGVjIHtcbiAgICByZXR1cm4ge1xuICAgICAgcmVwbGljYXM6IHRoaXMucmVwbGljYXMsXG4gICAgICBtaW5SZWFkeVNlY29uZHM6IHRoaXMubWluUmVhZHkudG9TZWNvbmRzKCksXG4gICAgICBwcm9ncmVzc0RlYWRsaW5lU2Vjb25kczogdGhpcy5wcm9ncmVzc0RlYWRsaW5lLnRvU2Vjb25kcygpLFxuICAgICAgdGVtcGxhdGU6IHtcbiAgICAgICAgbWV0YWRhdGE6IHRoaXMucG9kTWV0YWRhdGEudG9Kc29uKCksXG4gICAgICAgIHNwZWM6IHRoaXMuX3RvUG9kU3BlYygpLFxuICAgICAgfSxcbiAgICAgIHNlbGVjdG9yOiB0aGlzLl90b0xhYmVsU2VsZWN0b3IoKSxcbiAgICAgIHN0cmF0ZWd5OiB0aGlzLnN0cmF0ZWd5Ll90b0t1YmUoKSxcbiAgICB9O1xuICB9XG5cbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgRGVwbG95bWVudFN0cmF0ZWd5LnJvbGxpbmdVcGRhdGVgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIERlcGxveW1lbnRTdHJhdGVneVJvbGxpbmdVcGRhdGVPcHRpb25zIHtcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gbnVtYmVyIG9mIHBvZHMgdGhhdCBjYW4gYmUgc2NoZWR1bGVkIGFib3ZlIHRoZSBkZXNpcmVkIG51bWJlciBvZiBwb2RzLlxuICAgKiBWYWx1ZSBjYW4gYmUgYW4gYWJzb2x1dGUgbnVtYmVyIChleDogNSkgb3IgYSBwZXJjZW50YWdlIG9mIGRlc2lyZWQgcG9kcyAoZXg6IDEwJSkuXG4gICAqIEFic29sdXRlIG51bWJlciBpcyBjYWxjdWxhdGVkIGZyb20gcGVyY2VudGFnZSBieSByb3VuZGluZyB1cC5cbiAgICogVGhpcyBjYW4gbm90IGJlIDAgaWYgYG1heFVuYXZhaWxhYmxlYCBpcyAwLlxuICAgKlxuICAgKiBFeGFtcGxlOiB3aGVuIHRoaXMgaXMgc2V0IHRvIDMwJSwgdGhlIG5ldyBSZXBsaWNhU2V0IGNhbiBiZSBzY2FsZWQgdXAgaW1tZWRpYXRlbHkgd2hlbiB0aGUgcm9sbGluZyB1cGRhdGVcbiAgICogc3RhcnRzLCBzdWNoIHRoYXQgdGhlIHRvdGFsIG51bWJlciBvZiBvbGQgYW5kIG5ldyBwb2RzIGRvIG5vdCBleGNlZWQgMTMwJSBvZiBkZXNpcmVkIHBvZHMuXG4gICAqIE9uY2Ugb2xkIHBvZHMgaGF2ZSBiZWVuIGtpbGxlZCwgbmV3IFJlcGxpY2FTZXQgY2FuIGJlIHNjYWxlZCB1cCBmdXJ0aGVyLCBlbnN1cmluZyB0aGF0XG4gICAqIHRvdGFsIG51bWJlciBvZiBwb2RzIHJ1bm5pbmcgYXQgYW55IHRpbWUgZHVyaW5nIHRoZSB1cGRhdGUgaXMgYXQgbW9zdCAxMzAlIG9mIGRlc2lyZWQgcG9kcy5cbiAgICpcbiAgICogQGRlZmF1bHQgJzI1JSdcbiAgICovXG4gIHJlYWRvbmx5IG1heFN1cmdlPzogUGVyY2VudE9yQWJzb2x1dGU7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIG51bWJlciBvZiBwb2RzIHRoYXQgY2FuIGJlIHVuYXZhaWxhYmxlIGR1cmluZyB0aGUgdXBkYXRlLlxuICAgKiBWYWx1ZSBjYW4gYmUgYW4gYWJzb2x1dGUgbnVtYmVyIChleDogNSkgb3IgYSBwZXJjZW50YWdlIG9mIGRlc2lyZWQgcG9kcyAoZXg6IDEwJSkuXG4gICAqIEFic29sdXRlIG51bWJlciBpcyBjYWxjdWxhdGVkIGZyb20gcGVyY2VudGFnZSBieSByb3VuZGluZyBkb3duLlxuICAgKiBUaGlzIGNhbiBub3QgYmUgMCBpZiBgbWF4U3VyZ2VgIGlzIDAuXG4gICAqXG4gICAqIEV4YW1wbGU6IHdoZW4gdGhpcyBpcyBzZXQgdG8gMzAlLCB0aGUgb2xkIFJlcGxpY2FTZXQgY2FuIGJlIHNjYWxlZCBkb3duIHRvIDcwJSBvZiBkZXNpcmVkXG4gICAqIHBvZHMgaW1tZWRpYXRlbHkgd2hlbiB0aGUgcm9sbGluZyB1cGRhdGUgc3RhcnRzLiBPbmNlIG5ldyBwb2RzIGFyZSByZWFkeSwgb2xkIFJlcGxpY2FTZXQgY2FuXG4gICAqIGJlIHNjYWxlZCBkb3duIGZ1cnRoZXIsIGZvbGxvd2VkIGJ5IHNjYWxpbmcgdXAgdGhlIG5ldyBSZXBsaWNhU2V0LCBlbnN1cmluZyB0aGF0IHRoZSB0b3RhbFxuICAgKiBudW1iZXIgb2YgcG9kcyBhdmFpbGFibGUgYXQgYWxsIHRpbWVzIGR1cmluZyB0aGUgdXBkYXRlIGlzIGF0IGxlYXN0IDcwJSBvZiBkZXNpcmVkIHBvZHMuXG4gICAqXG4gICAqIEBkZWZhdWx0ICcyNSUnXG4gICAqL1xuICByZWFkb25seSBtYXhVbmF2YWlsYWJsZT86IFBlcmNlbnRPckFic29sdXRlO1xuXG59XG5cbi8qKlxuICogVW5pb24gbGlrZSBjbGFzcyByZXBzZW50aW5nIGVpdGhlciBhIHJhdGlvbiBpblxuICogcGVyY2VudHMgb3IgYW4gYWJzb2x1dGUgbnVtYmVyLlxuICovXG5leHBvcnQgY2xhc3MgUGVyY2VudE9yQWJzb2x1dGUge1xuXG4gIC8qKlxuICAgKiBQZXJjZW50IHJhdGlvLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBwZXJjZW50KHBlcmNlbnQ6IG51bWJlcik6IFBlcmNlbnRPckFic29sdXRlIHtcbiAgICByZXR1cm4gbmV3IFBlcmNlbnRPckFic29sdXRlKGAke3BlcmNlbnR9JWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFic29sdXRlIG51bWJlci5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYWJzb2x1dGUobnVtOiBudW1iZXIpOiBQZXJjZW50T3JBYnNvbHV0ZSB7XG4gICAgcmV0dXJuIG5ldyBQZXJjZW50T3JBYnNvbHV0ZShudW0pO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgdmFsdWU6IGFueSkge31cblxuICBwdWJsaWMgaXNaZXJvKCk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLnZhbHVlID09PSBQZXJjZW50T3JBYnNvbHV0ZS5hYnNvbHV0ZSgwKS52YWx1ZSB8fCB0aGlzLnZhbHVlID09PSBQZXJjZW50T3JBYnNvbHV0ZS5wZXJjZW50KDApLnZhbHVlO1xuICB9XG5cbn1cblxuLyoqXG4gKiBEZXBsb3ltZW50IHN0cmF0ZWdpZXMuXG4gKi9cbmV4cG9ydCBjbGFzcyBEZXBsb3ltZW50U3RyYXRlZ3kge1xuXG4gIC8qKlxuICAgKiBBbGwgZXhpc3RpbmcgUG9kcyBhcmUga2lsbGVkIGJlZm9yZSBuZXcgb25lcyBhcmUgY3JlYXRlZC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy93b3JrbG9hZHMvY29udHJvbGxlcnMvZGVwbG95bWVudC8jcmVjcmVhdGUtZGVwbG95bWVudFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWNyZWF0ZSgpOiBEZXBsb3ltZW50U3RyYXRlZ3kge1xuICAgIHJldHVybiBuZXcgRGVwbG95bWVudFN0cmF0ZWd5KHtcbiAgICAgIHR5cGU6ICdSZWNyZWF0ZScsXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIHJvbGxpbmdVcGRhdGUob3B0aW9uczogRGVwbG95bWVudFN0cmF0ZWd5Um9sbGluZ1VwZGF0ZU9wdGlvbnMgPSB7fSk6IERlcGxveW1lbnRTdHJhdGVneSB7XG5cbiAgICBjb25zdCBtYXhTdXJnZSA9IG9wdGlvbnMubWF4U3VyZ2UgPz8gUGVyY2VudE9yQWJzb2x1dGUucGVyY2VudCgyNSk7XG4gICAgY29uc3QgbWF4VW5hdmFpbGFibGUgPSBvcHRpb25zLm1heFVuYXZhaWxhYmxlID8/IFBlcmNlbnRPckFic29sdXRlLnBlcmNlbnQoMjUpO1xuXG4gICAgaWYgKG1heFN1cmdlLmlzWmVybygpICYmIG1heFVuYXZhaWxhYmxlLmlzWmVybygpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1xcJ21heFN1cmdlXFwnIGFuZCBcXCdtYXhVbmF2YWlsYWJsZVxcJyBjYW5ub3QgYmUgYm90aCB6ZXJvJyk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBEZXBsb3ltZW50U3RyYXRlZ3koe1xuICAgICAgdHlwZTogJ1JvbGxpbmdVcGRhdGUnLFxuICAgICAgcm9sbGluZ1VwZGF0ZTogeyBtYXhTdXJnZSwgbWF4VW5hdmFpbGFibGUgfSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBzdHJhdGVneTogazhzLkRlcGxveW1lbnRTdHJhdGVneSkge31cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvS3ViZSgpOiBrOHMuRGVwbG95bWVudFN0cmF0ZWd5IHtcbiAgICByZXR1cm4gdGhpcy5zdHJhdGVneTtcbiAgfVxuXG59Il19