"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.sonarqubeScanner = exports.createSonarqubeProject = exports.generateSonarqubeReports = void 0;
const cube = (path, action = "GET", args) => `curl -X ${action} -u $SONARQUBE_TOKEN: "$SONARQUBE_ENDPOINT${path}" ${args || ""}`;
const setupSonarqubeQualityGates = (defaultProfileOrGateName, specificProfileOrGateName) => [
    `export DEFAULT_GATE=\`${cube(`/api/qualitygates/search?gateName=${defaultProfileOrGateName}`)}\``,
    `export SPECIFIC_GATE=\`${cube(`/api/qualitygates/search?gateName=${specificProfileOrGateName}`)}\``,
    `if [[ "$(echo $SPECIFIC_GATE | jq .errors)" == "null" && "$(echo $SPECIFIC_GATE | jq '.results | length')" -gt 0 ]]; then export GATE_NAME=${specificProfileOrGateName}; else export GATE_NAME=${defaultProfileOrGateName}; fi`,
    `${cube("/api/qualitygates/select?projectKey=$PROJECT_NAME&gateName=$GATE_NAME", "POST")}`,
];
const setupSonarqubeQualityProfiles = (defaultProfileOrGateName, specificProfileOrGateName) => [
    `export DEFAULT_PROFILE=\`${cube(`/api/qualityprofiles/search?qualityProfile=${defaultProfileOrGateName}`)} | jq .profiles\``,
    `export SPECIFIC_PROFILE=\`${cube(`/api/qualityprofiles/search?qualityProfile=${specificProfileOrGateName}`)} | jq .profiles\``,
    `export MERGED_PROFILES=\`jq --argjson arr1 "$DEFAULT_PROFILE" --argjson arr2 "$SPECIFIC_PROFILE" -n '$arr1 + $arr2 | group_by(.language) | map(.[-1])'\``,
    `echo $MERGED_PROFILES | jq -c '.[]' | while read i; do ${cube("/api/qualityprofiles/add_project?project=$PROJECT_NAME&language=`echo $i | jq -r .language`&qualityProfile=`echo $i | jq -r .name`", "POST")}; done`,
];
const setupSonarqubePermissions = (authorizedGroup) => !authorizedGroup
    ? []
    : [
        "admin",
        "codeviewer",
        "issueadmin",
        "securityhotspotadmin",
        "scan",
        "user",
    ].map((p) => `${cube(`/api/permissions/add_group?projectKey=$PROJECT_NAME&groupName=${authorizedGroup}&permission=${p}`, "POST")}`);
const setupSonarqubeProject = ({ sonarqubeTags, sonarqubeAuthorizedGroup, sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName, }) => {
    return [
        ...setupSonarqubePermissions(sonarqubeAuthorizedGroup),
        `${cube("/api/project_branches/rename?project=$PROJECT_NAME&name=mainline", "POST")}`,
        `${cube(`/api/project_tags/set?project=$PROJECT_NAME&tags=${[
            sonarqubeAuthorizedGroup,
            ...(sonarqubeTags || []),
        ].join(",")}`, "POST")}`,
        ...setupSonarqubeQualityProfiles(sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName),
        ...setupSonarqubeQualityGates(sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName),
    ].join(";");
};
exports.generateSonarqubeReports = () => [
    cube("/api/bitegarden/report/pdf_issues_breakdown?resource=$PROJECT_NAME&branch=mainline", "GET", "--output reports/prototype-issues-report.pdf"),
    cube("/api/bitegarden/report/pdf?resource=$PROJECT_NAME&branch=mainline", "GET", "--output reports/prototype-executive-report.pdf"),
    cube("/api/security_reports/download?project=$PROJECT_NAME", "GET", "--output reports/prototype-security-report.pdf"),
];
exports.createSonarqubeProject = (props) => [
    `CREATE_PROJECT_OUTPUT=\`${cube("/api/projects/create?name=$PROJECT_NAME&project=$PROJECT_NAME&visibility=private", "POST")}\``,
    `if [[ "$(echo $CREATE_PROJECT_OUTPUT | jq .errors)" == "null" ]]; then ${setupSonarqubeProject(props)}; fi;`,
];
exports.sonarqubeScanner = (excludeGlobsForScan) => [
    "npx sonarqube-scanner -Dsonar.login=$SONARQUBE_TOKEN",
    "-Dsonar.projectKey=$PROJECT_NAME",
    "-Dsonar.projectName=$PROJECT_NAME",
    "-Dsonar.projectVersion=`echo $RESOLVED_SOURCE_VERSION | cut -c1-7`",
    "-Dsonar.branch.name=mainline",
    "-Dsonar.host.url=$SONARQUBE_ENDPOINT",
    "-Dsonar.cfn.nag.reportFiles=reports/cfn-nag-report.json",
    "-Dsonar.dependencyCheck.htmlReportPath=reports/dependency-check-report.html",
    "-Dsonar.javascript.lcov.reportPaths=**/coverage/lcov.info",
    "-Dsonar.clover.reportPath=**/coverage/clover.xml",
    `-Dsonar.exclusions="**/reports/**,**/coverage/**${excludeGlobsForScan ? `,${excludeGlobsForScan.join(",")}` : ""}"`,
    "-Dsonar.sources=.",
].join(" ");
//# sourceMappingURL=data:application/json;base64,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