"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const api = require("@aws-cdk/aws-apigateway");
test('snapshot test ApiGatewayToDynamoDB default params', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {};
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb-default', apiGatewayToDynamoDBProps);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check properties', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {};
    const construct = new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb-default', apiGatewayToDynamoDBProps);
    expect(construct.dynamoTable !== null);
    expect(construct.apiGateway !== null);
    expect(construct.apiGatewayRole !== null);
    expect(construct.apiGatewayCloudWatchRole !== null);
    expect(construct.apiGatewayLogGroup !== null);
});
test('check allow CRUD operations', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowReadOperation: true,
        allowCreateOperation: true,
        createRequestTemplate: "{}",
        allowDeleteOperation: true,
        allowUpdateOperation: true,
        updateRequestTemplate: "{}"
    };
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb', apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:PutItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:DeleteItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4"
            }
        ]
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "PUT",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: "{id}",
    });
});
test('check allow read and update only', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowUpdateOperation: true,
        updateRequestTemplate: "{}"
    };
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb', apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4"
            }
        ]
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
});
test('check using custom partition key for dynamodb', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        dynamoTableProps: {
            partitionKey: {
                name: 'page_id',
                type: aws_dynamodb_1.AttributeType.STRING
            }
        }
    };
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb', apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: "{page_id}",
    });
});
test('override apiGatewayProps for api gateway', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        apiGatewayProps: {
            description: 'This is a sample description for api gateway'
        }
    };
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb', apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::ApiGateway::RestApi", {
        Description: "This is a sample description for api gateway"
    });
});
test('Test deployment ApiGateway AuthorizationType override', () => {
    const stack = new core_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "NONE"
    });
});
//# sourceMappingURL=data:application/json;base64,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