# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_models_redis_cache', 'django_models_redis_cache.tests']

package_data = \
{'': ['*']}

install_requires = \
['Django>=3.0,<4.0', 'redis>=3.5.3,<4.0.0']

setup_kwargs = {
    'name': 'django-models-redis-cache',
    'version': '1.8.7',
    'description': 'Django Models Redis Cache (DMoReCa), library that gives your specified Django models regular caching via Redis',
    'long_description': '# django-models-redis-cache\n\n## **Django Models Redis Cache (DMoReCa), library that gives your specified Django models regular caching via Redis**\n\nFor one project, I needed to work with redis, but redis-py provides a minimum level of work with redis. I didn\'t find any Django-like ORM for redis, so I wrote library [python-redis-orm](https://github.com/gh0st-work/python_redis_orm/) ([PyPI](https://pypi.org/project/python-redis-orm/)).\n\nAlso, if you are searching for just django-like redis ORM, please check [django-redis-orm](https://github.com/gh0st-work/django_redis_orm/) ([PyPI](https://pypi.org/project/django-redis-orm/)).\n\n**And this library is port to Django that provides easy-to-use Django models caching via Redis**.\n\n### Working with this library, you are expected:\n\n- Fully works in 2021\n- Saving almost all types of fields automatically\n- Efficient data storage (SET model_name:instance_id "JSON string")\n- Async caching\n- Connection pooling\n- Easy adaptation to your needs\n- Adequate informational messages and error messages\n- Built-in RedisRoot class that stores specified models, with:\n    - **prefix** setting - prefix of this RedisRoot to be stored in redis\n    - **connection_pool** setting - your redis.ConnectionPool instance (from redis-py)\n    - **async_db_requests_limit** setting - your database connections limit\n    - **ignore_deserialization_errors** setting - do not raise errors, while deserializing data\n    - **save_consistency** setting - show structure-first data\n    - **economy** setting - to not return full data and save some requests (usually, speeds up your app on 80%)\n- Customizing caching settings by model:\n    - **enabled** setting - to cache or not\n    - **ttl** setting - cache period\n    - **save_related_models** setting - save ForeignKey-s and ManyToMany-s instances\n    - **exclude_fields** setting - field names to be excluded from caching\n    - **filter_by** - setting - only models that passed filter params will be cached\n- CRUD (Create Read Update Delete), in our variation: save, get, filter, order, update, delete:\n    - `example_instance = ExampleModel(example_field=\'example_data\').save()` - to create an instance and get its data dict\n    - `filtered_example_instances = redis_root.get(ExampleModel, example_field=\'example_data\')` - to get all ExampleModel instances with example_field filter and get its data dict\n    - `ordered_instances = redis_root.order(filtered_example_instances, \'-id\')` - to get ordered filtered_example_instances by id (\'-\' for reverse)\n    - `updated_example_instances = redis_root.update(ExampleModel, ordered_instances, example_field=\'another_example_data\')` - to update all ordered_instances example_field with value \'another_example_data\' and get its data dict\n    - `redis_root.delete(ExampleModel, updated_example_instances)` - to delete updated_example_instances\n\n# Installation\n`pip install django-models-redis-cache`\n\n[Here is PyPI](https://pypi.org/project/django-models-redis-cache/)\n\nAdd "django_models_redis_cache" to your INSTALLED_APPS setting like this::\n\n    INSTALLED_APPS = [\n        ...\n        \'django_models_redis_cache\',\n    ]\n\n# Usage\n\n1. Create **RedisRoot** with params:\n    - **prefix** - (str) prefix for your redis root\n    - **connection_pool** - (redis.ConnectionPool) redis-py redis.ConnectionPool instance, with decode_responses=True\n    - **async_db_requests_limit** - (int) your database has max connections limit, please enter it here\n    - **ignore_deserialization_errors** - (bool) to ignore deserialization errors or raise exception\n    - **economy** - (bool) if True, all create/update requests will return only instance id \n2. Call **register_django_models({...})** on your RedisRoot instance and provide dict, where keys are django models and values are dicts (django_model:dict) with config params (str:value):\n    - **enabled** - (bool) - to cache or not\n    - **ttl** - (int) - to cache every x seconds\n    - **save_related_models** - (bool) - to save ForeignKey-s and ManyToMany-s instances or not\n    - **exclude_fields** - (list of strings) - fields to exclude from caching\n    - **filter_by** - (dict str:value) - filter objects to cache by something\n3. Call **check_cache()** on your RedisRoot instance\n\n# Example usage\n\n### Settings\n\nYou can set this part in your project settings.py:\n\n```python\nfrom django_models_redis_cache.core import *\n\n\ndef get_connection_pool():\n    host = \'localhost\'\n    port = 6379\n    db = 0\n    connection_pool = redis.ConnectionPool(\n        decode_responses=True,\n        host=host,\n        port=port,\n        db=db,\n    )\n    return connection_pool\n\n\nREDIS_ROOTS = {\n    \'test_caching_root\': RedisRoot(\n        prefix=\'test_caching\',\n        connection_pool=get_connection_pool(),\n        async_db_requests_limit=100,\n        ignore_deserialization_errors=True,\n        save_consistency=False,\n        economy=True\n    )\n}\n\n```\n\n### Run in the background\n\nYou can just copy it to:\n\n`app/management/commands/command_name.py`\n    \nAnd just run with:\n\n`python manage.py command_name`\n    \nHelp:\n\n[Django custom management commands](https://docs.djangoproject.com/en/3.2/howto/custom-management-commands/)\n    \n[How to import something from settings](https://stackoverflow.com/a/14617309)\n\n```python\n\nredis_roots = getattr(settings, "REDIS_ROOTS", None)\nif redis_roots:\n    if type(redis_roots) == dict:\n        test_caching_root = redis_roots[\'test_caching_root\']\n        test_caching_root.register_django_models({\n            CustomUser: {\n                \'enabled\': True,\n                \'ttl\': 60 * 15,\n                \'save_related_models\': True,\n                \'exclude_fields\': [\n                    \'is_admin\',\n                    \'api_key\',\n                    \'first_name\',\n                    \'last_name\',\n                    \'email\',\n                    \'is_staff\',\n                    \'date_joined\',\n                    \'password\',\n                    \'last_login\',\n                    \'user_permissions\',\n                    \'is_superuser\',\n                    \'groups\',\n                ],\n            },\n            BotSoft: {\n                \'enabled\': True,\n                \'ttl\': 60 * 15,\n                \'save_related_models\': True,\n                \'exclude_fields\': [\n                    \'name\',\n                    \'image\',\n                    \'image_webp\',\n                    \'developer_url\'\n                ],\n            },\n            Service: {\n                \'enabled\': True,\n                \'ttl\': 60 * 15,\n                \'save_related_models\': True,\n                \'exclude_fields\': [\n                    \'name_append\',\n                    \'description\',\n                    \'min\',\n                    \'max\',\n                ],\n            },\n            CustomService: {\n                \'enabled\': True,\n                \'ttl\': 60 * 15,\n                \'save_related_models\': True,\n                \'exclude_fields\': [\n                    \'name_append\',\n                ],\n            },\n            UniqueTask: {\n                \'enabled\': True,\n                \'ttl\': 60 * 5,\n                \'save_related_models\': True,\n            },\n            Task: {\n                \'enabled\': True,\n                \'ttl\': 60 * 5,\n                \'save_related_models\': False,\n                \'filter_by\': {\n                    \'status\': \'in_work\',\n                }\n            },\n            Account: {\n                \'enabled\': True,\n                \'ttl\': 60 * 5,\n                \'save_related_models\': True,\n                \'filter_by\': {\n                    \'last_task_completed_in__gte\': datetime.datetime.now() - datetime.timedelta(days=14),\n                    \'last_checked_in__gte\': datetime.datetime.now() - datetime.timedelta(days=14),\n                }\n            },\n            BotSession: {\n                \'enabled\': True,\n                \'ttl\': 60 * 5,\n                \'save_related_models\': True,\n            },\n            TaskChallenge: {\n                \'enabled\': True,\n                \'ttl\': 60 * 1,\n                \'save_related_models\': True,\n            },\n        })\n        roots_to_cache = [\n            test_caching_root,\n        ]\n        print(\'STARTING CACHING\')\n        while True:\n            for redis_root in roots_to_cache:\n                redis_root.check_cache()\n    else:\n        raise Exception(\'redis_roots must be dict\')\nelse:\n    raise Exception(\'No redis_roots\')\n\n```\n\n',
    'author': 'Anton Nechaev',
    'author_email': 'antonnechaev990@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/gh0st-work/django_models_redis_cache',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
