"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.hasDependencies = exports.bundle = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
function bundle(options) {
    const { entry, runtime, outputPathSuffix } = options;
    const hasDeps = hasDependencies(entry);
    const depsCommand = chain([
        hasDeps ? `rsync -r ${exports.BUNDLER_DEPENDENCIES_CACHE}/. ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}` : '',
        `rsync -r . ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}`,
    ]);
    // Determine which dockerfile to use. When dependencies are present, we use a
    // Dockerfile that can create a cacheable layer. We can't use this Dockerfile
    // if there aren't dependencies or the Dockerfile will complain about missing
    // sources.
    const dockerfile = hasDeps
        ? 'Dockerfile.dependencies'
        : 'Dockerfile';
    const image = cdk.BundlingDockerImage.fromAsset(entry, {
        buildArgs: {
            IMAGE: runtime.bundlingDockerImage.image,
        },
        file: path.join(__dirname, dockerfile),
    });
    return lambda.Code.fromAsset(entry, {
        assetHashType: cdk.AssetHashType.BUNDLE,
        exclude: exports.DEPENDENCY_EXCLUDES,
        bundling: {
            image,
            command: ['bash', '-c', depsCommand],
        },
    });
}
exports.bundle = bundle;
/**
 * Checks to see if the `entry` directory contains a type of dependency that
 * we know how to install.
 */
function hasDependencies(entry) {
    if (fs.existsSync(path.join(entry, 'Pipfile'))) {
        return true;
    }
    if (fs.existsSync(path.join(entry, 'poetry.lock'))) {
        return true;
    }
    if (fs.existsSync(path.join(entry, 'requirements.txt'))) {
        return true;
    }
    return false;
}
exports.hasDependencies = hasDependencies;
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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