//! # DDEX Builder - Deterministic DDEX XML Generation
//!
//! A high-performance, memory-safe DDEX XML builder that generates deterministic,
//! deterministic XML using DB-C14N/1.0 canonicalization. Built in Rust with
//! comprehensive security features and bindings for JavaScript, Python, and WebAssembly.
//!
//! ## Key Features
//!
//! - **🔒 Security First**: XXE protection, input validation, rate limiting, and comprehensive security measures
//! - **⚡ High Performance**: Sub-millisecond generation for typical releases, memory-optimized streaming
//! - **🎯 Deterministic Output**: Guaranteed reproducible output using DB-C14N/1.0
//! - **🔄 Round-trip Fidelity**: Perfect compatibility with ddex-parser for Parse → Build → Parse workflows
//! - **🛠️ Partner Presets**: Pre-configured settings for Spotify, YouTube, Apple Music, and other platforms
//! - **🌐 Multi-platform**: Native Rust, Node.js, Python, and WebAssembly bindings
//! - **📊 Version Support**: Full support for ERN 3.8.2, 4.2, 4.3 with automatic conversion
//!
//! ## Quick Start
//!
//! ```rust
//! use ddex_builder::{Builder, DdexVersion};
//! use ddex_builder::builder::{BuildRequest, OutputFormat};
//!
//! // Create a builder with Spotify preset
//! let mut builder = Builder::new();
//! builder.preset("spotify_audio_43")?;
//!
//! // Build DDEX XML
//! let request = BuildRequest {
//!     source_xml: r#"<SoundRecording>...</SoundRecording>"#.to_string(),
//!     output_format: OutputFormat::Xml,
//!     preset: Some("spotify_audio_43".to_string()),
//!     validate_schema: true,
//! };
//!
//! let result = builder.build_internal(&request)?;
//! println!("Generated DDEX XML: {}", result.xml);
//! # Ok::<(), ddex_builder::BuildError>(())
//! ```
//!
//! ## Security Features
//!
//! DDEX Builder includes comprehensive security measures:
//!
//! ```rust
//! use ddex_builder::{InputValidator, SecurityConfig, ApiSecurityManager};
//!
//! // Configure security settings
//! let security_config = SecurityConfig {
//!     max_xml_size: 10_000_000,        // 10MB limit
//!     max_json_depth: 32,              // Prevent deep nesting attacks
//!     rate_limiting_enabled: true,
//!     max_requests_per_minute: 100,
//!     validate_urls: true,
//!     block_private_ips: true,
//!     ..Default::default()
//! };
//!
//! // Validate inputs
//! let validator = InputValidator::new(security_config.clone());
//! validator.validate_xml_content(&xml_input)?;
//!
//! // API security management
//! let mut api_security = ApiSecurityManager::new(security_config);
//! api_security.validate_request("build", "client_id", xml_input.len())?;
//! # Ok::<(), ddex_builder::BuildError>(())
//! ```
//!
//! ## Architecture Overview
//!
//! ```text
//! ┌─────────────────────────────────────────────────────────────────┐
//! │                        DDEX Builder                             │
//! ├─────────────────────────────────────────────────────────────────┤
//! │  Input Layer                                                    │
//! │  ┌─────────────┐ ┌──────────────┐ ┌─────────────┐              │
//! │  │ XML Parser  │ │ JSON Parser  │ │ Presets     │              │
//! │  │ (Security)  │ │ (Validation) │ │ (Partners)  │              │
//! │  └─────────────┘ └──────────────┘ └─────────────┘              │
//! ├─────────────────────────────────────────────────────────────────┤
//! │  Processing Layer                                               │
//! │  ┌─────────────┐ ┌──────────────┐ ┌─────────────┐              │
//! │  │ AST Builder │ │ Reference    │ │ Version     │              │
//! │  │ (Elements)  │ │ Linker       │ │ Converter   │              │
//! │  └─────────────┘ └──────────────┘ └─────────────┘              │
//! ├─────────────────────────────────────────────────────────────────┤
//! │  Output Layer                                                   │
//! │  ┌─────────────┐ ┌──────────────┐ ┌─────────────┐              │
//! │  │ XML         │ │ DB-C14N      │ │ Output      │              │
//! │  │ Generator   │ │ Canonicalize │ │ Sanitizer   │              │
//! │  └─────────────┘ └──────────────┘ └─────────────┘              │
//! └─────────────────────────────────────────────────────────────────┘
//! ```
//!
//! ## Performance Characteristics
//!
//! - **Parse 10KB**: <5ms
//! - **Parse 100KB**: <10ms  
//! - **Parse 1MB**: <50ms
//! - **Build typical release**: <15ms
//! - **Memory usage**: <50MB for large files with streaming
//! - **WASM bundle size**: <500KB
//!
//! ## Version Support
//!
//! | DDEX Version | Support Level | Notes |
//! |--------------|---------------|-------|
//! | ERN 3.8.2    | ✅ Full       | Legacy support |
//! | ERN 4.2      | ✅ Full       | Enhanced features |
//! | ERN 4.3      | ✅ Full       | Latest standard |
//!
//! ## Partner Presets
//!
//! Pre-configured settings for major platforms:
//!
//! - `spotify_audio_43` - Spotify audio releases (ERN 4.3)
//! - `youtube_video_43` - YouTube video content (ERN 4.3)
//! - `apple_music_43` - Apple Music releases (ERN 4.3)
//! - `universal_basic` - Universal Music basic preset
//! - `sony_enhanced` - Sony Music enhanced features
//!
//! See the [User Guide](https://docs.ddex-builder.io/user-guide) for detailed preset documentation.

#![forbid(unsafe_code)]
#![warn(missing_docs)]

pub mod api_security;
pub mod ast;
pub mod builder;
pub mod caching;
pub mod canonical;
pub mod determinism;
pub mod diff;
pub mod error;
pub mod fidelity;
pub mod generator;
pub mod guarantees;
pub mod id_generator;
pub mod linker;
pub mod memory_optimization;
pub mod messages;
pub mod namespace_minimizer;
pub mod optimized_strings;
pub mod parallel_processing;
pub mod preflight;
pub mod presets;
pub mod round_trip;
pub mod schema;
pub mod security;
pub mod streaming;
pub mod verification;
pub mod versions;

// Re-export main types
pub use builder::{BuildOptions, BuildRequest, BuildResult, DDEXBuilder};
pub use canonical::DB_C14N;
pub use determinism::DeterminismConfig;
pub use diff::formatter::DiffFormatter;
pub use diff::types::{ChangeSet, ChangeType, DiffPath, ImpactLevel, SemanticChange};
pub use diff::{DiffConfig, DiffEngine, VersionCompatibility};
pub use error::{BuildError, BuildWarning};
pub use guarantees::{DeterminismGuarantee, DeterminismGuaranteeValidator, GuaranteeReport};
pub use id_generator::{HashAlgorithm, StableHashConfig, StableHashGenerator};
pub use linker::{EntityType, LinkerConfig, LinkingError, ReferenceLinker};
pub use messages::{
    UpdateAction, UpdateConfig, UpdateGenerator, UpdateReleaseMessage, ValidationStatus,
};
pub use preflight::{PreflightLevel, PreflightValidator, ValidationConfig, ValidationResult};
pub use presets::DdexVersion;
pub use presets::PartnerPreset;
pub use schema::{JsonSchema, SchemaCommand, SchemaConfig, SchemaDraft, SchemaGenerator};
pub use versions::{
    ConversionOptions, ConverterResult as ConversionResult, VersionConverter, VersionManager,
};

// Security module exports
pub use api_security::{ApiSecurityConfig, ApiSecurityManager, BatchStats, FfiDataType};
pub use security::{InputValidator, OutputSanitizer, RateLimiter, SecureTempFile, SecurityConfig};

// Perfect Fidelity Engine exports
pub use fidelity::{FidelityConfig, FidelityStatistics, PreservationLevel};
pub use round_trip::{FidelityAnalysis, RoundTripTester};
pub use verification::{BuildVerifier, VerificationStatistics};

use indexmap::IndexMap;
use serde::{Deserialize, Serialize};
use std::collections::HashMap;
use std::time::Duration;

/// Version of the DB-C14N specification
pub const DB_C14N_VERSION: &str = "1.0";

/// Perfect Fidelity Engine configuration options
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct FidelityOptions {
    /// Enable perfect round-trip fidelity preservation
    pub enable_perfect_fidelity: bool,
    /// Preserve all XML comments in their original positions
    pub preserve_comments: bool,
    /// Preserve processing instructions
    pub preserve_processing_instructions: bool,
    /// Preserve unknown/extension elements and attributes
    pub preserve_extensions: bool,
    /// Preserve original attribute ordering when possible
    pub preserve_attribute_order: bool,
    /// Preserve namespace prefixes from input
    pub preserve_namespace_prefixes: bool,
    /// Canonicalization algorithm to use
    pub canonicalization: CanonicalizationAlgorithm,
    /// Custom canonicalization rules (used with Custom algorithm)
    pub custom_canonicalization_rules: Option<CustomCanonicalizationRules>,
    /// Enable deterministic element ordering
    pub enable_deterministic_ordering: bool,
    /// Collect detailed building statistics
    pub collect_statistics: bool,
    /// Enable build verification (double-check output)
    pub enable_verification: bool,
    /// Verification configuration
    pub verification_config: VerificationConfig,
}

impl Default for FidelityOptions {
    fn default() -> Self {
        Self {
            enable_perfect_fidelity: false,
            preserve_comments: false,
            preserve_processing_instructions: false,
            preserve_extensions: true,
            preserve_attribute_order: false,
            preserve_namespace_prefixes: false,
            canonicalization: CanonicalizationAlgorithm::DbC14N,
            custom_canonicalization_rules: None,
            enable_deterministic_ordering: true,
            collect_statistics: false,
            enable_verification: false,
            verification_config: VerificationConfig::default(),
        }
    }
}

/// Canonicalization algorithms supported by the Perfect Fidelity Engine
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub enum CanonicalizationAlgorithm {
    /// No canonicalization (preserves exact formatting)
    None,
    /// XML Canonicalization Version 1.0 (W3C C14N)
    C14N,
    /// XML Canonicalization Version 1.1 (W3C C14N11)
    C14N11,
    /// DDEX-specific DB-C14N/1.0 algorithm (default)
    DbC14N,
    /// Custom canonicalization with user-defined rules
    Custom(CustomCanonicalizationRules),
}

/// Custom canonicalization rules for specialized use cases
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CustomCanonicalizationRules {
    /// Preserve whitespace between elements
    pub preserve_whitespace: bool,
    /// Sort attributes alphabetically
    pub sort_attributes: bool,
    /// Normalize line endings to LF
    pub normalize_line_endings: bool,
    /// Remove redundant namespace declarations
    pub minimize_namespaces: bool,
    /// Custom attribute ordering rules
    pub attribute_ordering: Vec<String>,
    /// Custom element ordering rules
    pub element_ordering: HashMap<String, Vec<String>>,
}

impl Default for CustomCanonicalizationRules {
    fn default() -> Self {
        Self {
            preserve_whitespace: false,
            sort_attributes: true,
            normalize_line_endings: true,
            minimize_namespaces: true,
            attribute_ordering: Vec::new(),
            element_ordering: HashMap::new(),
        }
    }
}

/// Build verification configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct VerificationConfig {
    /// Enable round-trip verification (build -> parse -> build)
    pub enable_round_trip_verification: bool,
    /// Enable canonicalization verification
    pub enable_canonicalization_verification: bool,
    /// Enable schema validation after build
    pub enable_schema_validation: bool,
    /// Enable determinism verification (multiple builds identical)
    pub enable_determinism_verification: bool,
    /// Number of builds for determinism verification
    pub determinism_test_iterations: usize,
    /// Timeout for verification operations
    pub verification_timeout: Duration,
}

impl Default for VerificationConfig {
    fn default() -> Self {
        Self {
            enable_round_trip_verification: true,
            enable_canonicalization_verification: true,
            enable_schema_validation: false,
            enable_determinism_verification: true,
            determinism_test_iterations: 3,
            verification_timeout: Duration::from_secs(30),
        }
    }
}

/// Build verification result
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct VerificationResult {
    /// Overall verification success
    pub success: bool,
    /// Round-trip verification result
    pub round_trip_success: bool,
    /// Canonicalization verification result
    pub canonicalization_success: bool,
    /// Schema validation result
    pub schema_validation_success: bool,
    /// Determinism verification result
    pub determinism_success: bool,
    /// Verification errors and warnings
    pub issues: Vec<VerificationIssue>,
    /// Time taken for verification
    pub verification_time: Duration,
}

/// Verification issue (error or warning)
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct VerificationIssue {
    /// Issue severity
    pub severity: VerificationSeverity,
    /// Issue category
    pub category: String,
    /// Human-readable message
    pub message: String,
    /// Optional path to the problematic element
    pub path: Option<String>,
    /// Optional suggestion for fixing
    pub suggestion: Option<String>,
}

/// Verification issue severity
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub enum VerificationSeverity {
    /// Error that prevents successful verification
    Error,
    /// Warning that may indicate a problem
    Warning,
    /// Informational message
    Info,
}

/// Building statistics collected during the build process
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct BuildStatistics {
    /// Total build time
    pub build_time: Duration,
    /// Time spent on canonicalization
    pub canonicalization_time: Duration,
    /// Time spent on validation
    pub validation_time: Duration,
    /// Time spent on verification
    pub verification_time: Duration,
    /// Peak memory usage during build
    pub peak_memory_bytes: usize,
    /// Number of elements processed
    pub element_count: usize,
    /// Number of attributes processed
    pub attribute_count: usize,
    /// Size of input data
    pub input_size_bytes: usize,
    /// Size of output data
    pub output_size_bytes: usize,
    /// Number of namespaces processed
    pub namespace_count: usize,
    /// Number of comments preserved
    pub comment_count: usize,
    /// Number of processing instructions preserved
    pub processing_instruction_count: usize,
}

impl Default for BuildStatistics {
    fn default() -> Self {
        Self {
            build_time: Duration::ZERO,
            canonicalization_time: Duration::ZERO,
            validation_time: Duration::ZERO,
            verification_time: Duration::ZERO,
            peak_memory_bytes: 0,
            element_count: 0,
            attribute_count: 0,
            input_size_bytes: 0,
            output_size_bytes: 0,
            namespace_count: 0,
            comment_count: 0,
            processing_instruction_count: 0,
        }
    }
}

/// The main DDEX Builder for creating deterministic XML output.
///
/// `Builder` is the primary interface for generating DDEX-compliant XML with
/// guaranteed deterministic output. It supports partner presets, version conversion,
/// and comprehensive security features.
///
/// ## Features
///
/// - **Deterministic Output**: Uses DB-C14N/1.0 for reproducible output
/// - **Partner Presets**: Pre-configured settings for major music platforms
/// - **Version Management**: Support for ERN 3.8.2, 4.2, and 4.3 with conversion
/// - **Security**: Built-in validation, rate limiting, and XXE protection
/// - **Performance**: Memory-optimized with streaming support for large files
///
/// ## Usage Patterns
///
/// ### Basic Usage
///
/// ```rust
/// use ddex_builder::Builder;
///
/// let builder = Builder::new();
/// let available_presets = builder.available_presets();
/// println!("Available presets: {:?}", available_presets);
/// ```
///
/// ### With Partner Preset
///
/// ```rust
/// use ddex_builder::Builder;
///
/// let mut builder = Builder::new();
/// builder.preset("spotify_audio_43")?;
///
/// // Builder is now configured for Spotify Audio releases (ERN 4.3)
/// assert!(builder.is_preset_locked() == false); // Unlocked for further customization
/// # Ok::<(), ddex_builder::BuildError>(())
/// ```
///
/// ### Locked Preset Configuration
///
/// ```rust
/// use ddex_builder::Builder;
///
/// let mut builder = Builder::new();
/// builder.apply_preset("spotify_audio_43", true)?; // Lock the preset
///
/// assert!(builder.is_preset_locked());
/// # Ok::<(), ddex_builder::BuildError>(())
/// ```
///
/// ### Version Conversion
///
/// ```rust
/// use ddex_builder::{Builder, DdexVersion};
/// use ddex_builder::versions::ConversionOptions;
///
/// let builder = Builder::new();
///
/// // Check version compatibility
/// let compatible = builder.is_version_compatible(
///     DdexVersion::Ern382,
///     DdexVersion::Ern43
/// );
///
/// if compatible {
///     let options = Some(ConversionOptions::default());
///     let result = builder.convert_version(
///         &xml_content,
///         DdexVersion::Ern382,
///         DdexVersion::Ern43,
///         options
///     )?;
///     println!("Converted XML: {}", result.converted_xml);
/// }
/// # let xml_content = "<test></test>";
/// # Ok::<(), ddex_builder::BuildError>(())
/// ```
///
/// ## Thread Safety
///
/// `Builder` is `Send + Sync` and can be safely shared between threads.
/// Each thread should create its own instance for best performance.
///
/// ## Memory Usage
///
/// The builder uses memory-optimized data structures and streaming
/// where possible. Typical memory usage:
/// - Small releases (<100KB): ~5MB
/// - Large releases (>1MB): ~20-50MB with streaming
#[derive(Debug, Clone)]
pub struct Builder {
    config: DeterminismConfig,
    presets: IndexMap<String, PartnerPreset>,
    locked_preset: Option<String>,
    version_manager: versions::VersionManager,
    target_version: Option<DdexVersion>,
    fidelity_options: FidelityOptions,
    verification_config: VerificationConfig,
}

impl Default for Builder {
    fn default() -> Self {
        Self::new()
    }
}

impl Builder {
    /// Creates a new DDEX Builder with default configuration.
    ///
    /// The builder is initialized with:
    /// - Default determinism configuration for reproducible output
    /// - All available partner presets loaded
    /// - No preset locked (can be changed)
    /// - Latest supported DDEX version as target
    ///
    /// # Examples
    ///
    /// ```rust
    /// use ddex_builder::Builder;
    ///
    /// let builder = Builder::new();
    /// assert!(!builder.is_preset_locked());
    /// assert!(builder.available_presets().len() > 0);
    /// ```
    ///
    /// # Performance
    ///
    /// Creating a new builder is fast (~1μs) as presets are loaded from
    /// embedded configuration data.
    pub fn new() -> Self {
        Self {
            config: DeterminismConfig::default(),
            presets: Self::load_default_presets(),
            locked_preset: None,
            version_manager: versions::VersionManager::new(),
            target_version: None,
            fidelity_options: FidelityOptions::default(),
            verification_config: VerificationConfig::default(),
        }
    }

    /// Create builder with custom configuration
    pub fn with_config(config: DeterminismConfig) -> Self {
        Self {
            config,
            presets: Self::load_default_presets(),
            locked_preset: None,
            version_manager: versions::VersionManager::new(),
            target_version: None,
            fidelity_options: FidelityOptions::default(),
            verification_config: VerificationConfig::default(),
        }
    }

    /// Create builder with Perfect Fidelity Engine enabled
    pub fn with_perfect_fidelity() -> Self {
        let mut fidelity_options = FidelityOptions::default();
        fidelity_options.enable_perfect_fidelity = true;
        fidelity_options.preserve_comments = true;
        fidelity_options.preserve_processing_instructions = true;
        fidelity_options.preserve_extensions = true;
        fidelity_options.preserve_attribute_order = true;
        fidelity_options.preserve_namespace_prefixes = true;
        fidelity_options.enable_verification = true;

        Self {
            config: DeterminismConfig::default(),
            presets: Self::load_default_presets(),
            locked_preset: None,
            version_manager: versions::VersionManager::new(),
            target_version: None,
            fidelity_options,
            verification_config: VerificationConfig::default(),
        }
    }

    /// Create builder with custom fidelity options
    pub fn with_fidelity_options(fidelity_options: FidelityOptions) -> Self {
        Self {
            config: DeterminismConfig::default(),
            presets: Self::load_default_presets(),
            locked_preset: None,
            version_manager: versions::VersionManager::new(),
            target_version: None,
            fidelity_options,
            verification_config: VerificationConfig::default(),
        }
    }

    /// Create builder optimized for round-trip operations
    pub fn for_round_trip() -> Self {
        let mut fidelity_options = FidelityOptions::default();
        fidelity_options.enable_perfect_fidelity = true;
        fidelity_options.preserve_comments = true;
        fidelity_options.preserve_processing_instructions = true;
        fidelity_options.preserve_extensions = true;
        fidelity_options.preserve_attribute_order = true;
        fidelity_options.preserve_namespace_prefixes = true;
        fidelity_options.canonicalization = CanonicalizationAlgorithm::DbC14N;
        fidelity_options.enable_verification = true;
        fidelity_options.collect_statistics = true;

        let mut verification_config = VerificationConfig::default();
        verification_config.enable_round_trip_verification = true;
        verification_config.enable_canonicalization_verification = true;
        verification_config.enable_determinism_verification = true;

        Self {
            config: DeterminismConfig::default(),
            presets: Self::load_default_presets(),
            locked_preset: None,
            version_manager: versions::VersionManager::new(),
            target_version: None,
            fidelity_options,
            verification_config,
        }
    }

    /// Applies a partner preset configuration to the builder.
    ///
    /// Presets contain pre-configured settings optimized for specific music platforms
    /// and distribution partners. Each preset includes determinism settings, validation
    /// rules, and format preferences.
    ///
    /// # Arguments
    ///
    /// * `preset_name` - Name of the preset to apply (see [`available_presets`])
    /// * `lock` - Whether to lock the preset to prevent further modifications
    ///
    /// # Returns
    ///
    /// * `Ok(())` - Preset applied successfully
    /// * `Err(BuildError::InvalidFormat)` - Unknown preset name
    ///
    /// # Examples
    ///
    /// ```rust
    /// use ddex_builder::Builder;
    ///
    /// let mut builder = Builder::new();
    ///
    /// // Apply Spotify preset without locking
    /// builder.apply_preset("spotify_audio_43", false)?;
    /// assert!(!builder.is_preset_locked());
    ///
    /// // Apply and lock YouTube preset  
    /// builder.apply_preset("youtube_video_43", true)?;
    /// assert!(builder.is_preset_locked());
    /// # Ok::<(), ddex_builder::BuildError>(())
    /// ```
    ///
    /// # Available Presets
    ///
    /// Common presets include:
    /// - `spotify_audio_43` - Spotify audio releases (ERN 4.3)
    /// - `youtube_video_43` - YouTube video content (ERN 4.3)  
    /// - `apple_music_43` - Apple Music releases (ERN 4.3)
    /// - `universal_basic` - Universal Music basic preset
    /// - `sony_enhanced` - Sony Music enhanced features
    ///
    /// Use [`available_presets`] to get the complete list.
    ///
    /// [`available_presets`]: Self::available_presets
    pub fn apply_preset(&mut self, preset_name: &str, lock: bool) -> Result<(), error::BuildError> {
        let preset = self
            .presets
            .get(preset_name)
            .ok_or_else(|| error::BuildError::InvalidFormat {
                field: "preset".to_string(),
                message: format!("Unknown preset: {}", preset_name),
            })?
            .clone();

        // Apply the preset's determinism config
        self.config = preset.determinism;

        // Lock the preset if requested
        if lock {
            self.locked_preset = Some(preset_name.to_string());
        }

        Ok(())
    }

    /// Apply a preset configuration (alias for apply_preset for convenience)
    pub fn preset(&mut self, preset_name: &str) -> Result<&mut Self, error::BuildError> {
        self.apply_preset(preset_name, false)?;
        Ok(self)
    }

    /// Get available preset names
    pub fn available_presets(&self) -> Vec<String> {
        self.presets.keys().cloned().collect()
    }

    /// Get preset details
    pub fn get_preset(&self, preset_name: &str) -> Option<&PartnerPreset> {
        self.presets.get(preset_name)
    }

    /// Check if a preset is locked
    pub fn is_preset_locked(&self) -> bool {
        self.locked_preset.is_some()
    }

    /// Get the current configuration
    pub fn config(&self) -> &DeterminismConfig {
        &self.config
    }

    /// Get the current fidelity options
    pub fn fidelity_options(&self) -> &FidelityOptions {
        &self.fidelity_options
    }

    /// Set fidelity options
    pub fn set_fidelity_options(&mut self, options: FidelityOptions) -> &mut Self {
        self.fidelity_options = options;
        self
    }

    /// Enable Perfect Fidelity Engine with default settings
    pub fn enable_perfect_fidelity(&mut self) -> &mut Self {
        self.fidelity_options.enable_perfect_fidelity = true;
        self.fidelity_options.preserve_comments = true;
        self.fidelity_options.preserve_processing_instructions = true;
        self.fidelity_options.preserve_extensions = true;
        self.fidelity_options.preserve_attribute_order = true;
        self.fidelity_options.preserve_namespace_prefixes = true;
        self.fidelity_options.enable_verification = true;
        self
    }

    /// Disable Perfect Fidelity Engine
    pub fn disable_perfect_fidelity(&mut self) -> &mut Self {
        self.fidelity_options.enable_perfect_fidelity = false;
        self.fidelity_options.preserve_comments = false;
        self.fidelity_options.preserve_processing_instructions = false;
        self.fidelity_options.preserve_attribute_order = false;
        self.fidelity_options.preserve_namespace_prefixes = false;
        self.fidelity_options.enable_verification = false;
        self
    }

    /// Set canonicalization algorithm
    pub fn with_canonicalization(&mut self, algorithm: CanonicalizationAlgorithm) -> &mut Self {
        self.fidelity_options.canonicalization = algorithm;
        self
    }

    /// Enable DB-C14N/1.0 canonicalization (default for DDEX)
    pub fn with_db_c14n(&mut self) -> &mut Self {
        self.fidelity_options.canonicalization = CanonicalizationAlgorithm::DbC14N;
        self
    }

    /// Enable standard XML C14N canonicalization
    pub fn with_c14n(&mut self) -> &mut Self {
        self.fidelity_options.canonicalization = CanonicalizationAlgorithm::C14N;
        self
    }

    /// Enable XML C14N 1.1 canonicalization
    pub fn with_c14n11(&mut self) -> &mut Self {
        self.fidelity_options.canonicalization = CanonicalizationAlgorithm::C14N11;
        self
    }

    /// Set custom canonicalization rules
    pub fn with_custom_canonicalization(
        &mut self,
        rules: CustomCanonicalizationRules,
    ) -> &mut Self {
        self.fidelity_options.canonicalization = CanonicalizationAlgorithm::Custom(rules.clone());
        self.fidelity_options.custom_canonicalization_rules = Some(rules);
        self
    }

    /// Enable build verification
    pub fn with_verification(&mut self, config: VerificationConfig) -> &mut Self {
        self.fidelity_options.enable_verification = true;
        self.verification_config = config;
        self
    }

    /// Enable statistics collection
    pub fn with_statistics(&mut self) -> &mut Self {
        self.fidelity_options.collect_statistics = true;
        self
    }

    /// Check if Perfect Fidelity Engine is enabled
    pub fn is_perfect_fidelity_enabled(&self) -> bool {
        self.fidelity_options.enable_perfect_fidelity
    }

    /// Get current canonicalization algorithm
    pub fn canonicalization_algorithm(&self) -> &CanonicalizationAlgorithm {
        &self.fidelity_options.canonicalization
    }

    /// Set target DDEX version for building
    pub fn with_version(&mut self, version: DdexVersion) -> &mut Self {
        self.target_version = Some(version);
        self
    }

    /// Get the target DDEX version
    pub fn target_version(&self) -> Option<DdexVersion> {
        self.target_version
    }

    /// Detect version from XML content
    pub fn detect_version(&self, xml_content: &str) -> Result<DdexVersion, error::BuildError> {
        self.version_manager
            .detect_version(xml_content)
            .map(|detection| detection.detected_version)
            .map_err(|e| error::BuildError::InvalidFormat {
                field: "version".to_string(),
                message: format!("Version detection failed: {}", e),
            })
    }

    /// Convert XML between DDEX versions
    pub fn convert_version(
        &self,
        xml_content: &str,
        from_version: DdexVersion,
        to_version: DdexVersion,
        options: Option<ConversionOptions>,
    ) -> Result<versions::ConverterResult, error::BuildError> {
        let converter = versions::VersionConverter::new();
        Ok(converter.convert(xml_content, from_version, to_version, options))
    }

    /// Get version compatibility information
    pub fn is_version_compatible(&self, from: DdexVersion, to: DdexVersion) -> bool {
        self.version_manager.is_conversion_supported(from, to)
    }

    /// Get supported DDEX versions
    pub fn supported_versions(&self) -> Vec<DdexVersion> {
        versions::utils::supported_versions()
    }

    fn load_default_presets() -> IndexMap<String, PartnerPreset> {
        presets::all_presets()
    }

    /// Build DDEX XML with Perfect Fidelity Engine
    pub fn build_with_fidelity(
        &self,
        request: &builder::BuildRequest,
    ) -> Result<FidelityBuildResult, error::BuildError> {
        let start_time = std::time::Instant::now();
        let mut statistics = BuildStatistics::default();

        // Use the existing build options structure
        let build_options = builder::BuildOptions::default();

        // Build the XML using existing builder
        let ddex_builder = builder::DDEXBuilder::new();
        let build_result = ddex_builder.build(request.clone(), build_options)?;

        statistics.build_time = start_time.elapsed();
        statistics.output_size_bytes = build_result.xml.len();

        // Perform verification if enabled
        let verification_result = if self.fidelity_options.enable_verification {
            // Convert lib VerificationConfig to verification VerificationConfig
            let verification_config = verification::VerificationConfig {
                enable_round_trip_verification: self
                    .verification_config
                    .enable_round_trip_verification,
                enable_canonicalization_verification: self
                    .verification_config
                    .enable_canonicalization_verification,
                enable_schema_validation: self.verification_config.enable_schema_validation,
                enable_determinism_verification: self
                    .verification_config
                    .enable_determinism_verification,
                determinism_test_iterations: self.verification_config.determinism_test_iterations,
                verification_timeout: self.verification_config.verification_timeout,
            };
            let verifier = verification::BuildVerifier::new(verification_config);
            let result = verifier.verify(&build_result.xml, &self.fidelity_options)?;

            // Convert verification::VerificationResult to VerificationResult
            Some(VerificationResult {
                success: result.success,
                round_trip_success: result.round_trip_success,
                canonicalization_success: result.canonicalization_success,
                determinism_success: result.determinism_success,
                schema_validation_success: result.schema_validation_success,
                issues: result
                    .issues
                    .into_iter()
                    .map(|issue| VerificationIssue {
                        category: issue.category,
                        severity: match issue.severity {
                            verification::VerificationSeverity::Error => {
                                VerificationSeverity::Error
                            }
                            verification::VerificationSeverity::Warning => {
                                VerificationSeverity::Warning
                            }
                            verification::VerificationSeverity::Info => VerificationSeverity::Info,
                        },
                        message: issue.message,
                        path: issue.path,
                        suggestion: issue.suggestion,
                    })
                    .collect(),
                verification_time: result.verification_time,
            })
        } else {
            None
        };

        Ok(FidelityBuildResult {
            xml: build_result.xml,
            statistics: if self.fidelity_options.collect_statistics {
                Some(statistics)
            } else {
                None
            },
            verification_result,
            canonicalization_applied: self.fidelity_options.canonicalization
                != CanonicalizationAlgorithm::None,
            db_c14n_version: if self.fidelity_options.canonicalization
                == CanonicalizationAlgorithm::DbC14N
            {
                Some(DB_C14N_VERSION.to_string())
            } else {
                None
            },
        })
    }

    /// Verify build output meets fidelity requirements
    pub fn verify_build(&self, xml_output: &str) -> Result<VerificationResult, error::BuildError> {
        // Convert lib VerificationConfig to verification VerificationConfig
        let verification_config = verification::VerificationConfig {
            enable_round_trip_verification: self.verification_config.enable_round_trip_verification,
            enable_canonicalization_verification: self
                .verification_config
                .enable_canonicalization_verification,
            enable_schema_validation: self.verification_config.enable_schema_validation,
            enable_determinism_verification: self
                .verification_config
                .enable_determinism_verification,
            determinism_test_iterations: self.verification_config.determinism_test_iterations,
            verification_timeout: self.verification_config.verification_timeout,
        };

        let verifier = verification::BuildVerifier::new(verification_config);
        let result = verifier.verify(xml_output, &self.fidelity_options)?;

        // Convert verification::VerificationResult to VerificationResult
        Ok(VerificationResult {
            success: result.success,
            round_trip_success: result.round_trip_success,
            canonicalization_success: result.canonicalization_success,
            determinism_success: result.determinism_success,
            schema_validation_success: result.schema_validation_success,
            issues: result
                .issues
                .into_iter()
                .map(|issue| VerificationIssue {
                    category: issue.category,
                    severity: match issue.severity {
                        verification::VerificationSeverity::Error => VerificationSeverity::Error,
                        verification::VerificationSeverity::Warning => {
                            VerificationSeverity::Warning
                        }
                        verification::VerificationSeverity::Info => VerificationSeverity::Info,
                    },
                    message: issue.message,
                    path: issue.path,
                    suggestion: issue.suggestion,
                })
                .collect(),
            verification_time: result.verification_time,
        })
    }

    /// Test round-trip fidelity: XML → Parse → Build → Parse → Compare
    pub fn test_round_trip_fidelity(
        &self,
        original_xml: &str,
    ) -> Result<RoundTripResult, error::BuildError> {
        let round_trip = round_trip::RoundTripTester::new(self.fidelity_options.clone());
        let result = round_trip.test_round_trip(original_xml)?;

        // Convert round_trip::RoundTripResult to RoundTripResult
        Ok(RoundTripResult {
            success: result.success,
            original_xml: result.original_xml,
            rebuilt_xml: result.rebuilt_xml,
            byte_identical: result.byte_identical,
            differences: result.differences,
            test_time: result.test_time,
        })
    }

    /// Canonicalize XML using the configured algorithm
    pub fn canonicalize(&self, xml_content: &str) -> Result<String, error::BuildError> {
        match &self.fidelity_options.canonicalization {
            CanonicalizationAlgorithm::None => Ok(xml_content.to_string()),
            CanonicalizationAlgorithm::C14N => {
                // TODO: Implement C14N canonicalization
                Ok(xml_content.to_string())
            }
            CanonicalizationAlgorithm::C14N11 => {
                // TODO: Implement C14N11 canonicalization
                Ok(xml_content.to_string())
            }
            CanonicalizationAlgorithm::DbC14N => {
                // TODO: Implement DB-C14N canonicalization using canonical module
                Ok(xml_content.to_string())
            }
            CanonicalizationAlgorithm::Custom(rules) => {
                // TODO: Implement custom canonicalization
                let _ = rules; // Avoid unused parameter warning
                Ok(xml_content.to_string())
            }
        }
    }

    /// Get DB-C14N/1.0 configuration details
    pub fn db_c14n_config(&self) -> DbC14NConfig {
        DbC14NConfig {
            version: DB_C14N_VERSION.to_string(),
            algorithm: self.fidelity_options.canonicalization.clone(),
            deterministic_ordering: self.fidelity_options.enable_deterministic_ordering,
            preserve_comments: self.fidelity_options.preserve_comments,
            preserve_processing_instructions: self
                .fidelity_options
                .preserve_processing_instructions,
            namespace_handling: if self.fidelity_options.preserve_namespace_prefixes {
                NamespaceHandling::Preserve
            } else {
                NamespaceHandling::Minimize
            },
        }
    }

    /// Internal build method used by determinism verifier
    pub(crate) fn build_internal(
        &self,
        request: &builder::BuildRequest,
    ) -> Result<builder::BuildResult, error::BuildError> {
        let ddex_builder = builder::DDEXBuilder::new();
        let build_options = builder::BuildOptions::default();

        ddex_builder.build(request.clone(), build_options)
    }
}

/// Perfect Fidelity Engine build result
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct FidelityBuildResult {
    /// Generated XML output
    pub xml: String,
    /// Build statistics (if enabled)
    pub statistics: Option<BuildStatistics>,
    /// Verification result (if enabled)
    pub verification_result: Option<VerificationResult>,
    /// Whether canonicalization was applied
    pub canonicalization_applied: bool,
    /// DB-C14N version used (if applicable)
    pub db_c14n_version: Option<String>,
}

/// Round-trip test result
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct RoundTripResult {
    /// Whether round-trip was successful
    pub success: bool,
    /// Original XML input
    pub original_xml: String,
    /// XML after build process
    pub rebuilt_xml: String,
    /// Whether XMLs are byte-identical after canonicalization
    pub byte_identical: bool,
    /// Differences found (if any)
    pub differences: Vec<String>,
    /// Time taken for round-trip test
    pub test_time: Duration,
}

/// DB-C14N/1.0 configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct DbC14NConfig {
    /// DB-C14N specification version
    pub version: String,
    /// Canonicalization algorithm in use
    pub algorithm: CanonicalizationAlgorithm,
    /// Whether deterministic ordering is enabled
    pub deterministic_ordering: bool,
    /// Whether comments are preserved
    pub preserve_comments: bool,
    /// Whether processing instructions are preserved
    pub preserve_processing_instructions: bool,
    /// Namespace handling strategy
    pub namespace_handling: NamespaceHandling,
}

/// Namespace handling strategies
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub enum NamespaceHandling {
    /// Preserve original namespace prefixes
    Preserve,
    /// Minimize namespace declarations
    Minimize,
    /// Normalize namespace prefixes
    Normalize,
}

/// Version information for the builder
pub fn version_info() -> String {
    format!(
        "DDEX Builder v{} • DB-C14N/{} • Perfect Fidelity Engine • Rust {}",
        env!("CARGO_PKG_VERSION"),
        DB_C14N_VERSION,
        env!("CARGO_PKG_RUST_VERSION", "unknown")
    )
}

/// Get Perfect Fidelity Engine information
pub fn fidelity_engine_info() -> String {
    format!(
        "Perfect Fidelity Engine v{} • Round-trip: ✓ • DB-C14N/{} • Extensions: ✓",
        env!("CARGO_PKG_VERSION"),
        DB_C14N_VERSION
    )
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_builder_creation() {
        let builder = Builder::new();
        assert!(!builder.is_preset_locked());
    }

    #[test]
    fn test_preset_application() {
        let mut builder = Builder::new();
        assert!(builder.apply_preset("audio_album", false).is_ok());
        assert!(!builder.is_preset_locked());

        assert!(builder.apply_preset("audio_album", true).is_ok());
        assert!(builder.is_preset_locked());
    }

    #[test]
    fn test_unknown_preset() {
        let mut builder = Builder::new();
        assert!(builder.apply_preset("unknown_preset", false).is_err());
    }

    #[test]
    fn test_version_info() {
        let info = version_info();
        assert!(info.contains("DDEX Builder"));
        assert!(info.contains("DB-C14N/1.0"));
    }
}
