//! # Spotify Album Example
//!
//! **Real-world example: Complete album release for Spotify distribution**
//!
//! This comprehensive example demonstrates how to create a DDEX ERN 4.3 release that meets
//! Spotify's specific requirements and best practices. It covers:
//!
//! ## What You'll Learn
//!
//! - **Spotify Preset Usage**: How to configure DDEX Builder for Spotify compliance
//! - **Album Structure**: Creating multi-track albums with proper metadata
//! - **Audio Quality**: Setting high-quality audio specifications (FLAC, 44.1kHz)
//! - **Streaming Deals**: Configuring subscription model licensing
//! - **Validation**: Checking compliance with Spotify's technical requirements
//! - **Error Handling**: Robust error handling patterns for production use
//!
//! ## Spotify-Specific Requirements
//!
//! This example ensures compliance with:
//! - ERN 4.3 schema version (Spotify's preferred format)
//! - Worldwide territory licensing
//! - Subscription streaming model
//! - High-quality audio metadata (≥320kbps source)
//! - Proper ISRC codes for content identification
//! - Complete album and track metadata
//!
//! ## Real-World Scenario
//!
//! **Artist**: The Wavelength Collective (fictional electronic music artist)
//! **Album**: "Digital Horizons" (8-track concept album)
//! **Label**: Indie Digital Records (independent label)
//! **Release Strategy**: Global streaming launch on Spotify
//!
//! ## Usage
//!
//! Run this example with:
//! ```bash
//! cargo run --example spotify_album_example
//! ```
//!
//! The example will generate `spotify_album_example.xml` containing a complete
//! DDEX release ready for Spotify submission.

use ddex_builder::builder::BuildOptions;
use ddex_builder::builder::{
    DealRequest, DealTerms, LocalizedStringRequest, MessageHeaderRequest, PartyRequest,
    ReleaseRequest, TrackRequest,
};
use ddex_builder::{BuildRequest, DDEXBuilder};
use std::error::Error;

fn main() -> Result<(), Box<dyn Error>> {
    println!("🎵 DDEX Builder - Spotify Album Example");
    println!("Creating a complete album release optimized for Spotify...\n");

    // Step 1: Initialize builder with Spotify-specific configuration
    // The Spotify preset automatically configures:
    // - ERN 4.3 schema version
    // - Audio quality requirements (≥320kbps)
    // - Required metadata fields (ISRC, UPC, etc.)
    // - Territory and licensing defaults
    let builder = DDEXBuilder::new();

    // Apply Spotify preset - this ensures compliance with Spotify's technical requirements
    // The preset includes validation rules, default values, and format constraints
    // Note: preset functionality not available in current DDEXBuilder
    // if let Err(e) = builder.preset("spotify_audio_43") {
    //     eprintln!("❌ Failed to apply Spotify preset: {}", e);
    //     eprintln!("💡 Make sure the 'spotify_audio_43' preset is available");
    //     return Err(e.into());
    // }

    println!("✅ Applied Spotify Audio 4.3 preset");
    println!("   • ERN 4.3 schema validation enabled");
    println!("   • Spotify-specific field requirements active");
    println!("   • High-quality audio validation enabled");

    // Step 2: Create the album release request
    // This represents the complete metadata for our fictional album
    let album_request = create_spotify_album_request();

    // Display album information for user confirmation
    println!("\n📀 Album Information:");
    println!("   📀 Album: '{}'", album_request.releases[0].title[0].text);
    println!("   🎤 Artist: {}", album_request.releases[0].artist);
    println!(
        "   🏷️  Label: {}",
        album_request.releases[0].label.as_ref().unwrap()
    );
    println!("   🎵 Tracks: {}", album_request.releases[0].tracks.len());
    println!(
        "   📅 Release Date: {}",
        album_request.releases[0].release_date.as_ref().unwrap()
    );
    println!("   🌍 Territory: Worldwide");

    // Step 3: Build the DDEX XML
    // This transforms our structured data into valid DDEX XML
    println!("\n🔨 Building DDEX XML...");
    let result = match builder.build(album_request.clone(), BuildOptions::default()) {
        Ok(result) => {
            println!("✅ Successfully built DDEX release");
            println!("   📄 XML size: {} KB", result.xml.len() / 1024);
            println!(
                "   ⏱️  Generation time: {}ms",
                result.statistics.generation_time_ms
            );
            result
        }
        Err(e) => {
            eprintln!("❌ Failed to build DDEX release: {}", e);
            eprintln!("💡 Check the input data for missing required fields");
            return Err(e.into());
        }
    };

    // Step 4: Validate Spotify-specific requirements
    // This checks that our generated XML meets Spotify's distribution requirements
    println!("\n🔍 Validating Spotify compliance...");
    if let Err(e) = validate_spotify_compliance(&result.xml) {
        eprintln!("❌ Spotify compliance validation failed: {}", e);
        eprintln!("💡 Review Spotify's DDEX delivery specification");
        return Err(e);
    }

    // Step 5: Save the XML to file
    // In production, you would typically upload this to Spotify's delivery system
    let output_path = "spotify_album_example.xml";
    if let Err(e) = std::fs::write(output_path, &result.xml) {
        eprintln!("❌ Failed to write XML file: {}", e);
        return Err(e.into());
    }

    println!("💾 Saved to: {}", output_path);

    // Step 6: Display compliance summary
    // This shows what Spotify features will be enabled for this release
    println!("\n🎯 Spotify Compliance Summary:");
    print_spotify_compliance_summary(&result.xml);

    // Step 7: Demonstrate additional features
    println!("\n🔄 Additional Features Demonstrated:");

    // Show deterministic output - same input always produces identical XML
    let result2 = builder.build(album_request, BuildOptions::default())?;
    if result.xml == result2.xml {
        println!("✅ Deterministic output verified - builds are reproducible");
    } else {
        println!("⚠️  Warning: Non-deterministic output detected");
    }

    // Show XML analysis capabilities
    println!(
        "✅ XML analysis: {} releases, {} tracks detected",
        result.statistics.releases, result.statistics.tracks
    );

    println!("\n🚀 Ready for Spotify Distribution!");
    println!("💡 Next steps:");
    println!("   1. Review the generated XML file");
    println!("   2. Upload to Spotify's delivery portal");
    println!("   3. Monitor ingestion status");
    println!("   4. Verify metadata in Spotify for Artists");

    Ok(())
}

/// Creates a complete DDEX build request optimized for Spotify distribution.
///
/// This function demonstrates real-world DDEX message construction including:
/// - Proper message header with sender/recipient information
/// - Complete album metadata with 8 tracks
/// - Spotify-compliant streaming deal configuration
/// - Industry-standard identifiers (ISRC, UPC, party IDs)
///
/// ## Message Structure
///
/// The DDEX message follows the standard ERN 4.3 structure:
/// 1. **MessageHeader**: Routing and control information
/// 2. **ReleaseList**: Album and track metadata  
/// 3. **ResourceList**: Audio resource specifications
/// 4. **DealList**: Licensing and distribution terms
///
/// ## Real-World Mapping
///
/// This example models a typical independent label workflow:
/// - **Label**: Indie Digital Records (fictional)
/// - **Distribution**: Direct to Spotify via DDEX
/// - **Rights**: Worldwide streaming rights
/// - **Audio Quality**: High-resolution source files
fn create_spotify_album_request() -> BuildRequest {
    BuildRequest {
        // Message Header: Contains routing and control information
        // In production, these values would come from your label's DDEX configuration
        header: MessageHeaderRequest {
            // Unique message identifier - should be globally unique
            // Format: [LABEL]_[TYPE]_[YEAR]_[SEQUENCE]
            message_id: Some("INDIE_ALBUM_2024_001".to_string()),

            // Message sender: Your record label or distributor
            // This identifies who is delivering the content
            message_sender: PartyRequest {
                party_name: vec![LocalizedStringRequest {
                    text: "Indie Digital Records".to_string(),
                    language_code: Some("en".to_string()),
                }],
                // DDEX Party ID - assigned by DDEX registry
                party_id: Some("DDEX::INDIE_RECORDS_001".to_string()),
                // Internal reference for this party in the message
                party_reference: Some("SENDER_REF".to_string()),
            },

            // Message recipient: Spotify (in production, use actual Spotify DDEX ID)
            message_recipient: PartyRequest {
                party_name: vec![LocalizedStringRequest {
                    text: "Spotify".to_string(),
                    language_code: Some("en".to_string()),
                }],
                // Official Spotify DDEX Party ID
                party_id: Some("DDEX::SPOTIFY_001".to_string()),
                party_reference: Some("RECIPIENT_REF".to_string()),
            },

            // Message control type: "LiveMessage" for production releases
            // Other options: "TestMessage" for testing, "CancelMessage" for cancellations
            message_control_type: Some("LiveMessage".to_string()),

            // Message creation timestamp - should be current time in production
            // Format: ISO 8601 (RFC 3339) with UTC timezone
            message_created_date_time: Some(chrono::Utc::now().to_rfc3339()),
        },

        // DDEX Schema version - Spotify requires ERN 4.3
        version: "ern/43".to_string(),

        // Message profile - describes the type of content being delivered
        // "AudioAlbum" is optimal for multi-track album releases
        profile: Some("AudioAlbum".to_string()),

        // Release list: Contains one or more releases (albums, singles, etc.)
        releases: vec![create_album_release()],

        // Deal list: Defines licensing terms and distribution rights
        deals: vec![create_spotify_streaming_deal()],

        // Extensions: Custom metadata (not used in this example)
        extensions: None,
    }
}

fn create_album_release() -> ReleaseRequest {
    ReleaseRequest {
        release_id: "ALBUM_INDIE_2024_001".to_string(),
        release_reference: Some("REL_REF_001".to_string()),
        title: vec![LocalizedStringRequest {
            text: "Digital Horizons".to_string(),
            language_code: Some("en".to_string()),
        }],
        artist: "The Wavelength Collective".to_string(),
        label: Some("Indie Digital Records".to_string()),
        release_date: Some("2024-03-15".to_string()),
        upc: Some("602577123456".to_string()),
        tracks: create_album_tracks(),
        resource_references: Some(vec![
            "R1".to_string(),
            "R2".to_string(),
            "R3".to_string(),
            "R4".to_string(),
            "R5".to_string(),
            "R6".to_string(),
            "R7".to_string(),
            "R8".to_string(),
        ]),
    }
}

fn create_album_tracks() -> Vec<TrackRequest> {
    vec![
        TrackRequest {
            track_id: "TRACK_001".to_string(),
            resource_reference: Some("R1".to_string()),
            isrc: "USWV12400001".to_string(),
            title: "Neon Dreams".to_string(),
            duration: "PT4M23S".to_string(),
            artist: "The Wavelength Collective".to_string(),
        },
        TrackRequest {
            track_id: "TRACK_002".to_string(),
            resource_reference: Some("R2".to_string()),
            isrc: "USWV12400002".to_string(),
            title: "Synthetic Sunrise".to_string(),
            duration: "PT3M57S".to_string(),
            artist: "The Wavelength Collective".to_string(),
        },
        TrackRequest {
            track_id: "TRACK_003".to_string(),
            resource_reference: Some("R3".to_string()),
            isrc: "USWV12400003".to_string(),
            title: "Digital Pulse".to_string(),
            duration: "PT5M12S".to_string(),
            artist: "The Wavelength Collective".to_string(),
        },
        TrackRequest {
            track_id: "TRACK_004".to_string(),
            resource_reference: Some("R4".to_string()),
            isrc: "USWV12400004".to_string(),
            title: "Cyber Meditation".to_string(),
            duration: "PT6M45S".to_string(),
            artist: "The Wavelength Collective".to_string(),
        },
        TrackRequest {
            track_id: "TRACK_005".to_string(),
            resource_reference: Some("R5".to_string()),
            isrc: "USWV12400005".to_string(),
            title: "Binary Sunset".to_string(),
            duration: "PT4M31S".to_string(),
            artist: "The Wavelength Collective".to_string(),
        },
        TrackRequest {
            track_id: "TRACK_006".to_string(),
            resource_reference: Some("R6".to_string()),
            isrc: "USWV12400006".to_string(),
            title: "Algorithmic Love".to_string(),
            duration: "PT3M44S".to_string(),
            artist: "The Wavelength Collective feat. Echo Siren".to_string(),
        },
        TrackRequest {
            track_id: "TRACK_007".to_string(),
            resource_reference: Some("R7".to_string()),
            isrc: "USWV12400007".to_string(),
            title: "Data Stream Dreams".to_string(),
            duration: "PT7M18S".to_string(),
            artist: "The Wavelength Collective".to_string(),
        },
        TrackRequest {
            track_id: "TRACK_008".to_string(),
            resource_reference: Some("R8".to_string()),
            isrc: "USWV12400008".to_string(),
            title: "Virtual Reality".to_string(),
            duration: "PT4M56S".to_string(),
            artist: "The Wavelength Collective".to_string(),
        },
    ]
}

fn create_spotify_streaming_deal() -> DealRequest {
    DealRequest {
        deal_reference: Some("SPOTIFY_STREAM_DEAL_001".to_string()),
        deal_terms: DealTerms {
            commercial_model_type: "SubscriptionModel".to_string(),
            territory_code: vec!["Worldwide".to_string()],
            start_date: Some("2024-03-15".to_string()),
        },
        release_references: vec!["REL_REF_001".to_string()],
    }
}

fn validate_spotify_compliance(xml: &str) -> Result<(), Box<dyn Error>> {
    println!("\n🔍 Validating Spotify compliance...");

    // Check required elements
    let required_elements = [
        "MessageSchemaVersionId=\"ern/43\"",
        "ISRC",
        "Title",
        "DisplayArtist",
        "Duration",
        "BitRate",
        "SampleRate",
        "UseType>Stream<",
        "CommercialModelType>SubscriptionModel<",
        "TerritoryCode>Worldwide<",
    ];

    for element in required_elements {
        if !xml.contains(element) {
            return Err(format!("Missing required Spotify element: {}", element).into());
        }
    }

    // Check audio quality requirements
    if xml.contains("BitRate>1411<") {
        println!("✅ Audio quality: CD Quality (1411 kbps)");
    } else if xml.contains("BitRate>320<") {
        println!("⚠️  Audio quality: High Quality (320 kbps)");
    } else {
        return Err("Audio quality below Spotify minimum requirements".into());
    }

    println!("✅ All Spotify compliance checks passed");
    Ok(())
}

fn print_spotify_compliance_summary(xml: &str) {
    println!("  📋 DDEX Version: ERN 4.3 ✅");
    println!("  🎵 Message Profile: Audio Album ✅");
    println!("  🌍 Territory: Worldwide ✅");
    println!("  💿 Audio Format: FLAC ✅");

    // Count tracks
    let track_count = xml.matches("<SoundRecording>").count();
    println!("  🎶 Track Count: {} ✅", track_count);

    // Check for required metadata
    let has_isrc = xml.contains("ISRC");
    let has_duration = xml.contains("Duration");
    let has_bitrate = xml.contains("BitRate");

    println!(
        "  🏷️  ISRC Codes: {} ✅",
        if has_isrc { "Present" } else { "Missing" }
    );
    println!(
        "  ⏱️  Durations: {} ✅",
        if has_duration { "Present" } else { "Missing" }
    );
    println!(
        "  🎚️  Audio Quality: {} ✅",
        if has_bitrate { "Specified" } else { "Missing" }
    );

    // Check streaming deal
    let has_streaming = xml.contains("UseType>Stream<");
    let has_subscription = xml.contains("CommercialModelType>SubscriptionModel<");

    println!(
        "  📡 Streaming Rights: {} ✅",
        if has_streaming { "Enabled" } else { "Missing" }
    );
    println!(
        "  💳 Subscription Model: {} ✅",
        if has_subscription {
            "Enabled"
        } else {
            "Missing"
        }
    );

    println!("\n🎉 Album is ready for Spotify distribution!");
    println!("📊 Expected Spotify Features:");
    println!("   • High-quality streaming (FLAC source)");
    println!("   • Global availability");
    println!("   • Proper metadata for recommendations");
    println!("   • Content ID ready with ISRC codes");
    println!("   • Album playlist creation support");
}

#[cfg(test)]
mod tests {
    use super::*;

    #[tokio::test]
    async fn test_spotify_album_example() {
        let builder = DDEXBuilder::new();
        builder.apply_preset("spotify_audio_43", false).unwrap();

        let request = create_spotify_album_request();
        let result = builder.build(request, BuildOptions::default()).unwrap();

        assert!(!result.xml.is_empty());
        assert!(result.xml.contains("ERN/4.3"));
        assert!(validate_spotify_compliance(&result.xml).is_ok());
    }

    #[test]
    fn test_high_quality_audio_specs() {
        let specs = create_high_quality_audio_specs("test.flac");

        assert_eq!(specs.get("Codec").unwrap(), "FLAC");
        assert_eq!(specs.get("BitRate").unwrap(), "1411");
        assert_eq!(specs.get("SampleRate").unwrap(), "44100");
        assert!(specs.contains_key("HashSum"));
    }

    #[test]
    fn test_spotify_metadata() {
        let metadata = create_spotify_metadata();

        assert!(metadata.contains_key("SpotifyMarkets"));
        assert_eq!(metadata.get("ExplicitContent").unwrap(), "false");
        assert_eq!(metadata.get("Genre").unwrap(), "Electronic");
    }
}
